package com.polestar.sensors;

import android.annotation.TargetApi;
import android.app.PendingIntent;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothManager;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.os.SystemClock;

import com.polestar.helpers.ConnectivityHelper;
import com.polestar.helpers.Log;
import com.polestar.models.BleData;
import com.polestar.models.BleMeasurement;
import com.polestar.models.ISensorListener;
import com.polestar.models.SensorNao;
import com.polestar.naosdk.api.ISensorObserver;
import com.polestar.naosdk.api.TSENSORTYPE;
import com.polestar.naosdk.api.UuidMap;

import java.lang.ref.WeakReference;
import java.util.List;


public abstract class BleSensor extends SensorNao{

	private static final String BLE_SENSOR_THREAD_NAME = "BL_Thread"; // name of the thread BLE
	/** Default for the period of scan */
	public static final long BLE_SENSOR_DEFAULT_SLEEP_PERIOD_MSECONDS             = 10;
	public static final double BLE_SENSOR_DEFAULT_PERIOD_ACCUMULATOR_SECONDS  = 1100.0;
	public static final double BLE_SENSOR_DEFAULT_PERIOD_ACCUMULATOR_SECONDS_LOW_POWER  = 2200.0;
	public static final long DEFAULT_SCAN_PERIOD = 1100;
	public static final long DEFAULT_SCAN_PERIOD_LOW_POWER = 2200;
	/** The object to which this sensor sends its data when available */
	protected ISensorListener mObserver;
	
	/** The temp data to send to the receiver */
	//private BleData bleData;
	protected final BleMeasurement bleMeasurement; // accumulated bleData
	
    /** The timers to collect and push mems data to the receiver */
	protected long lastPushedDataTimestamp;

    /** The android sensors */
	protected BluetoothAdapter bluetooth;
	
    /** Tells whether Bluettoth was enabled when the user started the application */
	protected boolean mWasBlueToothEnabledAtAppStart = false;
	
	// messages to perform thread actions in the Handler
	protected static final int MSG_READY_TO_SEND = 1;
	protected static final int MSG_START = 2;
	protected static final int MSG_STOP = 3;
	protected static final int MSG_RESET = 4;
	protected static final int MSG_STOP_HARDWARE = 5;
	protected static final int MSG_START_HARDWARE = 6;
	protected static final int MSG_QUIT = 7;
	
	protected static final int BLE_SLEEP_PERIOD = 200 ;
	protected static final int BLE_TIMEOUT = 3000 ;

	protected long lastStartScanTimeStamp = 0L ;
	protected long ANDROID_N_MIN_SCAN_CYCLE = 6000L;



	protected long mScanPeriod = DEFAULT_SCAN_PERIOD;
	protected double mPushPeriodAccumulator = BLE_SENSOR_DEFAULT_PERIOD_ACCUMULATOR_SECONDS;


	// create handler to process incoming measurements
	protected Handler scanResultsHandler ;
	protected HandlerThread scanResultThread;


	/** nested static class to define the Handler and its actions */
	protected static BleCollectorHandler mBleCollectorHandler = null;

	protected UuidMap mUuidMap;
	protected PendingIntent mWakeUpIntent;
	protected BLEPOWERMODE mPowerMode = BLEPOWERMODE.HIGH;

	protected boolean resetEnabled = true;

	public void setResetEnabled(boolean resetEnabled) {
		this.resetEnabled = resetEnabled;
	}

	public enum BLEPOWERMODE {
		VERY_LOW,
		LOW,
		HIGH,
		;
	}

	/**
	 * Constructor
	 * @param caller, the context
	 */
	public BleSensor(ContextWrapper caller, ISensorObserver observer) {
		mSensorObserver = observer;
		mThread = new Thread(this);
		mThread.setName(BLE_SENSOR_THREAD_NAME);
		mThread.start();

		scanResultThread = new HandlerThread("Scan result thread");
		scanResultThread.start();
		scanResultsHandler = new Handler(scanResultThread.getLooper());

		//init status
		mStatus = STATUS_OFF;
		if (caller == null) {
			mStatus = STATUS_ERROR;
		}else{
			this.mCtxtWrap = caller;
			this.mStatus = STATUS_READY;
		}

		initBluetoothAdapter();
		//init bleMeasurement only in constructor
		bleMeasurement = new BleMeasurement();
	}

	private void initBluetoothAdapter(){
		// the sensors
		try {
			final BluetoothManager bluetoothManager = (BluetoothManager) this.mCtxtWrap.getApplicationContext().getSystemService(Context.BLUETOOTH_SERVICE);
			if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN_MR2) {
				bluetooth = bluetoothManager.getAdapter();
			} else {
				bluetooth = BluetoothAdapter.getDefaultAdapter();
			}

			if (bluetooth != null) {
				// Detect when bluetooth is enabled or not
				mWasBlueToothEnabledAtAppStart = bluetooth.isEnabled();
			}
		} catch (Exception e) {
			Log.alwaysWarn(this.getClass().getName(), "Cannot get bluetoothManager !");
		}
	}


	/**
	 *	thread run, BleSensor is a thread
	 */
	@Override
	public void run() {
		Looper.prepare();
		myLooper = Looper.myLooper();

		Log.restricted(this.getClass().getName(), "BLE Looper.prepare() ");
		// Initialize the handler to perform actions in this thread
		mBleCollectorHandler = new BleCollectorHandler(this);
		this.mOrder = ORDER_UNDEFINED;
		Looper.loop();
	}

	public void setUuidMap(UuidMap uuidMap) {
		this.mUuidMap = uuidMap;
	}


	protected static class BleCollectorHandler extends Handler {
		protected WeakReference<BleSensor> bleSensorRef;
		
		public BleCollectorHandler(BleSensor bs) {
			bleSensorRef = new WeakReference<BleSensor>(bs);
		}

		public void handleMessage(Message msg) {

			BleSensor bs = bleSensorRef.get();
			if (msg == null || bs == null)
				return;
			switch (msg.what) {
				case MSG_READY_TO_SEND:
					// if ble data has been collected during the current scan, send it immediately.
					synchronized (bs.bleMeasurement){
						if (bs.bleMeasurement.getNbMeas()>0) {
							bs.pushBleData();
						}
					}
					break;
				case MSG_START_HARDWARE:
					bs.startHardware() ;
					break ;
				case MSG_START:
					bs.startSensorImpl();
					break;
				case MSG_STOP:
					bs.stopSensorImpl();
					break;
				case MSG_RESET: // not implemented
					bs.resetImpl();
					break;
				case MSG_STOP_HARDWARE: // not implemented
					bs.stopHardware();
					break;
				case MSG_QUIT:
					bs.quitImpl();
					break;
				default:
					Log.alwaysError(this.getClass().getName(), "Unexpected message!");
					break;
			}
		}
	} // end of WifiCollectorHandler definition



	/**
	 * initBleSensors
	 * register to listen to BLE Android sensors and start sensing
	 * @return
	 */
	abstract protected boolean initBleSensors();
		
	/**
	 * isBleDataReadyToSend
	 * @param ts
	 * @return
	 */
	protected boolean isBleDataReadyToSend(long ts) {
		return (ts - lastPushedDataTimestamp > this.mPushPeriodAccumulator);
	}

	
	/**
	 *  clean-up BLE sensors and stop sensing
	 */
	abstract protected void finalizeBleSensors();

	abstract public void setDevicesFilter(List<String> devicesAdrr);
	
	/**
	 *  collect BLE measurement from sensors into the accumulator
	 */
	protected void collectBleData(String adress, int rssi, long ts, byte [] scanRecord){
		synchronized (bleMeasurement) {
			if (this.mStatus == STATUS_RUNNING) {
				BleData bleData = new BleData(ts, adress, scanRecord, rssi);
				bleMeasurement.addBleData(bleData);
			}
		}
	}

	/**
	 * push the accumulated BLE data to the recipient
	 */
	protected void pushBleData(){
		synchronized (bleMeasurement) {
			if (this.mStatus == STATUS_RUNNING){
				if (mObserver != null) {
					mObserver.onNewMeasurement(bleMeasurement);
				}

				if (mSensorObserver != null) {
					Log.restricted(this.getClass().getName(), "*************** bleMeas : " + bleMeasurement.getNbMeas());
					mSensorObserver.notifyOfNewData(TSENSORTYPE.BLE, bleMeasurement.toByteArray());
				}
				lastPushedDataTimestamp = bleMeasurement.getmTimeStamp();
				//reset values in bleMeasurement to prepare for next measurements
				bleMeasurement.reset();
			}
		}
	}

	@Override
	public void quit() {
		if (mBleCollectorHandler != null) {
			if (!mBleCollectorHandler.hasMessages(MSG_QUIT)){
				mBleCollectorHandler.sendEmptyMessage(MSG_QUIT);
			}
		}

	}

	public void quitImpl() {
		if (myLooper!=null) {
			Log.alwaysWarn(this.getClass().getName(), "BLE Sensor Looper quit");
			myLooper.quit();
		}
	}


	
	/**
	 * Initializes the sensor, starts the scanning ble. Can be called from any thread.<br>
	 * Does <i>not</i> start the thread 
	 * @return true if the method can be called in the right context
	 */
	@Override
	public synchronized boolean startSensor() {
		boolean returnValue;

		// If called from another thread: post execution to this thread
		if (mBleCollectorHandler != null) {
			if (!mBleCollectorHandler.hasMessages(MSG_START)){
				mBleCollectorHandler.sendEmptyMessage(MSG_START);
			}
			returnValue = true;
		} else {
			// Error: Handler not initialized
			returnValue = false;
		}
		return returnValue;
	}

	
	/**startSensorImpl
	 * start  Ble Sensors
	 * initTimers
	 * state READY -> RUNNING or ERROR
	 */
	public boolean startSensorImpl() {

		if(this.mIsLogger) {
			tryToStartHardware();
		}

		if(mStatus == STATUS_RUNNING) {
			return true;
		}

		this.mStatus = STATUS_READY;


		if(bluetooth!=null && bluetooth.getState()!=android.bluetooth.BluetoothAdapter.STATE_ON) {
	        Log.alwaysWarn(this.getClass().getName(), "BLE hardware: " + bluetooth.getState() );
		}


	    if (this.mStatus != STATUS_READY) {
			Log.alwaysWarn(this.getClass().getName(), "Cannot start BLE sensor: previous state = " + this.mStatus);
	        return false;
	    }
	    if (!initBleSensors()) {
	        this.mStatus = STATUS_ERROR;
			Log.alwaysWarn(this.getClass().getName(), "Cannot start BLE sensor: not available = " + this.mStatus);
	        return false;
	    }
	    this.mStatus = STATUS_RUNNING;
	    this.mOrder = ORDER_TURN_ON;

		return true;
	}
	
	private void tryToStartHardware(){
		boolean succeedInStartHardware = false ;
		int nbMaxTries = 3 ;
		int nbTries = 0 ;
		while (succeedInStartHardware==false && nbTries<nbMaxTries)
		{
			succeedInStartHardware = startHardware() ;
			if(!succeedInStartHardware) {
				Log.alwaysWarn(this.getClass().getName(),"startSensor(): didnt suceed in start hardware, new try...");
				try {
					Thread.sleep(100) ;
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			nbTries++ ;
		}
	}
	
	
	/**
	 * Stops the Wifi sensor. Can be called from any thread.<br>
	 * <li>Stop the scanning loop
	 * <li>Unregisters all asynchronous receivers
	 * <li>Releases the Wi-Fi lock
	 */
	@Override
	public synchronized void stopSensor() {
		// If called from another thread: post execution to this thread
		if (mBleCollectorHandler != null) {
			if (!mBleCollectorHandler.hasMessages(MSG_STOP)){
				mBleCollectorHandler.sendEmptyMessage(MSG_STOP);
			}
		}

		// wait a bit until the deed is done
		try {
			for (int k=0;k<10 && mStatus==STATUS_RUNNING;k++) {
					Thread.sleep(20);
			}
		} catch (InterruptedException e) {
			// void
		}
	}



	/**
	 * Function called by sensor owner for the sensor to properly stop before it completely ceases functionning.
	 * The sensor cannot be used any more after this call. This effectively exits its thread main loop.
	 * Has no effect if status is not RUNNING
	 * cancelTimers
	 * finalizeMemsSensors
	 * state RUNNING -> READY
	 */
	public void stopSensorImpl() {
		
		if (mBleCollectorHandler != null) {
	    	// remove remaining messages from the queue if any
			mBleCollectorHandler.removeMessages(MSG_READY_TO_SEND);
			mBleCollectorHandler.removeMessages(MSG_RESET);
			mBleCollectorHandler.removeMessages(MSG_STOP);
		}

	    if (this.mStatus != STATUS_RUNNING) {
	    	Log.alwaysWarn(this.getClass().getName(), "Cannot stop BLE sensor: previous state = " + this.mStatus);
	    	if(this.mOrderProvenance == ORDER_PROVENANCE_LOGGER) {
		    	stopHardware() ;
		    }
	        return;
	    }

		finalizeBleSensors();
	    
	    if(this.mOrderProvenance == ORDER_PROVENANCE_LOGGER) {
			stopHardware() ;
	    }
	    this.mOrder = ORDER_TURN_OFF;
	    this.mStatus = STATUS_READY;
	}



	/**
	 * disable the BLE hardware
	 * 
	 **/
	private boolean stopHardware() {
		return ConnectivityHelper.turnBleOFF(mCtxtWrap);
	}
	
	/**
	 * enable the BLE hardware
	 * 
	 **/
	private boolean startHardware() {
		return ConnectivityHelper.turnBleON(mCtxtWrap);
	}

	/**
	 * stops and restarts the sensor thread
	 * has no effect if state is not RUNNING.
	 *
	 * WARNING : this method may take a few hundred milliseconds and even a few seconds, since it blocks until the
	 * sensor thread ('main' method) ends.
	 **/
	@Override
	public void reset() {
		Log.alwaysWarn(this.getClass().getName(),"Reset Ble sensors from native .....");
		if (mBleCollectorHandler != null) {
			mBleCollectorHandler.removeMessages(MSG_RESET);
			mBleCollectorHandler.sendEmptyMessage(MSG_RESET);
		}
	}

	public void resetImpl() {
		if (this.mStatus == STATUS_RUNNING
				&& this.mOrder==ORDER_TURN_ON) {
				if(android.os.Build.VERSION.SDK_INT >= 24 && this.mPowerMode != BLEPOWERMODE.VERY_LOW) {
					long elapsedTimeSinceLastStart = SystemClock.elapsedRealtime() - lastStartScanTimeStamp ;
					if (elapsedTimeSinceLastStart >= ANDROID_N_MIN_SCAN_CYCLE) {
						stopSensorImpl();
						startSensorImpl();
						//save start scan timestamp : used to prevent too much start/stop on Android N (limited to 5 in 30seconds)
						lastStartScanTimeStamp = SystemClock.elapsedRealtime();
					} else {
						Log.alwaysWarn(getClass().getSimpleName(), "Reset BLE ignored to prevent reset scan too frequently on Android N ");
					}

				} else if(this.mIsLogger && bluetooth!=null && !bluetooth.isEnabled()) {
					tryToStartHardware();
				}
		}
	}


	@Override
	public void registerOutputInterface(ISensorListener targetInterface) {
		this.mObserver = targetInterface;
	}

	@Override
	public boolean isOn() {
		return (this.mStatus == STATUS_RUNNING);   
	}

	@Override
	public boolean hasFix() {
		return false;
	}

	@Override
	public boolean isActive() {

		if(mCtxtWrap!= null && Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
			// Version check: Android M requires location sensor to be switched on to use BLE
			if(!OSLocSensor.checkLocationProviders(mCtxtWrap))
				return false;
		}

		return isHardwareCompatible() && bluetooth != null && bluetooth.isEnabled();
	}
	
	/**
     * get if sensor is HW compatible
     */
	@TargetApi(18)
	public boolean isHardwareCompatible() {
		return (mCtxtWrap!= null && mCtxtWrap.getSystemService(Context.BLUETOOTH_SERVICE) != null
				&& mCtxtWrap.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE) ); 
	}
	
	/**
	 * destructor : cleanly stops the service.
	 */
	protected void finalize() {
		// set status

		mStatus = STATUS_OFF;
		mCtxtWrap = null;
	}

	/**
	 * if the ble was on when the app started, restart the ble if it is off
	 */
	public boolean setStateAsBefore() {
		if(mWasBlueToothEnabledAtAppStart){
	    	if(bluetooth != null && bluetooth.getState() != android.bluetooth.BluetoothAdapter.STATE_ON){
	    		Log.alwaysWarn(this.getClass().getName(), "Start hardware");
	    		mBleCollectorHandler.sendEmptyMessage(MSG_START_HARDWARE);
	    	} else {
	    		Log.alwaysWarn(this.getClass().getName(), "bluetooth is null");
	    	}
		} else {
			if(bluetooth != null && bluetooth.getState() != android.bluetooth.BluetoothAdapter.STATE_OFF){
	    		Log.alwaysWarn(this.getClass().getName(), "Stop hardware");
	    		mBleCollectorHandler.sendEmptyMessage(MSG_STOP_HARDWARE);
	    	} else {
	    		Log.alwaysWarn(this.getClass().getName(), "bluetooth is null");
	    	}
		}
		
		return true;
	}


	/**
	 * static method to get when ble is enabled
	 * @param ctx
	 * @return true if ble is enabled
	 */

	public static boolean isBleAvailable(ContextWrapper ctx) {
		boolean isHardwareCompatible = (ctx.getSystemService(Context.BLUETOOTH_SERVICE) != null
				&& ctx.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE) );
		boolean isBleEnabled = BluetoothAdapter.getDefaultAdapter().isEnabled()==true;

		return isHardwareCompatible && isBleEnabled;
	}
}

