package com.polestar.naosdk.managers;

import android.content.ContextWrapper;
import android.content.pm.PackageManager;

import com.polestar.models.BleMeasurement;
import com.polestar.models.Measurement;
import com.polestar.models.WifiMeasurement;
import com.polestar.naosdk.api.ISensorObserver;
import com.polestar.naosdk.api.ISensorProxy;
import com.polestar.naosdk.api.ISensorProxyFactory;
import com.polestar.naosdk.api.LoggerNaoLocationListener;
import com.polestar.naosdk.api.TSENSORTYPE;
import com.polestar.naosdk.api.UuidMap;
import com.polestar.naosdk.controllers.AndroidGeofencingService;
import com.polestar.sensors.BleSensor;
import com.polestar.sensors.BleSensor43;
import com.polestar.models.ISensorListener;
import com.polestar.helpers.Log;
import com.polestar.sensors.MemsMagnetoSensor;
import com.polestar.sensors.MemsSensor;
import com.polestar.sensors.OSLocSensor;
import com.polestar.models.SensorNao;
import com.polestar.sensors.WifiSensor;

import java.util.ArrayList;

/**
 * Created by jchouki on 08/12/2015.
 */
public class NaoSensorsManager extends ISensorProxyFactory implements ISensorListener {

    protected WifiSensor myWifiSensor;
    protected OSLocSensor myOSLocSensor;
    protected MemsSensor myMemsSensor;
    protected BleSensor myBleSensor;
    protected MemsMagnetoSensor myMemsMagnetoSensor;

    private ContextWrapper myContext;

    private LoggerNaoLocationListener myLoggerListener;

    //using synchronized methods / using the lock
    private Object lockMeasurement = new Object();

    public void regsiterLoggerListener(LoggerNaoLocationListener listener) {
        myLoggerListener = listener;
    }

    public NaoSensorsManager( ContextWrapper ctx) {
        this.myContext = ctx;
    }

     /**
     * checkBluetoothPermission
     * Bluetooth Permission check result
     * @return true if the permission has been granted to the given package.
     */
    private boolean checkBluetoothPermission(ContextWrapper contextWrapper){
        String permissionBLUETOOTH = "android.permission.BLUETOOTH";
        int resBLUETOOTH = contextWrapper.getApplicationContext().checkCallingOrSelfPermission(permissionBLUETOOTH);

        String permissionBLUETOOTH_ADMIN = "android.permission.BLUETOOTH_ADMIN";
        int resBLUETOOTH_ADMIN = contextWrapper.getApplicationContext().checkCallingOrSelfPermission(permissionBLUETOOTH_ADMIN);

        return (resBLUETOOTH == PackageManager.PERMISSION_GRANTED && resBLUETOOTH_ADMIN == PackageManager.PERMISSION_GRANTED);
    }

    /**
     * Start sensor ble only for samsung 4.2 and android API 4.3
     * startBleSensor
     * @return
     */
    private boolean startBleSensor(ContextWrapper contextWrapper, ISensorObserver observer) {

        //check permission Bluetooth first, if not , do nothing
        if(checkBluetoothPermission(contextWrapper) == false) {
            Log.alwaysWarn(this.getClass().getName(), "BLUETOOTH permission not granted to the given package ...");
            return false;
        }

        if(android.os.Build.VERSION.SDK_INT >=  android.os.Build.VERSION_CODES.JELLY_BEAN_MR2){
            Log.alwaysWarn(this.getClass().getName(), "Device on version >= 4.3  -> Create BleSensor43");
            myBleSensor = new BleSensor43(contextWrapper, observer);

            if(myBleSensor.isHardwareCompatible()) {
                return true;
            } else {
                Log.alwaysWarn(this.getClass().getName(), "BlueTooth LE not suported for this device: " + android.os.Build.MODEL);
                return false;
            }
        }

        //BlueTooth not started for version < android.os.Build.VERSION_CODES.JELLY_BEAN_MR2
        Log.alwaysWarn(this.getClass().getName(), "BlueTooth not started for this android version : " + android.os.Build.VERSION.SDK_INT);

        return false;
    }

    public void switchSensors(boolean useWifi, boolean useBle, boolean useMems, boolean useLocOS) {

        //init sensor if not yet created by CPP naoContext
        initAllSensors();

        //switch sensors when need
        if(this.myWifiSensor!= null && useWifi) {
            this.myWifiSensor.switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Starting wifi sensors from logger .....");
        } else if(this.myWifiSensor!= null && !useWifi) {
            this.myWifiSensor.switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Stoping wifi sensors from logger .....");
        }

        if(this.myBleSensor!= null && useBle) {
            this.myBleSensor.switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Starting ble sensors from logger .....");
        } else if(this.myBleSensor!= null && !useBle) {
            this.myBleSensor.switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Stoping ble sensors from logger .....");
        }

        if(this.myOSLocSensor!= null && useLocOS) {
            this.myOSLocSensor.switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Starting locos sensors from logger .....");
        } else if(this.myOSLocSensor!= null && !useLocOS) {
            this.myOSLocSensor.switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Stoping locos sensors from logger .....");
        }

        if(this.myMemsSensor!= null && useMems) {
            this.myMemsSensor.switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Starting mems sensors from logger .....");
        } else if(this.myMemsSensor!= null && !useBle) {
            this.myMemsSensor.switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Stoping mems sensors from logger .....");
        }

        if(this.myMemsMagnetoSensor!= null && useMems) {
            this.myMemsMagnetoSensor.switchSensor(SensorNao.ORDER_TURN_ON,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Starting mems magneto sensors from logger .....");
        } else if(this.myMemsMagnetoSensor!= null && !useBle) {
            this.myMemsMagnetoSensor.switchSensor(SensorNao.ORDER_TURN_OFF,SensorNao.ORDER_PROVENANCE_LOGGER) ;
            Log.alwaysWarn(this.getClass().getName(),"Stoping mems magneto sensors from logger .....");
        }
    }

    /**
     * create all sensors
     */
    private void initAllSensors() {

        if ((myWifiSensor == null))  {
            myWifiSensor = new WifiSensor( this.myContext,null,false);
            myWifiSensor.registerOutputInterface(this);
        }

        if ((myBleSensor == null))  {
            if(this.startBleSensor( this.myContext,null)) {
                myBleSensor.registerOutputInterface(this);
            }
        }

        if ((myMemsSensor == null))  {
            myMemsSensor = new MemsSensor( this.myContext,null);
        }

        if ((myOSLocSensor == null))  {
            myOSLocSensor = new OSLocSensor( this.myContext,null);
        }

        if ((myMemsMagnetoSensor == null))  {
            myMemsMagnetoSensor = new MemsMagnetoSensor( this.myContext,myMemsSensor);
        }

    }

    @Override
    public void setUuidMap(UuidMap uuidMap){
        if(this.myBleSensor!=null) {
            this.myBleSensor.setUuidMap(uuidMap);
        }
    }

    @Override
    public void bleResetEnabled(boolean enabled) {
        if(myBleSensor != null){
            myBleSensor.setResetEnabled(enabled);
        }
    }

    @Override
    public void requestWakelock(boolean checkOSLoc){
        if(!checkOSLoc || AndroidGeofencingService.isInsideOSGeofence()) {
            NaoServiceManager.acquireWakeLock(myContext.getApplicationContext());
        }
    }

    @Override
    public void releaseWakelock(boolean checkOSLoc) {
        if(!checkOSLoc || !AndroidGeofencingService.isInsideOSGeofence()) {
            NaoServiceManager.releaseWakeLock();
        }
    }

    @Override
    public ISensorProxy create(TSENSORTYPE sensorType, ISensorObserver observer) {
        Log.restricted(this.getClass().getName(), "Create ISensorProxy from native : " + sensorType);
        switch (sensorType) {
            case WIFI:
                if ((myWifiSensor == null))  {
                    myWifiSensor = new WifiSensor(myContext,observer,false);
                 } else {
                    myWifiSensor.setSensorObserver(observer);
                }
                myWifiSensor.registerOutputInterface(this);
                return myWifiSensor;

            case BLE:
                if ((myBleSensor == null))  {
                    if(this.startBleSensor(myContext,observer)) {
                        myBleSensor.registerOutputInterface(this);
                        return myBleSensor;
                    }
                } else {
                    myBleSensor.setSensorObserver(observer);
                    myBleSensor.registerOutputInterface(this);
                    return myBleSensor;
                }
                break;
            case MEMS:
                if ((myMemsSensor == null))  {
                    myMemsSensor = new MemsSensor(myContext,observer);
                } else {
                    myMemsSensor.setSensorObserver(observer);
                }
                return myMemsSensor;
            case LOCOS:
                if ((myOSLocSensor == null))  {
                    myOSLocSensor = new OSLocSensor(myContext,observer);
                } else {
                    myOSLocSensor.setSensorObserver(observer);
                }
                return myOSLocSensor;
            case MEMS_MAGNETO:
                if ((myMemsMagnetoSensor == null))  {
                    myMemsMagnetoSensor = new MemsMagnetoSensor(myContext,myMemsSensor);
                } else {
                    myMemsMagnetoSensor.setSensorObserver(observer);
                }
                return myMemsMagnetoSensor;
            default:
                Log.alwaysWarn(this.getClass().getName(),"sensorType problem ...");
                break;
        }
        return null;
    }

    @Override
    public void setBLEDevicesScanFilter(ArrayList<String> devicesList) {
        if(myBleSensor != null){
            Log.alwaysWarn("SCANFILTER", "size: " +devicesList.size());
            myBleSensor.setDevicesFilter(devicesList);
        }
    }

    @Override
    public void onNewMeasurement(Measurement meas) {

        synchronized(lockMeasurement) {
            if (meas.getClass() == WifiMeasurement.class) {
                //Notify Number of APs
                int numberOfAPs = ((WifiMeasurement) meas).getNumberOfAccessPoints() ;
                if(this.myLoggerListener != null) {
                  this.myLoggerListener.NotifyWifiAPsNumberUpdate(numberOfAPs) ;
                }
            } else if (meas.getClass() == BleMeasurement.class) {
                //Notify Number of APs
                int numberOfAPs = ((BleMeasurement) meas).getNbMeas() ;
                if(this.myLoggerListener != null) {
                    this.myLoggerListener.NotifyBleBeaconsNumberUpdate(numberOfAPs) ;
                }
            }
        }
    }

    @Override
    public void onNewLocOsStatus(boolean isOk) {

    }

    public void quit() {
        if(this.myWifiSensor!= null) {
            Log.alwaysWarn(this.getClass().getName(), "quit wifi sensor  .....");
            this.myWifiSensor.quit();
        }

        if(this.myBleSensor!= null) {
            Log.alwaysWarn(this.getClass().getName(), "quit ble sensor  .....");
            this.myBleSensor.quit();
        }

        if(this.myOSLocSensor!= null) {
            Log.alwaysWarn(this.getClass().getName(), "quit osloc sensor  .....");
            this.myOSLocSensor.quit();
        }

        if(this.myMemsSensor!= null) {
            Log.alwaysWarn(this.getClass().getName(), "quit mems sensor  .....");
            this.myMemsSensor.quit();
        }

        if(this.myMemsMagnetoSensor!= null) {
            Log.alwaysWarn(this.getClass().getName(), "quit mems magneto sensor  .....");
            this.myMemsMagnetoSensor.quit();
        }
    }

    public BleSensor getBleSensor() {
        return myBleSensor;
    }
}
