package com.polestar.naosdk.managers;

import android.content.Context;
import android.content.ContextWrapper;
import android.os.Handler;
import android.support.annotation.NonNull;

import com.polestar.helpers.NaoUtils;
import com.polestar.naosdk.api.IMeasureLogger;
import com.polestar.naosdk.api.INAOInternalClient;
import com.polestar.naosdk.api.LocationFix;
import com.polestar.naosdk.api.LoggerBeaconData;
import com.polestar.naosdk.api.LoggerNaoLocationListener;
import com.polestar.naosdk.api.TSENSORTYPE;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.NAOSensorsListener;

import java.util.ArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Created by jchouki on 19/01/2016.
 */
public class NaoLoggerProvider extends INAOInternalClient{

    private ContextWrapper myContext;
    private LoggerNaoLocationListener myLoggerListener;
    private Handler mainHandler;
    private ExecutorService serialExecutor = Executors.newSingleThreadExecutor();
    private ISensorRequestListenerImpl myISensorRequestListener;
    private boolean isBleEnabled = true;
    private boolean isWifiEnabled = false;

    public NaoLoggerProvider(Context ctx,Class<?> cls, LoggerNaoLocationListener listener, Handler handler, @NonNull NAOSensorsListener sensorsListener) {
        mainHandler = handler;

        myLoggerListener = listener;
        myContext = new ContextWrapper(ctx);
        NaoServiceManager.startService(ctx, cls);
        myISensorRequestListener = new ISensorRequestListenerImpl(mainHandler, sensorsListener, myContext);
    }

    public void startMeasurementLoop() {
        serialExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (NaoUtils.waitForNaoService()) {
                    if( !NaoServiceManager.getService().getNaoContext().naoServiceManager.hasServiceClients()) {
                        NaoServiceManager.getService().getNaoContext().naoServiceManager.start();
                    }
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });
    }

    public void switchSensors(final boolean wifi, final boolean ble, final boolean mems, final boolean osLoc) {
        serialExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (NaoUtils.waitForNaoService()) {
                    NaoServiceManager.getService().getNaoContext().getSensorsManager().switchSensors(wifi, ble, mems, osLoc);
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });
    }

    public void recordMarker(final long timestamp, final double lat, final double lon, final double alt, final int type) {
        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    NaoServiceManager.getService().getNaoContext().naoServiceManager.getMeasureLogger().onNewUserLoc(new LocationFix(timestamp, lon, lat, alt, 0, 0, true), type);
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });

    }


    public void startLogging(final String filename, final boolean isMems) {
        serialExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (NaoUtils.waitForNaoService()) {
                    IMeasureLogger measureLogger = NaoServiceManager.getService().getNaoContext().naoServiceManager.getMeasureLogger();
                    measureLogger.setMemsRecording(isMems);
                    measureLogger.startLoggingMeasurements(filename, false);
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });

    }

    public void stopLogging() {
        serialExecutor.execute(new Runnable() {
            @Override
            public void run() {
                if (NaoUtils.waitForNaoService()) {
                    NaoServiceManager.getService().getNaoContext().naoServiceManager.getMeasureLogger().stopLoggingMeasurements();
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });

    }

    @Override
    public ArrayList<TSENSORTYPE> getMonitoredSensors() {
        ArrayList<TSENSORTYPE> sensors = new ArrayList<>();

        if(isBleEnabled) {
            sensors.add(TSENSORTYPE.BLE);
        }
        if(isWifiEnabled) {
            sensors.add(TSENSORTYPE.WIFI);
        }
        return sensors;
    }

    @Override
    public void onBleListTranslated(ArrayList<LoggerBeaconData> data) {
        myLoggerListener.onNewNativeBleListArray(data);//TODO need to use Runnable?
    }

    @Override
    public void onError(NAOERRORCODE code, String msg) {
        myLoggerListener.onError(code, msg);
    }

    public void registerSensorsListener(final LoggerNaoLocationListener listener, final String key,final boolean isWifiEnabled, final boolean isBleEnabled) {
        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    if(isBleEnabled) {
                        registerInternalClient(key);
                    }
                    if(isWifiEnabled) {
                        NaoServiceManager.getService().getNaoContext().registerLoggerListener(listener);
                    }

                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }
        });

    }

    public void unregisterLoggerListener() {
        serialExecutor.execute(new Runnable()
        {
            @Override
            public void run()
            {
                if(NaoUtils.waitForNaoService()) {
                    unregisterInternalClient();
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }

        });

    }

    private void registerInternalClient(String key) {
        NaoServiceManager.getService().getNaoContext().naoServiceManager.registerInternalClient(this,key,myISensorRequestListener);
    }
    private void unregisterInternalClient() {
        NaoServiceManager.getService().getNaoContext().naoServiceManager.unregisterInternalClient(this);
    }
}
