package com.polestar.naosdk.emulators;

import android.content.ContextWrapper;
import android.os.Handler;
import android.util.Log;

import com.polestar.helpers.LocalPathHelper;

import java.io.File;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Created by lnguyen on 22/01/2016.
 */
public abstract class Emulator<ExternalClient> {

    public static final String EMULATOR_KEY = "emulator";

    public abstract int getOutputInitialDelayMillis();
    public abstract int getOutputPeriodMillis();
    public abstract String getEmulatorFileName();

    public abstract boolean start();
    public abstract void stop();

    protected abstract TimerTask createPeriodicFileReader();

    protected Timer mMeasurementReaderScheduler;

    protected boolean bIsRunning;

    protected int mReplayIndex;

    protected Handler mainHandler;

    protected String mFileName;

    ExternalClient mListener;

    public Emulator(ExternalClient listener, ContextWrapper context) {
        File emulatorDirectory = context.getExternalFilesDir(LocalPathHelper.NAO_ROOT_DIR);
        mainHandler = new Handler(context.getApplicationContext().getMainLooper());
        bIsRunning = false;
        mFileName = new File(emulatorDirectory, getEmulatorFileName()).getAbsolutePath();
        Log.d(this.getClass().getName(), "emulation file path " + mFileName);

        this.mListener = listener;
    }


    /**
     * @return target file name
     */
    public String getFileName()
    {
        return mFileName;
    }


    /**
     * stop periodically reading measurements from the binary log file. Does not close the file.
     */
    public void pause()
    {
        if (bIsRunning) {
            stopPeriodicReading();
        }
    }

    /**
     * resume periodically reading measurements from the binary log file. Does not restart from the beginning of the file.
     */
    public void resume()
    {
        if (bIsRunning) {
            startPeriodicReading();
        }
    }

    /**
     * Start the perioding reading (create the timer and scheduler the reading task)
     * @return true if successful, false else
     */

    protected boolean startPeriodicReading() {
        mMeasurementReaderScheduler = new Timer();
        mMeasurementReaderScheduler.scheduleAtFixedRate(createPeriodicFileReader(), getOutputInitialDelayMillis(), getOutputPeriodMillis());
        return true;
    }

    /**
     * pause the periodic reading of measurements
     */

    protected void stopPeriodicReading() {
        if (mMeasurementReaderScheduler != null) {
            mMeasurementReaderScheduler.cancel();
        }
    }

}
