package com.polestar.naosdk.controllers;


import android.app.Notification;
import android.app.Service;
import android.content.Intent;
import android.location.Location;
import android.os.Build;
import android.os.IBinder;
import android.support.annotation.Nullable;

import com.polestar.helpers.Log;
import com.polestar.helpers.NaoUtils;
import com.polestar.helpers.PrefHelper;
import com.polestar.naosdk.api.external.NAOERRORCODE;
import com.polestar.naosdk.api.external.NAOLocationHandle;
import com.polestar.naosdk.api.external.NAOLocationListener;
import com.polestar.naosdk.api.external.NAOSensorsListener;
import com.polestar.naosdk.api.external.NAOWakeUpNotifier;
import com.polestar.naosdk.api.external.TNAOFIXSTATUS;
import com.polestar.naosdk.api.external.TPOWERMODE;
import com.polestar.naosdk.managers.NaoServiceManager;

/**
 * Created by jchouki on 08/12/2015.
 */

public class AndroidGeofencingService extends Service implements NAOLocationListener, NAOSensorsListener {

    private static final int NOTIF_ID_1000 = 1000;
    private NAOLocationHandle recoveryLocation;
    private NAOWakeUpNotifier notifier; // this can live beyond the life of the service itself
    public static final String EVENT = "event";
    public static final String EVENT_ENTER = "enter";
    public static final String EVENT_EXIT = "exit";
    private static boolean isInsideOSGeofence = false;
    private boolean isInsideBeaconArea = false;
    private Notification serviceNotification;
    private boolean recoveryLocationRunning = false;

    public static boolean isInsideOSGeofence() {
        return isInsideOSGeofence;
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        //get notifier
        NAOWakeUpNotifier notifier = getWakeUpNotifier();
        if (notifier != null) {
            //if exist get notification from notifier
            serviceNotification = notifier.wakeUpNotification();
        }

        //if notification == null, it means that no wakeup notifier was set and geofencing
        // service has been started by sdk to acquire wakelock
        if (Build.VERSION.SDK_INT >= 26) {
            // start the service in foreground needed for Android > 8
            if(serviceNotification != null){
                 startForeground(NOTIF_ID_1000, serviceNotification);
             } else {
                 startForeground(NOTIF_ID_1000, new Notification());
             }

        }
        if(intent.hasExtra(EVENT)){
            String event = intent.getStringExtra(EVENT);
            if (event != null) {
                if (event.equalsIgnoreCase(EVENT_ENTER)) {
                    handleEnterGeofences();
                } else if (event.equalsIgnoreCase(EVENT_EXIT)) {
                    handleExitGeofences();
                }
            }
        }
        return START_NOT_STICKY;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        Log.writeToLog(this.getClass().getName(),"AndroidGeofencingService >> onDestroy");
        if(recoveryLocation != null){
            recoveryLocation.stop();
        }
    }

    //GeofenceTransition
    protected void handleEnterGeofences() {
        Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> onEnter");
        if(!isInsideOSGeofence){
            isInsideOSGeofence = true;
            if(hasServiceClients()) {
                NaoServiceManager.acquireWakeLock(this.getApplicationContext());
            }
            NAOWakeUpNotifier notifier = getWakeUpNotifier();
            if (notifier != null) {
                notifier.onEnterGPSArea();
                // enter site alert
                if (!isInsideBeaconArea) {
                    //start location client/handle dedicated to recovery
                    startRecoveryLocation(this,this);
                }
            } else stopCurrentService();

        } else if(!recoveryLocationRunning){
            //if we receive an other enter event do not stop service
            //because we had not yet trigger enter beacon areo
            stopCurrentService();
        }
    }

    //GeofenceTransition
    protected void handleExitGeofences() {
        Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> onExit");
        isInsideOSGeofence = false;

        NAOWakeUpNotifier notifier = getWakeUpNotifier();
        if (notifier != null) {
            notifier.onExitGPSArea();
        } else stopCurrentService();

        if(!hasTriggeredEnterSite()) {
            onExitCoverage();
        } else {
            //check exit coverage (both beacons and os loc geofence)
            if (recoveryLocation == null) {
                Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> checkExitCoverage");
                startRecoveryLocation(null, null);
                checkExitCoverage();
            }
        }
    }

    private boolean hasTriggeredEnterSite() {

        if(NaoServiceManager.getService()!= null && NaoServiceManager.getService().getNaoContext()!=null) {
            return NaoServiceManager.getService().getNaoContext().naoServiceManager.hasTriggeredEnterSite();
        }

        return false;
    }

    private boolean hasServiceClients() {

        if(NaoServiceManager.getService()!= null && NaoServiceManager.getService().getNaoContext()!=null) {
            return NaoServiceManager.getService().getNaoContext().naoServiceManager.hasServiceClients();
        }

        return false;
    }


    private void checkExitCoverage() {
        new Thread(new Runnable()
        {
            @Override
            public void run()
            {
                Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> waitForNaoServiceToBeStarted");
                if(NaoUtils.waitForNaoService()) {
                    Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> wait for exit site");

                   // Loop when user is out of the GPS area .And stop if we go again in the area.
                   // The false detections of exit gps zone can cause an infinite loop until an exit zone or beacons
                   while(!isInsideOSGeofence) {
                       if (!hasTriggeredEnterSite()) {
                           Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> onExitCoverage");
                           isInsideBeaconArea = false;
                           onExitCoverage();
                           break;
                       }
                       try {
                           Thread.sleep(10000);
                       } catch (InterruptedException e) {
                           e.printStackTrace();
                       }
                   }
                   if(isInsideOSGeofence) {
                       Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> isInsideOSGeofence now");
                       //stop the location client/handle dedicated to recovery
                       stopRecoveryLocation();
                   }
                } else {
                    onError(NAOERRORCODE.GENERIC_ERROR, "Can't instantiate NaoContext");
                }
            }
        }).start();
    }

    private void onExitCoverage() {

        // notify registered wakeup listeners of exit
        NAOWakeUpNotifier notifier = getWakeUpNotifier();
        if (notifier != null) {
            notifier.onExitCoverage();
        }

        // Release WakeLock to reduce battery drain outside of GPS geofence
        NaoServiceManager.releaseWakeLock();

        //stop the location client/handle dedicated to recovery if present
        stopRecoveryLocation();
        Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> stopSelf");
        stopCurrentService();

    }

    private void stopCurrentService() {
        stopSelf();

    }

    @Override //NAOLocationListener
    public void onEnterSite(String name) {
        isInsideBeaconArea = true;
        Log.restricted(getClass().getName(), "onEnterSite: Stop location and wake the app up");

        NAOWakeUpNotifier notifier = getWakeUpNotifier();
        if (notifier != null) {
            notifier.onEnterBeaconArea();
            //stop location client/handle dedicated to recovery
            stopRecoveryLocation();
            stopCurrentService();
        }
    }

    @Override //NAOLocationListener
    public void onExitSite(String name) {
        isInsideBeaconArea = false;
        NAOWakeUpNotifier notifier = getWakeUpNotifier();
        if (notifier != null) {
            notifier.onExitBeaconArea();
        }
        if (!isInsideOSGeofence){
            onExitCoverage();
        }
    }

    private NAOWakeUpNotifier getWakeUpNotifier() {
        if (notifier == null) {
            try {
                String className = PrefHelper.get(this.getApplicationContext(), PrefHelper.PREF_WAKE_UP_NOTIFIER_NAME,null);
                if (className != null) {
                    Class<NAOWakeUpNotifier> notifierClazz = (Class<NAOWakeUpNotifier>) Class.forName(className);
                    notifier = notifierClazz.newInstance();
                    notifier.setContext(this);
                }
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
        return notifier;
    }

    private void startRecoveryLocation(NAOLocationListener client,NAOSensorsListener sensorsListener) {
        String serviceName = PrefHelper.get(getApplicationContext(), PrefHelper.PREF_SERVICE_NAME, null);
        Log.writeToLog(this.getClass().getName(), "AndroidGeofencingService >> startRecoveryLocation : " + serviceName + " (client -> " + (client!=null) + ")");

        if (serviceName != null) {
            try {
                Class<NaoServiceManager> serviceRecovery = (Class<NaoServiceManager>) Class.forName(serviceName);
                recoveryLocation = new NAOLocationHandle(this, serviceRecovery, PrefHelper.get(getApplicationContext(), PrefHelper.PREF_APP_KEY, null), client, sensorsListener);
                recoveryLocation.setPowerMode(TPOWERMODE.LOW);
                recoveryLocationRunning = true;
                recoveryLocation.start();

            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            }
        }
    }

    private void stopRecoveryLocation() {
        if (recoveryLocation != null) {
            recoveryLocation.stop();
            recoveryLocationRunning = false;
            recoveryLocation = null;
            Log.writeToLog(getClass().getName(), "Recovery location stopped ...");
        } else {
            Log.alwaysWarn(getClass().getName(), "Cannot stop recovery location because not instantiated");
        }
    }

    @Override //NAOLocationListener
    public void onError(NAOERRORCODE errCode, String message) {
        Log.alwaysError(AndroidGeofencingService.class.getName(), "Error: " + errCode + " " + message);
    }

    @Override //NAOLocationListener
    public void onLocationChanged(Location loc) {
        Log.alwaysWarn(getClass().getName(), "Recovery Service Loc " + loc.getLongitude() + ", " + loc.getLatitude());
    }

    @Override //NAOLocationListener
    public void onLocationStatusChanged(TNAOFIXSTATUS status) {
        Log.restricted(getClass().getName(), "Recovery Service Loc Status " + status.toString());
    }

    @Override
    public void requiresCompassCalibration() {
        Log.restricted(getClass().getSimpleName(), "Require Compass");
    }

    @Override
    public void requiresWifiOn() {
        Log.restricted(getClass().getSimpleName(), "Require Wifi");
    }

    @Override
    public void requiresBLEOn() {
        Log.restricted(getClass().getSimpleName(), "Require BLE");
    }

    @Override
    public void requiresLocationOn() {}

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

}
