package com.polestar.naosdk.api.external;

import android.Manifest;
import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;

import com.polestar.helpers.PrefHelper;
import com.polestar.naosdk.api.TSENSORTYPE;
import com.polestar.naosdk.controllers.AndroidGeofencingService;
import com.polestar.naosdk.controllers.GeofencingOSController;
import com.polestar.naosdk.managers.NaoContext;
import com.polestar.naosdk.managers.NaoServiceManager;

/**
 * Entry point for NAO Services
 */
public final class NAOServicesConfig {

    /**
     * @exclude
     */
    public static final String[] BASIC_PERMISSIONS = new String[]{
            Manifest.permission.ACCESS_FINE_LOCATION
    };


    /**
     * Sets the root URL
     *
     * @param rootURL url of the server hosting NAO services input data
     *                Default points to Pole Star NAO Cloud server.
     */
    public static void setRootURL(String rootURL) {
        NaoContext.rootURL = rootURL;

        if (NaoServiceManager.getService() != null && NaoServiceManager.getService().getNaoContext() != null)
            NaoServiceManager.getService().getNaoContext().setRootURL(rootURL);
    }

    public static void setProxy(String host, int port, String user, String password) {
        NaoContext.proxy = new NaoContext.NAOProxy(host, port, user, password);
        if (NaoServiceManager.getService() != null && NaoServiceManager.getService().getNaoContext() != null)
            NaoServiceManager.getService().getNaoContext().setProxy(host, port, user, password);
    }

    /**
     * Gets the root URL
     *
     * @return URL of the server hosting NAO services input data
     */
    public static String getRootURL() {
        return NaoContext.getRootURL();
    }

    /**
     * Gets the global power mode of NAO SDK, resulting from all the various running Services.
     * If at least one service handle uses HIGH power mode, the global power mode will be HIGH.
     *
     * @return power mode currently used in NAO SDK
     */
    public static TPOWERMODE getPowerMode() {
        if (NaoServiceManager.getService() == null || NaoServiceManager.getService().getNaoContext() == null)
            return TPOWERMODE.LOW;
        return NaoServiceManager.getService().getNaoContext().getPowerMode();
    }

    /**
     * Gets SDK software version
     *
     * @return NAO SDK version
     */
    public static String getSoftwareVersion() {
        return NaoContext.getSoftwareVersion();
    }

    /**
     * Enable the Wake Up Notifier to receive notifications when the device arrives on site.
     * It is based on Android GPS localisation.
     * It requires the AndroidGeofencingService to be declared in AndroidManifest.xml.
     *
     * @param context             The context of your application
     * @param apiKey              API key for your site. Key is available on your NAO Cloud account.
     * @param wakeUpNotifierClass a class that enables waking up the application after it is killed.
     * @param listener            Listener for registration status
     */
    public static void enableOnSiteWakeUp(@NonNull Context context, @NonNull String apiKey, @NonNull Class<? extends NAOWakeUpNotifier> wakeUpNotifierClass, NAOWakeUpRegistrationListener listener) {

        // Register the wake up notifier class that need to be called in SharedPref
        PrefHelper.put(context, PrefHelper.PREF_WAKE_UP_NOTIFIER_NAME, wakeUpNotifierClass.getName());

        GeofencingOSController geofencingOSController = new GeofencingOSController(context);
        geofencingOSController.setListener(listener);
        geofencingOSController.registerGeofencesWithKey(apiKey);

    }

    /**
     * Disable the Wake Up Notifier when the device arrives on site.
     * All Geofences previously registered for this Api key will be unregistered
     *
     * @param context  The context of your application
     * @param apiKey   API key for your site. Key is available on your NAO Cloud account.
     * @param listener Listener for registration status
     */
    public static void disableOnSiteWakeUp(@NonNull Context context, @NonNull String apiKey, NAOWakeUpRegistrationListener listener) {

        //kill recovery service on disable on site wake
        context.stopService(new Intent(context, AndroidGeofencingService.class));
        PrefHelper.put(context, PrefHelper.PREF_WAKE_UP_NOTIFIER_NAME, null);

        GeofencingOSController geofencingOSController = new GeofencingOSController(context);
        geofencingOSController.setListener(listener);
        geofencingOSController.unregisterGeofencesWithKey(apiKey);
    }

    /**
     * @exclude
     */
    public static boolean isWakeUpNotifierRegistered(Context context) {
        return null != PrefHelper.get(context, PrefHelper.PREF_WAKE_UP_NOTIFIER_NAME, null);
    }

    /**
     * @exclude Debug utility - Pole Star use only
     * uploads NAO Services logs to server for analysis
     * @param message : message to accompany the log bundle
     */
    public static void uploadNAOLogInfo(String message) throws NAOException {
        if(message == null) {
            message = new String("");
        }
        getNaoContextOrThrow().naoServiceManager.uploadNAOLogInfo(message);
    }

    /**
     * @exclude Debug utility - Pole Star use only
     * starts logging sensor measurements
     */
    public static void startLoggingMeasurements() throws NAOException {
        getNaoContextOrThrow().naoServiceManager.getMeasureLogger().startLoggingMeasurements("", true);
    }

    /**
     * @exclude Debug utility - Pole Star use only
     * stops logging sensor measurements
     */
    public static void stopLoggingMeasurements() throws NAOException {
        getNaoContextOrThrow().naoServiceManager.getMeasureLogger().stopLoggingMeasurements();
    }


    /**
     * @exclude Debug utility - Pole Star use only
     * ignore / stop ignoring sensor measurements
     */
    public static void setSensorBypass(String sensorType, boolean flag) throws NAOException {
        getNaoContextOrThrow().naoServiceManager.setSensorBypass(TSENSORTYPE.valueOf(sensorType), flag);
    }

    /**
     * @exclude Debug utility - Pole Star use only
     */
    public static NaoContext getNaoContextOrThrow() throws NAOException {
        if (NaoServiceManager.getService() == null || NaoServiceManager.getService().getNaoContext() == null)
            throw new NAOException("No NaoContext instance: you must first instantiate a service handle.");

        return NaoServiceManager.getService().getNaoContext();
    }
}