//______________________________________________________________________________________
//
//  WifiMeasurement.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________
package com.polestar.models;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;

import android.net.wifi.ScanResult;
import android.os.Parcel;
import android.os.Parcelable;

/**
 * Defines a simple class for Wifi measurement data<br/>
 * Contains data for multiple access points.
 *    
 * <br/><br/><b>Revision history:</b><br/>
 * <table border>
 * <tr>
 *   <td>Author</td>
 *   <td>Modification date</td>
 *   <td>Tracking Number</td>
 *   <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>4 mars 2010</td>
 *   <td>trunk</td>
 *   <td>Creation of this class</td>
 * </tr>
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>15 mars 2010</td>
 *   <td>trunk</td>
 *   <td>moved from WifiSensor to NaoNativeBridge</td>
 * </tr>
 * </table>    
 * @author sterrenoir
 */
public class WifiMeasurement extends Measurement implements Parcelable {
	static final int PARCELINDENTIFIER = WifiMeasurement.class.hashCode();
	
    public static final Parcelable.Creator<WifiMeasurement> CREATOR = new Parcelable.Creator<WifiMeasurement>() {
        public WifiMeasurement createFromParcel(Parcel in) {
            return new WifiMeasurement(in);
        }

        public WifiMeasurement[] newArray(int size) {
            return new WifiMeasurement[size];
        }
    };
        
	// members : see below NaoWifiAccessPoint class definition
	/**
	 * 
	 * A custom class to hold AP data<br>
	 * Comparison is done by comparing BSSID strings<br>
	 * <br>
	 * Note: despite overloading the 'equals' and 'hashcode' methods, using this class in a HashSet
	 * allows duplicates, so using a TreeSet -taking advantage of the compareTo method- instead.
	 *        
	 * @author sterrenoir
	 */
	public class NaoWifiAccessPoint implements Comparable<NaoWifiAccessPoint>
	{
		/** AP 802.11 channel */
		public int channel;
		/** AP frequency in MHz */
		public int frequency;
		/** SSID (network name) */
		public String SSID;
		/** BSSID (most often the MAC address of the hardware interface) - formatted as '0123456789AB' */
		public String BSSID;
		/** RSSI level */
		public int RSSI;
		/**  
		 *  WEP encryption (Privacy) 
		 * 	0:free
		 * 	1:Privacy
		*/
		public int WEP;
		
		/**
		 * default constructor (all members to 0)
		 */
		public NaoWifiAccessPoint() 
		{
			reset();
		}
		
		/**
		 * Constructor from a ScanResult object
		 */
		public NaoWifiAccessPoint(ScanResult sr) {
			reset();
			set(sr);
		}

		/**
		 * Constructor with all members
		 */
		public NaoWifiAccessPoint(int _channel,String _SSID,String _BSSID,int _RSSI,int _WEP) {
			channel = _channel;
			frequency = channelToFrequency(_channel);
			SSID = _SSID;
			BSSID = _BSSID.replaceAll(":", "");
			RSSI = _RSSI;
			WEP = _WEP;
		}
		
		/**
		 * initializes all members to default values
		 */
		private void reset() {
			channel = 0;
			frequency = 0;
			SSID = "";
			BSSID = "";
			RSSI = 0;
			WEP = 2;
		}
		
		/**
		 * sets all members from a scanresult object
		 */
		public void set(ScanResult sr) {
			if (sr != null) {
				frequency = sr.frequency;
				channel = frequencyToChannel(sr.frequency);
				SSID = sr.SSID;
				BSSID = sr.BSSID.replaceAll(":", "").substring(0,12);
				RSSI = sr.level;
				if(getScanResultSecurity(sr) == null) {
				   WEP = 2;	
				} else {
				   WEP = (getScanResultSecurity(sr).contains(OPEN))?0:1;
				}
			}
		}

		@Override
		public int hashCode() {
			return (this.BSSID.hashCode());
		}

		@Override
		public boolean equals(Object o) {
			return (this.BSSID.equals(o));
		}

		public int compareTo(NaoWifiAccessPoint another) {
			return (this.BSSID.compareTo(another.BSSID));
		}
	}; // end of class definition : NaoWifiAccessPoint
	
	/** Array of scan results */
	private TreeSet<NaoWifiAccessPoint> mScanResultList;
	/** System timestamp of the beginning of scan, in milliseconds since boot. */
	private long mTimeStampStart;
	/** System timestamp of the end of scan, in milliseconds since boot. */
	private long mTimeStampEnd;
	/** String representation of the measurement */
	private String mStrValue;  
	
   /** Constants used for different security types. */
   public static final String WPA2 = "WPA2";
   public static final String WPA = "WPA";
   public static final String WEP = "WEP";
   public static final String OPEN = "Open";

	/**
	 * 
	 * WifiMeasurement constructor : initializes an empty object 
	 */
	public WifiMeasurement() {
		mScanResultList = new TreeSet<NaoWifiAccessPoint>();
		reset();
	}
	
	
	/**
	 * Copy constructor
	 * @param wm : input measurement
	 */
	public WifiMeasurement(WifiMeasurement wm) {
		mScanResultList = new TreeSet<NaoWifiAccessPoint>();
		copyOf(wm);
	}

	
	/**
	 * constructor from parcel
	 * @param p : input Parcel
	 */
	public WifiMeasurement(Parcel p) {
		this();
		readFromParcel(p);
	}


	/**
	 * Reset all data
	 */
	private void reset() {
		mTimeStampStart = 0;
		mTimeStampEnd = 0;
		mScanResultList.clear();
		mStrValue = "";
	}
	
	/**
	 * Adds entries of a list of scan results and sets the timestamps
	 * @param list : input list of ScanResult
	 * @param tsStartMillis : date of start of scan in milliseconds since system boot
	 * @param tsEndMillis : date of end of scan in milliseconds since system boot
	 */
	public void add(List<ScanResult> list,long tsStartMillis,long tsEndMillis) {
		if (list != null) {
			for (int k = 0;k<list.size();k++) {
				mScanResultList.add(new NaoWifiAccessPoint(list.get(k)));
			}
			mTimeStampStart = tsStartMillis;
			mTimeStampEnd = tsEndMillis;
		} // else nothing to do
	}
	
	/**
	 * get the number of AP
	 * @return the number of AP in the list
	 */
	public int getNumberOfAccessPoints() {
		return mScanResultList.size();
	}

	/**
	 * Get an iterator over the set of access points
	 */
	public Iterator<NaoWifiAccessPoint> getAccessPointIterator() {
		return mScanResultList.iterator();
	}
	
	/**
	 * Add an access point to the hashset, or replace its parameters if already there
	 * @param channel 802.11b/g channel number
	 * @param SSID network name
	 * @param BSSID most often the MAC address of the hardware interface - formatted as '0123456789AB'
	 * @param RSSI RSSI level
	 * @return true if the hashset did not already contain the AP, false else
	 */
	public boolean addAccessPoint(int channel,String SSID,String BSSID,int RSSI,int WEP) {
		NaoWifiAccessPoint wap = new NaoWifiAccessPoint(channel,SSID,BSSID,RSSI,WEP);
		if (!mScanResultList.add(wap)) {
			// remove first, then add it back
			mScanResultList.remove(wap);
			mScanResultList.add(wap);
			return false;
		} else {
			return true;
		}
	}
	
	public void setTimeStampStart(long ts) {
		mTimeStampStart = ts;
	}

	public void setTimeStampEnd(long ts) {
		mTimeStampEnd = ts;
	}
	
	/* (non-Javadoc)
	 * @see com.polestar.models.Measurement#toHumanString()
	 */
	
	public String toHumanString() 
	{
		mStrValue = "";
		
		// summary
		mStrValue = String.valueOf(mScanResultList.size())+"AP spotted (scan lasted "+String.valueOf(getDuration()) + " ms)\n";
		// all AP
		Iterator<NaoWifiAccessPoint> it = mScanResultList.iterator();
		while (it.hasNext()) {
			NaoWifiAccessPoint ap = it.next();
			mStrValue = mStrValue + ap.BSSID +" "+String.valueOf(ap.RSSI)+"dBm ["+ap.SSID  +"] "+ap.frequency+"MHz   " + ((ap.WEP==0)?"(Free)":"(Privacy)")+ "\n";
		}
		
		return mStrValue;
	}
	
	/* (non-Javadoc)
	 * @see com.polestar.models.Measurement#toByteArray()
	 */
	@Override
	public byte[] toByteArray() 
	{
		// compute array length :
		// 12 = 8+4 bytes
		// 49 = 12+2+32+1+2+1 bytes (this is a maximum size, SSID is <= 32 bytes)
		int serializedBytes = 12 + ((mScanResultList!=null)?mScanResultList.size()*51:0);
		
		// create new output stream with default size
		ByteArrayOutputStream bos = new ByteArrayOutputStream(serializedBytes);
		DataOutputStream dos = new DataOutputStream(bos);
		// add date (hex (16) value )
		try {
			dos.writeLong(mTimeStampEnd);//8 bytes
			if(mScanResultList!=null) {
				// add number of APs (decimal (3))
				dos.writeInt(mScanResultList.size()); // 4 bytes, 12 total
				//Log.alwaysWarn(this.getClass().getName(), "nb APs: " + mScanResultList.size());//DBG
				Iterator<NaoWifiAccessPoint> it = getAccessPointIterator();
				while (it.hasNext()) {
					NaoWifiAccessPoint tmp_objet = it.next();
					if( tmp_objet instanceof NaoWifiAccessPoint )
					{
						// add BSSID as a 12-char string
						dos.writeBytes(tmp_objet.BSSID);
						if (tmp_objet.SSID.length() > (33-1) ) {
							// SSID too long - truncate to expected max in native
							tmp_objet.SSID = tmp_objet.SSID.substring(0, (33-1) );
						}
						// add number of characters in SSID string (2)
						dos.writeShort((short) (tmp_objet.SSID.length()));
						// add SSID string value (length = numberSSID)
						dos.writeBytes(tmp_objet.SSID);
						// add channel (1)
						dos.writeByte(tmp_objet.channel);
						// add RSSI (2)
						dos.writeShort((short) (Math.abs(tmp_objet.RSSI)));
						//add Wep (1)
						dos.writeByte(tmp_objet.WEP);
					}
				}
			} else {
				// list is null => 0 access points
				dos.writeInt(0); //12 bytes total
			}
			
			dos.flush();	
			dos.close();	
		} catch (IOException ie) {
			// error : impossible to write in the stream
			return null;
		}
		
		return bos.toByteArray();		
	} // end of toByteArray

	
	/**
	 * Converts a 802.11 channel number into a frequency
	 * @param _802_11_channel : 802.11b/g channel number (1-14)
	 * @return the center frequency in MHz, or 0 if channel is not between 1 and 14
	 */
	public static int channelToFrequency(int _802_11_channel) {
		if (_802_11_channel<1) {
			return 0;
		} else if (_802_11_channel < 14) {
			return 2407+5*_802_11_channel;
		} else if (_802_11_channel == 14) {
			return 2484;
		} else if (_802_11_channel >= 36) {
			return 5000;
		} else {
			return 0;
		}
	}
	

	/**
	 * Converts a a frequency into a 802.11b/g channel numbeer
	 * @param frequency frequency in MHz
	 * @return the 802.11b/g channel number (1-14)
	 */
	public static int frequencyToChannel(int frequency) {
		if (frequency<=2412) {
			return 1;
		} else if (frequency <= 2474 ) {
			float ch = (float) (frequency-2407);
			return Math.round(ch*0.2f);
		} else if (frequency <= 2484 ){
			return 14;
		} else if (frequency <= 5180 ){
			return 36;
		}  else if (frequency <= 5320){
			return ((((frequency - 5180)/20) * 4) + 36);
		} else if (frequency <= 5500 ){
			return 100;
		}  else if (frequency <= 5805 ){
			return ((((frequency - 5500)/20) * 4) + 100);
		} else {
			return 0;
		}
			
	}
	
	
	/**
	 * Copies the contents of a Wifi Measurement into this object
	 * @param wm : input measurement object
	 */
	public void copyOf(WifiMeasurement wm) {
		this.mStrValue = wm.mStrValue;
		this.mTimeStampStart = wm.mTimeStampStart;
		this.mTimeStampEnd = wm.mTimeStampEnd;
		this.mScanResultList.clear();
		Iterator<NaoWifiAccessPoint> it = wm.mScanResultList.iterator();
		while (it.hasNext()) {
			this.mScanResultList.add(it.next());
		}
	}

	public long getTimeStampStart() {
		return mTimeStampStart;
	}

	/**
	 * @return the duration of the scan in [ms]
	 */
	public long getDuration() {
		return mTimeStampEnd-mTimeStampStart;
	}
	
	public void clear() {
		this.mStrValue = "";
		this.mTimeStampStart = 0;
		this.mTimeStampEnd = 0;
		this.mScanResultList.clear();
	}


	/**
	 * Returns the hashcode for the WifiMeasurement class
	 * (constant : WifiMeasurement.PARCELINDENTIFIER)
	 */
	public int describeContents() {
		return WifiMeasurement.PARCELINDENTIFIER;
	}
	
	public void writeToParcel(Parcel dest, int flags) {
		dest.writeLong(this.mTimeStampStart);
		dest.writeInt(this.getNumberOfAccessPoints());
		
		// parcours du tableau d'AP --> écriture channel, BSSID et SSID et RSSI
		for (NaoWifiAccessPoint a:this.mScanResultList) {
			dest.writeString(a.SSID);
			dest.writeString(a.BSSID);
			dest.writeInt(a.RSSI);
			dest.writeInt(a.channel);
			dest.writeInt(a.WEP);
			
		}
	}
	
	
	private void readFromParcel(Parcel src) {

		// clean up first
		this.mScanResultList.clear();
		
		//read timstamp
		this.mTimeStampStart = src.readLong(); 
		// read the number of AP stored
		int nAP = src.readInt();

		// read the list of access points
		for(int i = 0; i < nAP; i++) {
			NaoWifiAccessPoint a = new NaoWifiAccessPoint();
			a.SSID = src.readString();
			a.BSSID = src.readString();
			a.RSSI = src.readInt();
			a.channel = src.readInt();
			a.WEP = src.readInt();
			this.mScanResultList.add(a);
		}

	}
	
	 /**
	   * @return The security of a given Scan Result.
	   */
	  public static String getScanResultSecurity(ScanResult scanResult) {
	    if(scanResult.capabilities == null) {
	    	return null;
	    }
		  
		final String cap = scanResult.capabilities;
	    final String[] securityModes = { WEP, WPA, WPA2 };
	    for (int i = securityModes.length - 1; i >= 0; i--) {
	      if (cap.contains(securityModes[i])) {
	        return securityModes[i];
	      }
	    }
	    return OPEN;
	  }
}
