package com.polestar.models;

import com.polestar.helpers.Log;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Vector;


public class MemsMeasurement extends Measurement  {
	
	private long mTimeStamp;    // timestamps of the last memsData added
	private int mLength;
	private boolean isMagFiledAvailable;
	private boolean isAccelAvailable;
	private boolean isGravityAvailable;
	private boolean isGyroAvailable;
	private boolean isRotationAvailable;
	private Vector<MemsData> memsData;
	
	public MemsMeasurement() {
	    isMagFiledAvailable = false;
	    isAccelAvailable  = false;
	    isGravityAvailable  = false;
	    isGyroAvailable  = false;
	    isRotationAvailable = false;
	    mLength = 0;
		mTimeStamp = 0;
	    memsData = new Vector<MemsData>();
	}
	
	/**
	 * Add a memsData object
	 * we suppose the data is added in chronologic order
	 * @param data
	 */
	public void addMemsData(MemsData data){
		memsData.add(data);
		mTimeStamp = data.timestamp;
	}

	@Override
	public byte[] toByteArray() {
		// compute array length :
		// 3 + (nbr of MemsData * 76), 76 is the size in bytes of a memsData object
		int serializedBytes = 3 + ((memsData==null)? 0:memsData.size()*80);
		
		// create new output stream with default size
		ByteArrayOutputStream bos = new ByteArrayOutputStream(serializedBytes);
		DataOutputStream dos = new DataOutputStream(bos);
		
		byte dataContents = (byte) ((isMagFiledAvailable ? 0 : 1)  +
                ((isGravityAvailable ? 0 : 1) << 1) +
                ((isAccelAvailable ? 0 : 1) << 2) +
                ((isRotationAvailable ? 0 : 1) << 3) +
                ((isGyroAvailable ? 0 : 1) << 4));
		
		// Writing in the buffer
		try {
			dos.writeByte(dataContents); // 1 bit
			dos.writeShort(getNbMeas()); // 2 bits, total 3
			
			for (Iterator<MemsData> iterator = memsData.iterator(); iterator.hasNext();) {
				MemsData ap = (MemsData) iterator.next();
		        
				// add date=64 bits (8)
		        dos.writeLong(ap.timestamp);
		        
				// add Accel x,y, and z
				dos.writeInt((int)(ap.acceleration[0] * 1e07));
		        dos.writeInt((int)(ap.acceleration[1] * 1e07));
		        dos.writeInt((int)(ap.acceleration[2] * 1e07));
		        //Log.alwaysWarn(this.getClass().getName(), "Accel " + ap.acceleration[0] + " " + ap.acceleration[1] + " " +ap.acceleration[2]);//DBG
		        //Log.alwaysWarn(this.getClass().getName(), "Accel " + (double)((int)(ap.acceleration[0] * 1e07)*1e-07) + " " + (double)((int)(ap.acceleration[1] * 1e07)*1e-07) + " " +(double)((int)(ap.acceleration[2] * 1e07)*1e-07));//DBG
		        
		        // add gyro
				dos.writeInt((int)(ap.rotation[0] * 1e07));
		        dos.writeInt((int)(ap.rotation[1] * 1e07));
		        dos.writeInt((int)(ap.rotation[2] * 1e07));
		        //Log.alwaysWarn(this.getClass().getName(), "Rotation " + ap.Roll+ " " +ap.Pitch+ " " + ap.Yaw);//DBG
		        
		        //add Gravity
		        dos.writeInt((int)(ap.gravity[0] * 1e07));
		        dos.writeInt((int)(ap.gravity[1] * 1e07));
		        dos.writeInt((int)(ap.gravity[2] * 1e07));
		        //Log.alwaysWarn(this.getClass().getName(), "Grav " + ap.gravity[0]+ " " +ap.gravity[1]+ " " + ap.gravity[2]);//DBG
		        
		        //add user accel
		        dos.writeInt((int)(ap.userAcceleration[0] * 1e07));
		        dos.writeInt((int)(ap.userAcceleration[1] * 1e07));
		        dos.writeInt((int)(ap.userAcceleration[2] * 1e07));
		        
		        //add rotation rate
		        dos.writeInt((int)(ap.rotationRate[0] * 1e07));
		        dos.writeInt((int)(ap.rotationRate[1] * 1e07));
		        dos.writeInt((int)(ap.rotationRate[2] * 1e07));
		        //Log.alwaysWarn(this.getClass().getName(), "RotRate " + ap.rotRateX + " " + ap.rotRateY+ " " +ap.rotRateZ);//DBG
		        
		        //add magnetic field
		        dos.writeShort((short)(ap.magField[0] * 1e02));
		        dos.writeShort((short)(ap.magField[1] * 1e02));
		        dos.writeShort((short)(ap.magField[2] * 1e02));
		        //Log.alwaysWarn(this.getClass().getName(), "Mag " + ap.magFiledX+ " " +ap.magFiledY+ " " +ap.magFiledZ);//DBG
		        
		        //add heading info
		        dos.writeShort((short)(ap.trueHeading * 1e02));
		        dos.writeShort((short)(ap.magHeading * 1e02));
		        dos.writeShort((short)(ap.headingAccuracy * 1e02));
			}
			if (dos.size() != serializedBytes){
				Log.alwaysWarn(this.getClass().getName(), " We only send " + dos.size() + "/" + serializedBytes + " bytes, something's wrong!");
			}
			
			dos.flush();	
			dos.close();	
		} catch (IOException ie) {
			// error : impossible to write in the stream
			return null;
		}
		
		mLength = serializedBytes;
		
		return bos.toByteArray();
	}

	@Override
	public String toHumanString() {
		return "MemsMeasurement :: nbMEas " + getNbMeas() + 
				" Accel " + isAccelAvailable  + 
				" Grav " + isGravityAvailable +
				" Mag " + isMagFiledAvailable +
				" Gyro " + isGyroAvailable 
				;
	}
	
	
	
	/********************************************
	 * SETTERS & GETTERS
	 ********************************************/
	
	public int getNbMeas(){
		return (memsData == null) ? 0 : memsData.size();
	}

	public long getmTimeStamp() {
		return mTimeStamp;
	}

	public void setmTimeStamp(long mTimeStamp) {
		this.mTimeStamp = mTimeStamp;
	}

	public int getmLength() {
		return mLength;
	}
	
	public int getLength() {
		return mLength;
	}

	public void setmLength(int mLength) {
		this.mLength = mLength;
	}

	public boolean isMagFiledAvailable() {
		return isMagFiledAvailable;
	}

	public void setMagFiledAvailable(boolean isMagFiledAvailable) {
		this.isMagFiledAvailable = isMagFiledAvailable;
	}

	public boolean isAccelAvailable() {
		return isAccelAvailable;
	}

	public void setAccelAvailable(boolean isAccelAvailable) {
		this.isAccelAvailable = isAccelAvailable;
	}

	public boolean isGravityAvailable() {
		return isGravityAvailable;
	}

	public void setGravityAvailable(boolean isGravityAvailable) {
		this.isGravityAvailable = isGravityAvailable;
	}

	public boolean isGyroAvailable() {
		return isGyroAvailable;
	}

	public void setGyroAvailable(boolean isGyroAvailable) {
		this.isGyroAvailable = isGyroAvailable;
	}

	public boolean isRotationAvailable() {
		return isRotationAvailable;
	}

	public void setRotationAvailable(boolean isRotationAvailable) {
		this.isRotationAvailable = isRotationAvailable;
	}

	public Vector<MemsData> getMemsData() {
		return memsData;
	}

	public void setMemsData(Vector<MemsData> memsData) {
		this.memsData = memsData;
	}
	
	

}
