package com.polestar.models;
//____________________________________________________________________________________
//
//  LogFile.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________

import com.polestar.helpers.Log;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;


/**
 * Manages text logs to files
 * Implementation : thread-safe singleton
 *     
 * <br/><br/><b>Revision history:</b><br/>
 * <table border>
 * <tr>
 *   <td>Author</td>
 *   <td>Modification date</td>
 *   <td>Tracking Number</td>
 *   <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>8 octobre 2010</td>
 *   <td>trunk</td>
 *   <td>creation</td>
 * </tr>
 * </table>    
 * @author sterrenoir
 */
public class LogFile {
	/** File is closed or has not been opened yet */
	public static final int STATUS_CLOSED = 0;
	/** File is open in write mode */
	public static final int STATUS_OPENWRITE = 1;
	/** A critical error has happened, the file can not be read from/written to anymore */
	public static final int STATUS_ERROR = 2;

	/** Internal variables for file management: */ 
	private FileWriter logFileWriter;
	private String logFileName = null;
	private int logStatus = STATUS_CLOSED;
	
	/** data formatting */
	static SimpleDateFormat logDateFormat = null;
	
	/** singleton instance */
	static LogFile instance;
	
	/**
	 * get the single instance of logfile
	 */
	public static LogFile getInstance() {
		if (instance == null) {
			synchronized (LogFile.class) {
				// second check within the sync block ("Double-Check Locking")
				if (instance == null) {
					instance = new LogFile();
				}
			}
		}
		return instance;
	}
	
	/**
	 * LogFile private and void constructor.
	 */
	private LogFile() {
	}
		
	/**
	 * Opens the text log file in write mode. Log level must be at least DEBUG.
	 * @return the status after the operation (one of the STATUS_XXX constants)
	 * 
	 * Has no effect if status is not STATUS_CLOSED
	 * Does not overwrite existing files.
	 */
	public int openWrite(String fileName) {
		// check initial status
		if (logStatus != STATUS_CLOSED) {
			return logStatus;
		}
		
		if (!Log.getIsRestrictedLogActive()) {
			return logStatus;
		}
		
		logFileName = fileName;
		
		// check input file name
		if (logFileName == null) {
			Log.alwaysError(this.getClass().getName(),"Impossible to open text log file : (null)");
			logStatus = STATUS_ERROR;
		} else {
			try {
				logFileWriter = new FileWriter(logFileName,true);
				logStatus = STATUS_OPENWRITE;
			} catch (IOException ie) {
				Log.alwaysError(this.getClass().getName(),"Impossible to open text log file (IOException) : "+ie.getMessage());
				logStatus = STATUS_ERROR;
			}
		}		
		return logStatus;
	} // end of openWrite

	/**
	 * close the file
	 * @return the status after the operation
	 */
	public int close() {
		if(logStatus != STATUS_OPENWRITE) {	// can't close
			return logStatus;
		}
		
		try {
			synchronized (LogFile.class) {
				logFileWriter.close();
				logStatus = STATUS_CLOSED;
			}
		} catch (IOException ie) {
			Log.alwaysError(this.getClass().getName(),"Impossible to close text log file (IOException) : "+ie.getMessage());
			logStatus = STATUS_ERROR;
		}	
		return logStatus;
	}
	
	@Override
	public String toString() {
		return "LogFile "+logFileName;
	}
	
	/**
	 * Write a string to the log file.<br/>
	 * Appends the current date as a prefix, and an end-of-line as a suffix to the string if not already present.
	 * @param s : string to write in the log file.
	 * @return the status after the operation
	 */
	public int write(String s) {
		if(logStatus != STATUS_OPENWRITE) {
			// can't write
			return logStatus;
		}
		if (logFileWriter == null) {
			logStatus = STATUS_ERROR;
			return logStatus;
		}
		synchronized (LogFile.class) {
			try {
				logFileWriter.write(s);
				if (!s.endsWith("\n")) {
					logFileWriter.write("\n");
				}
				logFileWriter.flush();
			} catch (IOException ie) {
				Log.alwaysError(this.getClass().getName(),"Impossible to write into the log file (IOException) : "+ie.getMessage());
				logStatus = STATUS_ERROR;
			}
		}
		return logStatus;
	}
	
	public int getStatus() {
		return logStatus;
	}

	public String getFileName() {
		return logFileName;
	}

	/**
	 * Get the current date and build a nice string.<br/>format :
	 * <pre>yyyy-MM-dd-hh:mm:ss</pre>
	 * @return the string generated
	 */
	public static String getDateString()
	{	
		if (logDateFormat == null) {
			logDateFormat = new SimpleDateFormat("yyyy-MM-dd-hh:mm:ss");
		}
		return logDateFormat.format(new Date());
	}
	/**
	 * Remove all files from the target directory
	 * @param targetDirectory absolute path of the target directory
	 */
	public static void deleteAllInDirectory(String targetDirectory) {
		File directory = new File(targetDirectory);
		directory.mkdirs(); // create the directory and all parents if needed
		// Get all files in directory
		File[] files = directory.listFiles();
		for (File file : files)	{
		   // Delete each file
		   if (!file.delete()){
		       // Failed to delete file
			   Log.alwaysError(LogFile.class.getName(), "Failed to delete " + file);
		   }
		}
	}
	
}
