//______________________________________________________________________________________
//
//  GpsMeasurement.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________
package com.polestar.models;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import android.location.GpsSatellite;
import android.location.GpsStatus;
import android.location.Location;
import android.location.LocationManager;

/**
 * Defines a simple class for GPS measurement data<br/>
 *    
 * <br/><br/><b>Revision history:</b><br/>
 * <table border>
 * <tr>
 *   <td>Author</td>
 *   <td>Modification date</td>
 *   <td>Tracking Number</td>
 *   <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 *   <td>sterrenoir</td>
 *   <td>15 mars 2010</td>
 *   <td>trunk</td>
 *   <td>Creation of this class</td>
 * </tr>
 * </table>    
 * @author sterrenoir
 */
public class GpsMeasurement extends Measurement 
{
	public static final int SATELLITE_STATUS_KO = 0;
	public static final int SATELLITE_STATUS_HAS_ALMANAC = 1;
	public static final int SATELLITE_STATUS_HAS_EPHEMERIS = 2;
	public static final int SATELLITE_STATUS_USED_IN_FIX = 3;
	public static final double DEG_2_RAD_FACTOR = 0.0174532925199432949;
	/** t0 of GPS time (00:00:00, Jan. 6th, 1980) in UNIX time */
	public static final long GPS_TIME_ZERO_MS = 315986400;
	
	public static final double LATLON_SCALE = (1.0e6);
	public static final double ALT_SCALE = (1.0e3);
	public static final double ACCURACY_SCALE = 4.0f;
	public static final float ELAZCN0_SCALE = (10.0f);
	public static final float SPEED_SCALE = (float) (1.0e2);
	
	// members : see below NaoGpsSatellite class definition
	/**
	 * 
	 * A custom class to hold satellite data
	 *        
	 * @author sterrenoir
	 */
	public class NaoGpsSatellite 
	{
		/** Satellite azimuth in degrees */
		public float azimuth;
		/** Satellite elevation in degrees */
		public float elevation;
		/** satellite PRN */
		public int prn;
		/** satellite C/N0 in dBHz */
		public float cn0;
		/** satellite status */
		private int status;

		/**
		 * default constructor (all members to 0)
		 */
		public NaoGpsSatellite() 
		{
			reset();
		}
		
		/**
		 * Constructor from a GpsSatellite object
		 */
		public NaoGpsSatellite(GpsSatellite inputSvData) {
			reset();
			set(inputSvData);
		}

		/**
		 * Constructor with all members
		 */
		public NaoGpsSatellite(float _azimuth,float _elevation,int _prn,float _cn0,int  _status) {
			azimuth = _azimuth;
			elevation = _elevation;
			prn = _prn;
			cn0 = _cn0;
			status = _status;
		}
		
		/**
		 * initializes all members to default values
		 */
		private void reset() {
			azimuth = 0;
			elevation = 0;
			prn = 0;
			cn0 = 0;
			status = SATELLITE_STATUS_KO;
		}
		
		/**
		 * sets all members from a GpsSatellite object
		 */
		public void set(GpsSatellite inputSvData) {
			if (inputSvData != null) {
				azimuth = inputSvData.getAzimuth();
				elevation = inputSvData.getElevation();
				prn = inputSvData.getPrn();
				cn0 = inputSvData.getSnr();
				if (inputSvData.usedInFix()) {
					status = SATELLITE_STATUS_USED_IN_FIX;
				} else if (inputSvData.hasEphemeris()) {
					status = SATELLITE_STATUS_HAS_EPHEMERIS;
				} else if (inputSvData.hasAlmanac()) {
					status = SATELLITE_STATUS_HAS_ALMANAC;
				} else {
					status = SATELLITE_STATUS_KO;
				}
			}
		} // end of set()
		
		/**
		 * get the status
		 */
		public int getStatus() {
			return status;
		}
	}; // end of class definition : NaoGpsSatellite
	
	// GpsMeasurement members
	/** GPS mode :
	 * 	<li>0 = invalid
	 *  <li>1 = GPS fix
	 *  <li>2 = DGPS fix
	 *  <li>3 = PPS fix
	 *  <li>4 = RTK
	 *  <li>5 = Float RTK
	 *  <li>6 = estimated (dead reckoning)
	 *  <li>7 = Manual input
	 *  <li>8 = Simulation
	 */
	private int mGpsMode;
	/** GPS fix system timestamp in milliseconds */
	private long mSystemTimestamp;
	/** list of satellites */
	private List<NaoGpsSatellite> mSatellites;
	/** location fix */
	private Location mLocation;
	
	/**
	 * 
	 * GpsMeasurement constructor : initializes members.
	 */
	public GpsMeasurement() {
		mGpsMode = 0;
		mSystemTimestamp = 0;
		mSatellites = new ArrayList<NaoGpsSatellite>();
		mLocation = new Location(LocationManager.GPS_PROVIDER);
	}
		
	
	/**
	 * Copy constructor
	 * @param gm : input measurement
	 */
	public GpsMeasurement(GpsMeasurement gm) {
		mSatellites = new ArrayList<NaoGpsSatellite>();
		copyOf(gm);
	}
	
	
	/**
	 * 
	 * sets new location data
	 * @param systemTimestamp : system time associated with the location in milliseconds since 1/1/1970
	 * @param newLocation : new location object to be recorded in this instance
	 */
	public void setLocation(long systemTimestamp,Location newLocation) {
		mLocation.reset();
		mSystemTimestamp = 0;
		// copy location data
		if (newLocation != null) {
			mLocation = newLocation;
			mSystemTimestamp = systemTimestamp;
		}
	}
	
	
	public void setGpsMode(int mode) {
		mGpsMode = mode;
	}
		
	
	/**
	 * 
	 * sets the GPS satellites received
	 * @param newStatus : new GPS status object to be recorded in this instance
	 */
	public void setSatellites(GpsStatus newStatus) {
		mSatellites.clear();
		// copy satellite data
		if (newStatus != null) {
			Iterator<GpsSatellite> itSv = newStatus.getSatellites().iterator();
			while (itSv.hasNext()) {
				mSatellites.add(new NaoGpsSatellite(itSv.next()));
			}
		}
	}
	
	
	/**
	 * Add a satellite to the satellite list
	 * @param azimuth satellite azimuth in degrees
	 * @param elevation satellite elevation in degrees
	 * @param prn satellite PRN
	 * @param cn0 satellite C/N0 in dBHz
	 * @param status satellite status
	 */
	public void addSatellite(float azimuth,float elevation,int prn,float cn0,int status) {
		mSatellites.add(new NaoGpsSatellite(azimuth,elevation,prn,cn0,status));
	}
	

	/* (non-Javadoc)
	 * @see com.polestar.models.Measurement#toHumanString()
	 */
	@Override
	public String toHumanString() {
		String hstr = "";
		
		// timestamp
		if (mLocation == null) {
			hstr += "Error in GpsMeasurement.toHumanString() :  mLocation is null\n";
		} else { 
			Date timestamp = new Date(mLocation.getTime());
			hstr += "Date "+timestamp.toLocaleString()+"\n";
			// location
			hstr += "Location :\n\tlat. "+String.valueOf(mLocation.getLatitude())+"°\n\tlon. "+String.valueOf(mLocation.getLongitude())+"°\n\talt. "+String.valueOf(mLocation.getAltitude())+"\n";
		}
		// satellites
		if (mSatellites == null) {
			hstr += "Error in GpsMeasurement.toHumanString() :  mSatellites is null\n";
		} else if (mSatellites.isEmpty()) {
			hstr += "No satellites\n";
		} else {
			hstr += "SV used for fix :\n";
			for (int k=0;k<mSatellites.size();k++) {
				if (mSatellites.get(k).getStatus() == SATELLITE_STATUS_USED_IN_FIX) {
					hstr += "\t"+String.valueOf(mSatellites.get(k).prn)+" ("+String.valueOf(mSatellites.get(k).cn0)+" dBHz)\n";
				}	
			}
		}

		return hstr;
	}

	/* (non-Javadoc)
	 * @see com.polestar.models.Measurement#toByteArray()
	 */
	@Override
	public byte[] toByteArray() {
		// compute array length :
		// 28 = 8+18+2+5 bytes
		// 9 = 2+2+2+2+1 bytes
		int serializedBytes = 33 + ((mSatellites!=null)?mSatellites.size()*9:0);
		
		// create new output stream with default size
		ByteArrayOutputStream bos = new ByteArrayOutputStream(serializedBytes);
		DataOutputStream dos = new DataOutputStream(bos);
		
		// add date (hex (16) value )
		try {
			// timestamp
			dos.writeLong(mSystemTimestamp);
			
			// location fix data
			if (mLocation != null) {
				dos.writeShort((short) computeGpsWeekNumber(mLocation.getTime()));
				dos.writeInt(computeGpsTimeOfWeek(mLocation.getTime()));
				dos.writeInt((int) Math.round(mLocation.getLatitude()*LATLON_SCALE));
				dos.writeInt((int) Math.round(mLocation.getLongitude()*LATLON_SCALE));
				dos.writeInt((int) Math.round(mLocation.getAltitude()*ALT_SCALE));
				//add gps mode
				dos.writeByte(mGpsMode);
				//add accuracy v/h
				float Accuracy_h = mLocation.getAccuracy();
				float Accuracy_v = 0;
				dos.writeShort((int)(Accuracy_h * ACCURACY_SCALE));
				dos.writeShort((int)(Accuracy_v * ACCURACY_SCALE));
				} else {
				dos.writeShort(0);
				dos.writeInt(0);
				dos.writeInt(0);
				dos.writeInt(0);
				dos.writeInt(0);
				dos.writeByte(0);
				dos.writeShort(0);
				dos.writeShort(0);
			}				
			
			if(mSatellites!=null) {
				if(mSatellites!=null) {
					// add number of sat
					dos.writeShort(mSatellites.size());
					// loop over number of satellites
					for (int k = 0;k<mSatellites.size();k++) {
						// add PRN
						dos.writeShort(mSatellites.get(k).prn);
						// add satellite elevation
						dos.writeShort(Math.round(mSatellites.get(k).elevation*ELAZCN0_SCALE));
						// add satellite azimuth and set it into the [0;360°[ range first
			            // note : 0.002777777778 == 1/360.0
			            float azDeg = mSatellites.get(k).azimuth;
			            if (azDeg < 0) {
			                azDeg += 360.0*Math.ceil((-azDeg)*0.002777777778);
			            } else if (azDeg >= 360.0) {
			                azDeg -= 360.0*Math.floor(azDeg*0.002777777778);
			            }
						dos.writeShort(Math.round(azDeg*ELAZCN0_SCALE));
						// add satellite signal power
						dos.writeShort(Math.round(mSatellites.get(k).cn0*ELAZCN0_SCALE));
						// add satellite status used in fix (0 or 1)
						dos.writeByte((mSatellites.get(k).getStatus()==SATELLITE_STATUS_USED_IN_FIX)?1:0);
					}
				}
				else {
					// 0 satellites
					dos.writeShort(0);
				}
			} else {
				// list is null => 0 access points
				dos.writeInt(0);
			}
			
			dos.flush();	
			dos.close();	
		} catch (IOException ie) {
			// error : impossible to write in the stream
			return null;
		}
		
		return bos.toByteArray();
	} // end of toByteArray
	
	
	/**
	 * Get the number of satellites in the list
	 */
	public int getNumberOfSatellites() {
		if (mSatellites == null)
		{
			return 0;
		}
		else
		{
			return mSatellites.size();
		}
	}
	
	
	/**
	 * Get any satellite from the list of satellites
	 * @param index : index of the target satellite in the list
	 * @return the target satellite, or null if error
	 */
	public NaoGpsSatellite getSatellite(int index) {
		if (mSatellites == null)
		{
			return null;
		}
		
		try {
			return mSatellites.get(index);
		} catch (IndexOutOfBoundsException e) {
			return null;
		}
	}

	public long getystemTimeStamp() {
		return mSystemTimestamp;
	}

	public int getMode() {
		return mGpsMode;
	}
	
	public Location getLocation() {
		return mLocation;
	}
	
	/**
	 * Build the unix timestamp from GPS WN and TOW
	 * @param gpsWeekNumber GPS week number (weeks since jan 6th, 1980)
	 * @param gpsTimeOfWeek GPS Time Of Week in seconds
	 * @return unix timestamp in milliseconds
	 */
	public static long computeUnixTime(int gpsWeekNumber, long gpsTimeOfWeek) {
		return GpsMeasurement.GPS_TIME_ZERO_MS + ((long) gpsWeekNumber)*((long) 86400000) + gpsTimeOfWeek;
	}
	
	/**
	 * compute the GPS WN from a UNIX timestamp
	 * @param unixTimeStamp time in milliseconds
	 * @return the GPS week number
	 */
	public static int computeGpsWeekNumber(long unixTimeStamp) {
		return (int) ((unixTimeStamp-GpsMeasurement.GPS_TIME_ZERO_MS)/((long) 86400000));
	}

	/**
	 * compute the GPS TOW from a UNIX timestamp
	 * @param unixTimeStamp time in milliseconds
	 * @return the GPS Time Of Week in milliseconds
	 */
	public static int computeGpsTimeOfWeek(long unixTimeStamp) {
		return (int) ((unixTimeStamp-GpsMeasurement.GPS_TIME_ZERO_MS)%((long) 86400000));
	}
	
	/**
	 * Copies the contents of a GPS Measurement into this object
	 * @param gm : input measurement object
	 */
	public void copyOf(GpsMeasurement gm) {
		this.mGpsMode = gm.mGpsMode;
		this.mSystemTimestamp = gm.mSystemTimestamp;
		this.mLocation = new Location(gm.mLocation);
		this.mSatellites.clear();
		for (int k=0;k<gm.mSatellites.size();k++) {
			this.mSatellites.add(gm.mSatellites.get(k));
		}
	}
}
