package com.polestar.models;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import com.polestar.enums.NRProximityEnum;

public class Beacon implements Comparable<Beacon>{

	public static List<Beacon> list = new ArrayList<Beacon>(); // this list will always be ordered by proximity (immediate, near, far, not seen by rssi)
	public static Beacon current = null; // currently used beacon (for displaying purpose, etc)
	public static List<Beacon> seen = new ArrayList<Beacon>();

	private int id; // can be negatif if not yet stored on cloud
	private int babid;
	private String name;
	private String serviceUUID;
	private String macAddress;
	private Date lastSeendDate;
	private Ranger ranger; // update proximity based on new rssi measurement
	private String label;
	

	// used when scan
	public Beacon(int babid){
		this.id = getTempId();// negative
		this.babid = babid;
		list.add(this);
	}

	public static int getTempId(){
		int maxId = 0;
		for (Iterator<Beacon> iterator = list.iterator(); iterator.hasNext();) {
			Beacon beacon = (Beacon) iterator.next();
			if (Math.abs(beacon.getId()) > maxId){
				maxId = Math.abs(beacon.getId()) ;
			}
		}
		return -maxId -1;
	}

	// used when create from nao cloud list
	public Beacon(int id, String serviceUUID, int babid, String macAddress, String name, String label) {
		this.id = id;
		this.serviceUUID = serviceUUID;
		this.babid = babid;
		this.macAddress = macAddress;
		this.name = name;
		this.label = label;
		list.add(this);
	}

	public boolean isSeen(){
		return ranger != null;
	}

	public boolean enterIfNecessary(){
		boolean res = false;
		if (!isSeen()){
			ranger = new Ranger(this);
			res = true;
		}
		return res;
	}

	// new measure when already enter
	public void range(int rssi){
		lastSeendDate = new Date();
		if (isSeen()){
			ranger.range(rssi);
		}
	}

	public static void sort(){
		Collections.sort(list);
	}

	public boolean hasChangedProximity(){
		return isSeen() && ranger.hasChangedProximity();
	}

	// not seen anymore for a long time
	public boolean exitIfNecessary(){
		boolean res = false;
		if (isSeen() && ((new Date().getTime() - lastSeendDate.getTime())/1000 > GlobalAlgoParams.instance().getInt(GlobalAlgoParams.MAX_TIME_NOT_SEEN_BEACON_BEFORE_EXIT_IN_S))){
			ranger = null; // exited
			res = true;
		}
		return res;
	}

	public NRProximityEnum getProximity(){
		return ranger == null ? NRProximityEnum.NRProximityUnknown : ranger.getProximity();
	}

	public String toString(){
		return label + " (" + babid + ")";
	}

	public List<NAORegion> regions(){
		return BeaconRegionPair.regionsForBeacon(this);
	}


	// sort comparaison
	// -1 to be before in the list
	@Override
	public int compareTo(Beacon another) {
		if (!isSeen() && !another.isSeen()){
			return 0;
		}
		if (!isSeen() && another.isSeen()){
			return 1;
		}
		if (isSeen() && !another.isSeen()){
			return -1;
		}
		if (isSeen() && another.isSeen()){
			return  another.ranger.getLastRssi() - ranger.getLastRssi(); // rssi values are negative
		}
		return 0;
	}





	/**
	 * getters setters
	 */


	public int getId() {
		return id;
	}

	public String label(){
		return label + "[" + babid + "]";
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getServiceUUID() {
		return serviceUUID;
	}

	public int getBabid() {
		return babid;
	}

	public String getMacAddress() {
		return macAddress;
	}

	public Ranger getRanger() {
		return ranger;
	}

	public Date getLastSeenDate(){
		return lastSeendDate;
	}
	

	public String getLabel() {
		return label;
	}

	public void setLabel(String label) {
		this.label = label;
	}

	public static Beacon findById(int id) {
		for (Iterator<Beacon> iterator = list.iterator(); iterator.hasNext();) {
			Beacon beacon = (Beacon) iterator.next();
			if (beacon.id == id){
				return beacon;
			}
		}
		return null;
	}

	public static void addAllBeaconToARegion(NAORegion region){
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			Beacon beacon = (Beacon) iterator.next();
			new BeaconRegionPair(beacon, region);
		}
	}

	public void resetProximity() {
		ranger = null;
	}

}
