//______________________________________________________________________________________
//
//  KmlPath.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________
package com.polestar.helpers;

import java.io.File;
import java.io.IOException;
import java.util.Vector;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;


import android.location.Location;


/**
 * This class reads paths from KML files.
 * 
 * <br/>
 * <br/>
 * <b>Revision history:</b><br/>
 * <table border>
 * <tr>
 * <td>Author</td>
 * <td>Modification date</td>
 * <td>Tracking Number</td>
 * <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 * <td>sterrenoir</td>
 * <td>26 octobre 2010</td>
 * <td>trunk</td>
 * <td>Creation of this class from logger.KmlReader</td>
 * </tr>
 * </table>
 * 
 * @author sterrenoir
 */
public class KmlPath extends DefaultHandler {
	// constants
	private static final String _COORDINATES_LABEL = "coordinates";
	private static final String _PLACEMARK_LABEL = "Placemark";
	private static final String _HACCURACY_LABEL = "hAccuracy";
	
	// members
	/** Boolean used during parsing to tell if a read character belongs to the "coordinates" section */
	private boolean withinCoordinates;
	/** Boolean used during parsing to tell if a read character belongs to the "Placemark" section */
	private boolean withinPlacemark;
	/** Boolean used during parsing to tell if a read character belongs to the "Placemark" section */
	private boolean withinAccuracy;
	/** Accuracy to apply to the current placemark, in meters */
	private float currentAccuracy;
	/** String that serves as reading buffer to accumulate characters on a line, before interpreting them */
	private String coordinatesLine;
	/** Internal temporary storage for point coordinates as they get read: */
	Vector<Location> points;
	
	/** Constructor, to initialize references to the needed context */
	public KmlPath() {
		withinCoordinates = false;
		withinPlacemark = false;
		withinAccuracy = false;
		currentAccuracy = 0.0f;
		points = new Vector<Location>();
		coordinatesLine = new String();
	};
	
	/** 
	 * Reads path points from a file.
	 * 
	 * @param path the (absolute) path of the KML file to be read
	 * @return true if the file is successfully read
	 */
	public boolean readFromFile(String path) {
		if (path == null) {
			Log.alwaysError(this.getClass().getName(), "Error : null file name");
			return false;
		}
		
		boolean success = false;
		try {
			// Prepare the parser:
			// Note: current Android emulator and probably phone do not support any setting for the parser.
			SAXParserFactory myParserConfigurator = SAXParserFactory.newInstance();
			SAXParser myParser= myParserConfigurator.newSAXParser();

			// Now parse the file:
			myParser.parse(new File(path), this);
			success = true;
		} catch (SAXException e) {
			// Could not get a parser "not because of configuration". Looks bad.
			Log.alwaysError(this.getClass().getName(), "Unable to get a parser instance for KML file or XML structure problem! "+e.getMessage());
		} catch (ParserConfigurationException e) {
			// Could not get a parser. Looks bad.
			Log.alwaysError(this.getClass().getName(), "Unable to get a parser instance for KML file, indeed! "+e.getMessage());
		} catch (FactoryConfigurationError e) {
			Log.alwaysError(this.getClass().getName(), "Unable to get an XML parser: " + e.getMessage());
		} catch (IOException e) {
			Log.alwaysError(this.getClass().getName(), "Error while reading KML file: " + e.getMessage());
		}
		
		return success;
	} // readFromFile

	/**
	 * Callback for the parser to signal each time a new KML element is encountered.
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
		if(localName.equalsIgnoreCase(_COORDINATES_LABEL)) {
			withinCoordinates = true;
		} else if (localName.equalsIgnoreCase(_PLACEMARK_LABEL)) {
			withinPlacemark = true;
		}  else if (localName.equalsIgnoreCase(_HACCURACY_LABEL)) {
			withinAccuracy = true;
			currentAccuracy = 0.0f;
		}// else ignore other tags in the file.
	}
	
	
	/** Callback for the parser to signal each time a new character is encountered between start and end tags. */
	//FIXME: cannot handle coordinates separated by "end of line" (only separated by space): 
	//Actually it does not read the last character when separated by "end of line"
	public void characters(char[] ch, int start, int length) throws SAXException {
		if (withinPlacemark && withinCoordinates) {
			int myStart = start; // marks the start of a line while reading through "ch"
			int charIndex = start;
			while (charIndex < start + length) {
				// Check if we have an end of line:
				if (  (ch[charIndex] == '\r') || (ch[charIndex] == '\n')  || (ch[charIndex] == ' ')) {
					if (  (charIndex > myStart) || (coordinatesLine.length()>0)  ) {
						// There are unprocessed characters either in "ch" or in "coordinatesLine"
						if (charIndex > myStart) {
							// Add start-to-charIndex to the buffer:
								coordinatesLine = coordinatesLine.concat(new String(ch, myStart, charIndex-myStart+1));
						}
						coordinatesLine = coordinatesLine.trim();
						if (coordinatesLine.length()>0) {
							// Interpret the line:
							addPointFromLine(coordinatesLine);
						}
						// Remove the line from the buffer:
						coordinatesLine = new String();
					} // else this is an empty line, just skip over it.
					// Move myStart over the end of line:
					myStart = charIndex+1;
				} // else this is just a standard character - skip over it until 
				  // a line is completed or the last character is read.
				charIndex++;
			}
			if (myStart < start + length - 1) {
				// Some characters were read after the last carriage return, but did not get interpreted.
				// Buffer them for use either in next call to this function, or in endElement():
				coordinatesLine = coordinatesLine.concat(new String(ch, myStart, charIndex-myStart));
			}
		} else if (withinPlacemark && withinAccuracy) {
			// within a "hAccuracy" tag
			String s = new String(ch,start,length);
			try {
				currentAccuracy = Float.parseFloat(s);		
			} catch (NumberFormatException e) {
				Log.alwaysWarn(this.getClass().getName(), "Cannot read the accuracy for the current placemark");
			}	
		}
		
	}
	
	/** Callback for the parser to signal when a closing tag is found in the file */
	public void endElement(String uri, String localName, String qName) throws SAXException {
		if(localName.equalsIgnoreCase(_COORDINATES_LABEL)) {
			withinCoordinates = false;
			coordinatesLine = coordinatesLine.trim();
			if (coordinatesLine.length()>0)
				addPointFromLine(coordinatesLine);
		} if(localName.equalsIgnoreCase(_PLACEMARK_LABEL)) {
			withinPlacemark = false;
			currentAccuracy = 0.0f;
		} if(localName.equalsIgnoreCase(_HACCURACY_LABEL)) {
			withinAccuracy = false;
		} // else ignore other tags in the file.
	}

	private void addPointFromLine(String line) {
		if (false == withinCoordinates)
			return;
		
		// Expecting a line formed "A.a,B.b,C.c"
		// Double-check the number of dots and commas, in case locals have force decimal pointer to be different:
		if (line.matches(".*(,).*(,).*(,).*(,).*(,).*")) {
			// Unexpected locals have forced coordinates to be written with "," as decimal separator.
			// Replace the unexpected commas with dots:
			int first_decimal = line.indexOf(',', 0);
			int second_decimal = line.indexOf(',',   line.indexOf(',', first_decimal+1)  +1);
			int third_decimal = line.indexOf(',',   line.indexOf(',', second_decimal+1)  +1);
			line = line.substring(0, first_decimal-1) + "."
				+ line.substring(first_decimal+1, second_decimal-1) + "."
				+ line.substring(second_decimal+1, third_decimal-1) + "."
				+ line.substring(third_decimal+1);
		}
		if (line.matches(".*(.).*(,).*(.).*(,).*(.).*")) {
			try {
				// Parse all fields:
				int end_of_double = line.indexOf(',', 0);
				int end_of_previous_double = 0;
				double longitude = Double.parseDouble(line.substring(end_of_previous_double, end_of_double));
				end_of_previous_double = end_of_double;
				end_of_double = line.indexOf(',', end_of_double+1);//+1 to go after the comma
				double latitude = Double.parseDouble(line.substring(end_of_previous_double+1, end_of_double));
				end_of_previous_double = end_of_double;
				end_of_double = line.indexOf(',', end_of_double+1);//+1 to go after the comma
				if (end_of_double<0) {
					// No more comma, which is possible for altitude.
					end_of_double = line.length();
				}
				double altitude = Double.parseDouble(line.substring(end_of_previous_double+1, end_of_double));

				// Add to the points:
				Location l = new Location("KML");
				l.setLongitude(longitude);
				l.setLatitude(latitude);
				l.setAltitude(altitude);
				
				// if accuracy is set : use it
				if (currentAccuracy > 0) {
					l.setAccuracy(currentAccuracy);
				}				
				points.add(l);
	
			} catch (NumberFormatException e) {
				Log.alwaysError(this.getClass().getName(), "coordinates cannot be converted to Double: "+line+"; error: "+e.getMessage());
			}
		} else {
			Log.alwaysError(this.getClass().getName(), "Unexpected coordinates format: "+line);
		}
	}	
	
	/**
	 * get an element in the location vector
	 * @param index index of the target element
	 * @return the target Location, or null
	 */
	public Location get(int index) {
		if ((index >= 0) && (points != null) && (index < points.size())) {
			return points.get(index);
		} else {
			return null;
		}
	}
	
	/**
	 * @return the number of elements in the vector of positions
	 */
	public int size() {
		if (points == null) {
			return 0;
		} else {
			return points.size();
		}
	}
	
} // class KmlPath
