package com.polestar.helpers;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class FileHelper implements FilenameFilter {


    /**
     * Get all the files which has the given name
     *
     * @param name
     * @param folderPath
     * @return
     */
    public static ArrayList<String> getFilesByName(String name, String folderPath) {
        if (name == null || name.contentEquals("")) {
            return null;
        }
        ArrayList<String> paths = new ArrayList<String>();
        ArrayList<String> subfolderPaths = FileHelper.getSubfolderPaths(folderPath);
        for (String path : subfolderPaths) {
            File[] subFolderContent = new File(path).listFiles();
            for (File current : subFolderContent) {
                if (current.getName().contains(name.substring(0, name.lastIndexOf(".")))) {
                    paths.add(current.getAbsolutePath());
                }
            }
        }
        return paths;

    }

    /**
     * Check if file exists at path
     *
     * @param path
     * @return
     */
    public static boolean exist(String path) {
        return ((new File(path)).exists());
    }


    /**
     * Get all the files from directory
     *
     * @param folderPath
     * @return
     */
    public static ArrayList<String> getFilesName(String folderPath) {
        if (folderPath == null || folderPath.contentEquals("")) {
            return null;
        }
        ArrayList<String> paths = new ArrayList<String>();
        File[] folderContent = new File(folderPath).listFiles();
        for (File current : folderContent) {
            paths.add(current.getAbsolutePath());
        }
        return paths;
    }

    /**
     * getNameFromPath
     *
     * @param folderPath
     * @return
     */
    public static String getNameFromPath(String folderPath) {
        if (folderPath == null || folderPath.contentEquals("")) {
            return null;
        }

        File folderContent = new File(folderPath);
        return folderContent.getName();

    }


    /**
     * get the xmls files in a every sub folders within recursive
     *
     * @param folderPath
     * @return
     */
    public static ArrayList<String> getXMLPathsInSubFolders(String folderPath) {
        ArrayList<String> paths = new ArrayList<String>();
        ArrayList<String> subfolderPaths = FileHelper
                .getSubfolderPaths(folderPath);
        for (String path : subfolderPaths) {
            paths.addAll(FileHelper.getPathsByExtension("xml", path));
        }
        return paths;
    }


    /**
     * Get absolute paths sub folders
     *
     * @param folderPath
     * @return
     */
    public static ArrayList<String> getSubfolderPaths(String folderPath) {
        if (folderPath == null) {
            return null;
        }
        ArrayList<String> paths = new ArrayList<String>();
        File folder = new File(folderPath);
        if (folder.exists()) {
            File[] filesList = folder.listFiles();
            if (filesList != null && filesList.length > 0) {
                Arrays.sort(filesList);
                for (File tempFile : filesList) {
                    if (tempFile.isDirectory()
                            && !tempFile.getName().equalsIgnoreCase(".svn")) {
                        paths.add(tempFile.getAbsolutePath());
                    }
                }
            }
            return paths;
        }
        return null;
    }


    /**
     * filter files by name matching a regex
     *
     * @param folderPath
     * @param regex
     * @return
     */
    public static ArrayList<String> filterByName(String folderPath, String regex) {
        ArrayList<String> filtered = new ArrayList<String>();
        File folder = new File(folderPath);
        if (folder.exists()) {
            File[] files = folder.listFiles();
            for (File file : files) {
                if (file.getName().matches(regex)) {
                    filtered.add(file.getAbsolutePath());
                }
            }
        }
        return filtered;
    }


    /**
     * append a text at the end of a file
     *
     * @param filePath
     * @param text
     * @throws IOException
     */
    public static void append(String filePath, String text) throws IOException {
        BufferedWriter bw = null;
        try {
            bw = new BufferedWriter(new FileWriter(filePath, true));
            bw.write(text);
            bw.newLine();
            bw.flush();
        } finally { // always close the file
            if (bw != null)
                try {
                    bw.close();
                } catch (IOException ioe2) {
                    // just ignore it
                }
        } // end try/catch/finally
    }


    /**
     * Move files
     *
     * @param srcFile
     * @param dstFile
     * @throws Exception
     */
    public static void move(String srcFile, String dstFile) throws Exception {
        File src = new File(srcFile);
        if (src.exists()) {
            File dst = new File(dstFile);
            if (!dst.exists()) {
                dst.mkdir();
            }

            if (src.isDirectory()) {
                copyDirectory(src, dst);
            } else {
                copyFile(src, dst);
            }
            delete(src);
        }


    }


    /**
     * copy files
     *
     * @param source
     * @param dest
     * @throws Exception
     */
    public static void copyFile(String source, String dest) throws Exception {
        copyFile(new File(source), new File(dest));
    }


    /**
     * rename all files with the same root name but with different extensions
     * cmd ren <path_to_file>.* newName.*
     *
     * @param srcPath
     * @param newName
     * @throws Exception
     */
    public static void rename(String srcPath, String newName) throws Exception {
        File src = new File(srcPath);
        if (src.exists()) {
            String dirPath = src.getParent();
            if (src.isDirectory()) {
                copyDirectory(srcPath, dirPath + "\\" + newName);
                ;
            } else {
                copyFile(srcPath, dirPath + "\\" + newName);
            }
            delete(srcPath);
        }
    }


    /**
     * Delete a file or directory
     *
     * @param path
     * @return
     * @throws Exception
     */
    public static boolean delete(String path) throws Exception {
        return delete(new File(path));
    }


    /**
     * Get all the file with a given extension in a folder
     *
     * @param extension without '.'
     * @param folderPath
     * @return
     */
    public static File[] getFilesByExtension(String extension, String folderPath) {
        File folder = new File(folderPath);
        FilenameFilter filter = new FileHelper("." + extension);
        File[] list = folder.listFiles(filter);
        return list;
    }


    /**
     * Get all the file with a given extension in a folder
     *
     * @param extension
     * @param folderPath
     * @return
     */
    public static ArrayList<String> getPathsByExtension(String extension,
                                                        String folderPath) {
        ArrayList<String> paths = new ArrayList<String>();
        File folder = new File(folderPath);
        FilenameFilter filter = new FileHelper("." + extension);
        File[] list = folder.listFiles(filter);
        if (list != null) {
            for (File file : list) {
                paths.add(file.getAbsolutePath());
            }
        }
        return paths;
    }


    /**
     * @param sourceDir
     * @param destDir
     * @throws IOException
     */
    public static void copyDirectory(String sourceDir, String destDir) throws IOException {
        copyDirectory(new File(sourceDir), new File(destDir));
    }


    /**
     * Copy directory to another
     *
     * @param sourceDir
     * @param destDir
     * @throws IOException
     */
    public static void copyDirectory(File sourceDir, File destDir)
            throws IOException {
        if (!destDir.exists()) {
            destDir.mkdir();
        }
        File[] children = sourceDir.listFiles();
        for (File sourceChild : children) {
            String name = sourceChild.getName();
            File destChild = new File(destDir, name);
            if (sourceChild.isDirectory()) {
                copyDirectory(sourceChild, destChild);
            } else {
                copyFile(sourceChild, destChild);
            }
        }
    }


    /**
     * @param source
     * @param dest
     * @throws IOException
     */
    public static void copyFile(File source, File dest) throws IOException {
        if (source.exists()) {
            if (!dest.exists()) {
                dest.createNewFile();
            }
            InputStream in = null;
            OutputStream out = null;
            try {
                in = new FileInputStream(source);
                out = new FileOutputStream(dest);
                byte[] buf = new byte[1024];
                int len;
                while ((len = in.read(buf)) > 0) {
                    out.write(buf, 0, len);
                }
            } finally {
                if (in != null) {
                    in.close();
                }
                if (out != null) {
                    out.close();
                }

            }
        }
    }

    /**
     * @param resource
     * @return
     * @throws IOException
     */
    public static boolean delete(File resource) throws IOException {
        if (resource.isDirectory()) {
            File[] childFiles = resource.listFiles();
            if (childFiles != null) {
                for (File child : childFiles) {
                    delete(child);
                }
            }
        }
        return resource.delete();
    }


    /************************
     * to be a extension filter :D
     ************************/
    private String extension;

    public FileHelper(String extension) {
        this.extension = extension;
    }

    public boolean accept(File directory, String filename) {
        return filename.endsWith(extension);
    }

    public static final void mkdir(String path) {
        if (!((new File(path)).exists())) {
            new File(path).mkdirs();
        }
    }

    public static void rmdir(String path) {
        File directory = new File(path);
        if (directory.isDirectory()) {
            for (File file : directory.listFiles()) {
                if (file.isDirectory()) {
                    rmdir(file.getAbsolutePath());
                } else {
                    file.delete();
                }
            }
            directory.delete();
        }

    }


    public static byte[] getDataFromFile(String path) {
        File file = new File(path);

        if (file.exists() == false) {
            return null;
        }

        byte[] fileData = new byte[(int) file.length()];

        try {
            DataInputStream dis = new DataInputStream(new FileInputStream(file));
            dis.readFully(fileData);
            dis.close();
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        //delete file timestamp
        file.delete();

        return fileData;
    }

    /**
     * Recursively delete all the empty folders inside the given file (if it is
     * a folder), EXCEPT for itself.
     *
     * @param file The parent.
     */
    public static void deleteAllEmptyFoldersExceptRoot(final File file) {
        if (file.exists() && file.isDirectory()) {
            File[] children = file.listFiles();

            int childrenLength = children.length;
            for (int i = 0; i < childrenLength; i++) {
                deleteAllEmptyFolders(children[i]);
            }
        }
    }

    /**
     * Recursively delete all the empty folders inside the given file (if it is
     * a folder), itself included.
     *
     * @param file The parent.
     */
    public static void deleteAllEmptyFolders(final File file) {
        if (file.isDirectory()) {
            File[] children = file.listFiles();
            int childrenLength = children.length;

            if (childrenLength == 0) {
                // file is an empty folder, so we delete it.
                file.delete();
            } else {
                // file is a non empty folder, so we recursively invoke
                // deleteAllEmptyFolders on its children...
                for (int i = 0; i < childrenLength; i++) {
                    deleteAllEmptyFolders(children[i]);
                }

                // ... and then, if it has no more children, it is empty and
                // thus deleted.
                if (file.listFiles().length == 0) {
                    file.delete();
                }
            }
        }
    }

    /**
     * Append string to file
     * @param stringToAdd
     * @param filePath
     * @throws IOException
     */
    public static void addStringToFile(String stringToAdd, String filePath) throws IOException {
        if (!FileHelper.exist(filePath)){
            writeStringAsFile(stringToAdd, filePath);
        }else{
            File file = new File(filePath);
            FileOutputStream outStream = new FileOutputStream(file, true);
            OutputStreamWriter outStreamWriter = new OutputStreamWriter(outStream);
            outStreamWriter.append(stringToAdd);
            outStreamWriter.flush();
        }
    }

    /**
     * Write string in a new file
     * @param fileContents
     * @param filePath
     * @throws IOException
     */
    public static void writeStringAsFile(final String fileContents, String filePath) throws IOException {
        File file = new File(filePath);
        String folder = getFolderPath(filePath);
        if (!FileHelper.exist(folder)){
            FileHelper.mkdir(folder);
        }
        FileWriter out = new FileWriter(file);
        out.write(fileContents);
        out.close();
    }

    public static String getFolderPath(String filePath){
        File file = new File(filePath);
        String folder = file.getParentFile().getAbsolutePath();
        return folder;
    }

    /**
     * Get extension from a given filename
     * @param fileName
     * @return
     */
    public static String getFileExtension(String fileName) {
        String extension = "";

        int i = fileName.lastIndexOf('.');
        int p = Math.max(fileName.lastIndexOf('/'), fileName.lastIndexOf('\\'));

        if (i > p) {
            extension = fileName.substring(i+1);
        }

        return extension;
    }

    /**
     * Get file content as string
     * @param filePath
     * @return
     * @throws IOException
     */
    public static String readFileAsString(String filePath) throws IOException {
        StringBuffer fileData = new StringBuffer();
        BufferedReader reader = new BufferedReader(
                new FileReader(filePath));
        char[] buf = new char[1024];
        int numRead=0;
        while((numRead=reader.read(buf)) != -1){
            String readData = String.valueOf(buf, 0, numRead);
            fileData.append(readData);
        }
        reader.close();
        return fileData.toString();
    }
    public static List<File> getFileRecursively(File root) {
        final List<File> results = new ArrayList<File>();
        File[] files = root.listFiles();
        if(files != null) {
            for(File file : files) {
                if(file.isDirectory()) {
                    results.addAll(getFileRecursively(file));
                } else if(file.isFile()) {
                    results.add(file);
                }
            }
        }

        return results;
    }

}
