package com.polestar.helpers;

import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothManager;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.wifi.WifiManager;

import com.polestar.naosdk.api.IConnectivityHelper;

/**
 * Created by asimonigh on 08/02/2017.
 */

public class ConnectivityHelper {

	private static final int BLE_SLEEP_PERIOD = 200;
	private static final int BLE_TIMEOUT = 3000;

	private static final String TAG = ConnectivityHelper.class.getSimpleName();

	//********************************************************************
	//Bluetooth
	//********************************************************************

	private static BluetoothAdapter getBluetoothAdapter(Context ctxt){
		BluetoothAdapter bluetooth;
		// the sensors
		BluetoothManager bluetoothManager = (BluetoothManager) ctxt.getApplicationContext().getSystemService(Context.BLUETOOTH_SERVICE);
		if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN_MR2) {
			bluetooth = bluetoothManager.getAdapter();
		} else {
			bluetooth = BluetoothAdapter.getDefaultAdapter();
		}

		return bluetooth;
	}

	public static boolean isOnline(Context context){
		ConnectivityManager cm =
				(ConnectivityManager)context.getSystemService(Context.CONNECTIVITY_SERVICE);

		NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
		return activeNetwork != null &&
				activeNetwork.isConnectedOrConnecting();
	}

	public synchronized static boolean turnBleOFF(Context ctxt){
		BluetoothAdapter bluetooth = getBluetoothAdapter(ctxt);
		if(null == bluetooth){
			Log.alwaysError(TAG, "Cannot get Bluetooth adapter");
			return false;
		}
		// stop only if hardware is on
		if(bluetooth.getState() == android.bluetooth.BluetoothAdapter.STATE_ON || bluetooth.getState() == android.bluetooth.BluetoothAdapter.STATE_TURNING_ON) {

			Log.alwaysWarn(TAG, "bluetooth.disable()");
			bluetooth.disable();

			Log.alwaysWarn(TAG, "Waiting for bluetooth hardware to get stopped...");

			// wait for chipset to be started
			int periodCount = 0 ;
			while(bluetooth.getState() != android.bluetooth.BluetoothAdapter.STATE_OFF && periodCount<=BLE_TIMEOUT) {
				try {
					Thread.sleep(BLE_SLEEP_PERIOD) ;
					periodCount += BLE_SLEEP_PERIOD;
					Log.alwaysWarn(TAG, "Waiting for bluetooth hardware to get stopped...");
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}

			if(bluetooth.getState() == android.bluetooth.BluetoothAdapter.STATE_OFF){
				Log.alwaysWarn(TAG, "Bluetooth hardware stopped !");
				return true ;
			} else {
				Log.alwaysWarn(TAG, "Failed to stop bluetooth hardware !");
				return false ;
			}

		} else {
			Log.alwaysWarn(TAG, "Bluetooth hardware already stopped !");
			return true ;
		}
	}


	public synchronized static boolean turnBleON(Context ctxt){
		BluetoothAdapter bluetooth = getBluetoothAdapter(ctxt);
		if(null == bluetooth){
			Log.alwaysError(TAG, "Cannot get Bluetooth adapter");
			return false;
		}

		// start only if hardware is off
		if(bluetooth.getState() != android.bluetooth.BluetoothAdapter.STATE_ON && bluetooth.getState() != android.bluetooth.BluetoothAdapter.STATE_TURNING_ON) {

			Log.alwaysWarn(TAG, "bluetooth.enable()");
			bluetooth.enable();

			Log.alwaysWarn(TAG, "Waiting for bluetooth hardware to get started...");

			// wait for chipset to be started
			int periodCount = 0 ;
			while(bluetooth.getState() != android.bluetooth.BluetoothAdapter.STATE_ON && periodCount<=BLE_TIMEOUT) {
				try {
					Thread.sleep(BLE_SLEEP_PERIOD) ;
					periodCount += BLE_SLEEP_PERIOD;
					Log.alwaysWarn(TAG, "Waiting for bluetooth hardware to get started...");
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}

			if(bluetooth.getState() == android.bluetooth.BluetoothAdapter.STATE_ON){
				Log.alwaysWarn(TAG, "Bluetooth hardware started !");
				return true ;
			} else {
				Log.alwaysWarn(TAG, "Failed to start bluetooth hardware !");
				return false ;
			}

		} else {
			Log.alwaysWarn(TAG, "Bluetooth hardware already started !");
			return true ;
		}

	}

	private static boolean isBleOn(Context ctxt){
		return getBluetoothAdapter(ctxt).getState() == android.bluetooth.BluetoothAdapter.STATE_ON;
	}

	//********************************************************************
	//Wifi
	//********************************************************************

	private static WifiManager getWifiManager(Context ctxt){
		return (WifiManager) ctxt.getSystemService(Context.WIFI_SERVICE);
	}

	/**
	 * Initialize the Wifi Sensor, starts the Wifi if needed.
	 * @return true if OK, false else
	 */
	public synchronized static boolean turnWifiON(Context ctxt) {
		WifiManager wifiManager = getWifiManager(ctxt);
		if (null == wifiManager) {
			Log.alwaysError(TAG, "Cannot get WiFi manager");
			return false;
		}
		wifiManager.setWifiEnabled(true);
		// wait at most 10 seconds until Wi-Fi turns on
		for (int k = 0; (k < 40) && (!wifiManager.isWifiEnabled()); k++) {
			try {
				Thread.sleep(250);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		if (wifiManager.isWifiEnabled()) {
			return true;
		} else {
			return false;
		}
	}

	public synchronized static boolean turnWifiOFF(Context ctxt) {
		WifiManager wifiManager = getWifiManager(ctxt);
		if (null == wifiManager) {
			Log.alwaysError(TAG, "Cannot get WiFi manager");
			return false;
		}

		wifiManager.setWifiEnabled(false);
		// wait at most 10 seconds until Wi-Fi turns on
		for (int k = 0; (k < 40) && (wifiManager.isWifiEnabled()); k++) {
			try {
				Thread.sleep(250);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		if (!wifiManager.isWifiEnabled()) {
			return true;
		} else {
			return false;
		}
	}

	private static boolean isWifiON(Context context){
		return getWifiManager(context).isWifiEnabled();


	}

	public class ConnectivityHelperProxy extends IConnectivityHelper{

		private Context context;

		public ConnectivityHelperProxy(Context ctxt){
			context = ctxt.getApplicationContext();
		}

		@Override
		public boolean turnWiFiON() {
			return ConnectivityHelper.turnWifiON(context);
		}

		@Override
		public boolean turnWifiOFF() {
			return ConnectivityHelper.turnWifiOFF(context);
		}

		@Override
		public boolean isWifiON() {
			return ConnectivityHelper.isWifiON(context);
		}

		@Override
		public boolean turnBleON() {
			return ConnectivityHelper.turnBleON(context);
		}

		@Override
		public boolean turnBleOff() {
			return ConnectivityHelper.turnBleOFF(context);
		}

		@Override
		public boolean isBleON() {
			return ConnectivityHelper.isBleOn(context);
		}

		@Override
		public boolean isOnline() {
			return ConnectivityHelper.isOnline(context);
		}
	}

}
