//____________________________________________________________________________________
//
//  AssetCopier.java
// 
//  Pole Star Confidential Proprietary
//    Copyright (c) Pole Star 2010, All Rights Reserved
//    No publication authorized. Reverse engineering prohibited
//______________________________________________________________________________________


package com.polestar.helpers;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;


import android.content.ContextWrapper;
import android.content.res.AssetManager;

/**
 * The class manages copying files from the asset directory into the SDcard
 *    
 * <br/><br/><b>Revision history:</b><br/>
 * <table border>
 * <tr>
 *   <td>Author</td>
 *   <td>Modification date</td>
 *   <td>Tracking Number</td>
 *   <td>Description of Change</td>
 * </tr>
 * <!-- add lines to the table for modifications, following the model -->
 * <tr>
 *   <td>Stéphane Terrenoir</td>
 *   <td>11 May 2010</td>
 *   <td>trunk</td>
 *   <td>Creation of this class</td>
 * </tr>
 * </table>    
 * @author sterrenoir
 */
public class AssetCopier {
	private static final String LOG_TAG = "com.polestar.helpers.AssetCopier";
	public static final String SDCARD_PATH = android.os.Environment.getExternalStorageDirectory().toString()+"/";
	private static final int COPY_BUFFER_LENGTH = 2000;

	private AssetManager assets;
	
	/**
	 * create a new AssetCopier
	 * @param cw reference to the calling Activity or Service
	 */
	public AssetCopier(ContextWrapper cw) {
		if (cw == null) {
			Log.alwaysError(this.getClass().getName(), "null reference in AssetCopier constructor");
			assets = null;
			return;
		}
		assets = cw.getAssets();
	}
	
	/**
	 * Create a directory at the target path
	 * @param path : name of the directory to create
	 */
    public static void createDirectory(String path) {
        Log.restricted(LOG_TAG,"creating directory : "+path);
        try {
        	new File(path).mkdirs();
        } catch (Exception e) {
        	Log.alwaysError(LOG_TAG,"ERROR : "+e.toString()+"\n");
        }
    }
    /**
     * Copies the target file from the source directory in the assets to the destination path. 
     * @param fileName name of the file (e.g. "algoConfig.txt")
     * @param fromPath path without final "/" (e.g. "cnf" or "")
     * @param toPath path without final "/" (e.g. "sdcard/.nao")
     */
    public void copyAssetFile(String fileName, String fromPath, String toPath) {
    	if (assets == null) {
    		return; // error
    	}
		String completeFileName;
		if (fromPath.length() > 0) {
			completeFileName = fromPath+"/"+fileName;
		} else {
			completeFileName = fileName;
		}
    	InputStream is = null;
    	OutputStream os = null;
    	byte [] b = new byte[COPY_BUFFER_LENGTH];
    	
    	// open input asset file
    	try {
			is = assets.open(completeFileName);
		} catch (IOException e) {
			Log.restricted(this.getClass().getName(),"ERROR in copyAssetFile(): "+e.toString());
			return;
		}
		
		// open output stream
    	try {
    		os = new FileOutputStream(toPath+"/"+fileName);
    	} catch (Exception e) {
			Log.restricted(this.getClass().getName(),"ERROR in copyAssetFile(): "+e.toString());
			try {
				is.close();
			} catch (IOException ie) {
				// void
			}
			return;
    	}
		
		try {
			int n;
			while ((n = is.read(b))>0) {
				os.write(b,0,n);
			}
			is.close();
			os.close();
		} catch (IOException e) {
			Log.restricted(this.getClass().getName(),"ERROR in copyAssetFile(): "+e.toString());
		}
    } // end of copyAssetFile
    /**
     * Copies the target file from the source directory in the assets to the destination path. 
     * @param fileName name of the file (e.g. "algoConfig.txt")
     * @param toPath path without final "/" (e.g. "sdcard/.nao")
     */
    public boolean copyAssetFile(String fileName, String toPath) {
    	if (assets == null) {
    		return false; // error
    	}


		
    	String completeFileName = fileName;

    	InputStream is = null;
    	OutputStream os = null;
    	byte [] b = new byte[COPY_BUFFER_LENGTH];
    	
    	// open input asset file
    	try {
			is = assets.open(completeFileName);
		} catch (IOException e) {
			Log.restricted(this.getClass().getName(),"ERROR in copyAssetFile(): "+e.toString());
			return false;
		}
		
		// open output stream
    	try {
    		os = new FileOutputStream(toPath+"/"+fileName);
    	} catch (Exception e) {
			Log.restricted(this.getClass().getName(),"ERROR in copyAssetFile(): "+e.toString());
			try {
				is.close();
			} catch (IOException ie) {
				// void
			}
			return false;
    	}
		
		try {
			int n;
			while ((n = is.read(b))>0) {
				os.write(b,0,n);
			}
			is.close();
			os.close();
		} catch (IOException e) {
			Log.alwaysError(this.getClass().getName(),"ERROR in copyAssetFile(): "+e.toString());
			return false;
		}

		return true;
    } // end of copyAssetFile

	/**
	 * this method check if need to force overwriting assets even if exist yet
	 * return true if file force_asset_overwrite.txt exist on application assets
	 */
	public boolean forceAssetsOverwrite() {
		// open input asset file
		InputStream is = null;
		try {
			is = assets.open("force_asset_overwrite.json");
		} catch (IOException e) {
			Log.alwaysError(this.getClass().getName(),"ERROR in forceAssetsOverwrite(): "+e.toString());
			return false;
		}

		try {
			is.close();
		} catch (IOException e) {
			Log.alwaysError(this.getClass().getName(),"ERROR in forceAssetsOverwrite(): "+e.toString());
		}

		return true;
	}

    /**
     * Utility function to copy files.
     * Unfortunately the need for AssetCopier to use the AssetManager file prevents it from using this function.
     * @param sourcePath
     * @param destinationPath
     */
    public static void copyFile(String sourcePath, String destinationPath) {
    	InputStream is = null;
    	OutputStream os = null;
    	byte [] b = new byte[COPY_BUFFER_LENGTH];
    	
    	// open input asset file
    	try {
			is = new FileInputStream(sourcePath);
		} catch (IOException e) {
			Log.alwaysError(LOG_TAG,"ERROR in copyFile: "+e.toString());
			return;
		}
		
		// open output stream
    	try {
    		os = new FileOutputStream(destinationPath);
    	} catch (Exception e) {
			Log.alwaysError(LOG_TAG,"ERROR in copyFile: "+e.toString());
			try {
				is.close();
			} catch (IOException ie) {
				// void
			}
			return;
    	}
		
		try {
			int n;
			while ((n = is.read(b))>0) {
				os.write(b,0,n);
			}
			is.close();
			os.close();
		} catch (IOException e) {
			Log.alwaysError(LOG_TAG,"ERROR in copyFile: "+e.toString());
		}
    }
    
    /**
     * Copies all the files in an asset directory into another directory
     * Destination directory is the concatenation of destinationRoot and relativePath
     * @param relativePath : source directory without final '/'
     * @param destinationRoot : destination directory without final '/'
     * @param replaceAll : if it is true, all the files from the assets are copied into the destination, else only files not present in the target directory are copied
     */
    public void copyAssetDirectory(String relativePath,String destinationRoot, boolean replaceAll) {
    	if (assets == null) {
    		return; // error
    	}
    	// create destination directory
    	String destinationDir = destinationRoot+"/"+relativePath;
    	createDirectory(destinationDir);
    	// list files
    	String [] fileList = null;
    	try {
    		fileList = assets.list(relativePath);
    	    for (int k=0;k<fileList.length;k++) {
				// check if file is present
				boolean filePresent = new File(destinationDir,fileList[k]).exists();
				// copy
				if (!filePresent || replaceAll)
					copyAssetFile(fileList[k],relativePath,destinationDir);
			}
			
		} catch (IOException e) {
			Log.alwaysError(this.getClass().getName(),"ERROR in copyAssetDirectory : "+e.toString());
		}
    }
    
    /**
     * Copies all the files in an asset directory into another directory
     * Destination directory is the concatenation of destinationRoot and relativePath
     * @param relativePath : source directory without final '/'
     * @param destinationRoot : destination directory without final '/'
     * @param replaceAll : if it is true, all the files from the assets are copied into the destination, else only files not present in the target directory are copied
     */
    public void copyAssetAllDirectories(String relativePath,String destinationRoot, boolean replaceAll,int level) {
    	if (assets == null) {
    		return; // error
    	}
 	   	
    	// list files
    	String [] fileList = null;
    	try {
    		fileList = assets.list(relativePath);
    		if(fileList.length == 0) {
    			boolean filePresent = new File(destinationRoot,relativePath).exists();
    			if (!filePresent || replaceAll) {
    				Log.alwaysWarn(this.getClass().getName(),"copyAssetFile : "+relativePath);
    				copyAssetFile(relativePath,destinationRoot);
    			}
    		} else {
    			// create destination directory
    	    	String destinationDir = destinationRoot+"/"+relativePath;
    	    	createDirectory(destinationDir);
    			for (int k=0;k<fileList.length;k++) {
    				if(!fileList[k].equals("webkit") && !fileList[k].equals("images") && !fileList[k].equals("sounds") && !fileList[k].contains(".jar")) {
    					Log.alwaysWarn(this.getClass().getName(),"copyAssetDirectory : "+relativePath + "/" + fileList[k]);
    					if(level >=1) {
    						copyAssetAllDirectories(relativePath + "/" + fileList[k],destinationRoot, replaceAll,level+1);
    					} else {
    						copyAssetAllDirectories(fileList[k],destinationRoot, replaceAll,level+1);
    					}
     				}
    			}
    		}
    		
			/*
    		for (int k=0;k<fileList.length;k++) {
				// check if file is present
				File file = new File(mapsFolder,"downloaded_file.png");
				boolean filePresent = new File(destinationDir,fileList[k]).exists();
				// copy
				if (!filePresent || replaceAll)
					copyAssetFile(fileList[k],relativePath,destinationDir);
			}
			*/
		} catch (IOException e) {
			Log.alwaysError(this.getClass().getName(),"ERROR in copyAssetDirectory : "+e.toString());
		}
    }
    
    
    /**
     * Remove all assets that may have been copied into the SD card
     * @param naoRootDirectory full path to the NAO root directory (e.g. "/sdcard/.nao")
     * @param naoCnfDirectory full path to the NAO cnf directory (e.g. "/sdcard/.nao/cnf")
     */
    public static void deleteAllAssetsFromSDCard(File naoRootDirectory,File naoCnfDirectory) {

    	// delete
		try {
			if (  (naoCnfDirectory != null) && (naoCnfDirectory.exists())  ) {
				// delete all files in cnf
				if (naoCnfDirectory.isDirectory()) {
					// list sub-directories. do not support further nested files.
					for (String sub_dir_entry : naoCnfDirectory.list() ) {
						File to_delete = new File(naoCnfDirectory.getCanonicalPath()+"/"+sub_dir_entry);
						if ((to_delete == null) || (false == to_delete.delete())) {
							Log.restricted(AssetCopier.class.getName(), "Error deleting file in "+naoCnfDirectory.getCanonicalPath());
						}
					}
				}
				// delete the directory
				if (false == naoCnfDirectory.delete()) {
					Log.restricted(AssetCopier.class.getName(), "Error deleting "+naoCnfDirectory.getCanonicalPath());
				} 
			}
			
			// now try to delete NAO root (e.g. //sdcard/.nao) (only works if directory is empty)
			if (false == naoRootDirectory.delete()) {
				Log.restricted(AssetCopier.class.getName(), "Error deleting "+naoRootDirectory.getCanonicalPath()+" (directory is probably not empty)");
			} 
			
		} catch (Exception e) {
			// Do nothing
			Log.restricted(AssetCopier.class.getName(), "Error managing data: "+e.getMessage());
		}
    }
}
