package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.entity.PaymentInfo;
import com.aitime.android.payment.payumoney.R;
import com.payumoney.core.PayUmoneySdkInitializer;
import com.payumoney.sdkui.ui.utils.PayUmoneyFlowManager;

import java.lang.ref.WeakReference;
import java.util.Map;

/**
 * PayUmoney payment
 */
@Payment(channel = PayChannel.PAYUMONEY)
public class PayUmoney implements IPayment {

    private final WeakReference<Activity> mActivity;

    public PayUmoney(Activity activity) {
        this.mActivity = new WeakReference<>(activity);
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {

    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code name}、{@code orderId}、{@code merchantId}、{@code merchantKey}、{@code hash}、{@code isDebug}、{@code surl} and {@code furl})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return;
        }
        String name = getParameter(params, "name");
        String orderId = getParameter(params, "orderId");
        String merchantId = getParameter(params, "merchantId");
        String merchantKey = getParameter(params, "merchantKey");
        String hash = getParameter(params, "hash");
        String isDebug = getParameter(params, "isDebug");
        String surl = getParameter(params, "surl");
        String furl = getParameter(params, "furl");
        try {
            String payName = info.getPayName();
            PayUmoneySdkInitializer.PaymentParam paymentParam = new PayUmoneySdkInitializer.PaymentParam.Builder()
                    .setAmount(info.getPayMoney())
                    .setTxnId(orderId)
                    .setPhone(info.getContact())
                    .setFirstName(name)
                    .setEmail(info.getEmail())
                    .setProductName(payName != null ? payName : "repayment")
                    .setsUrl(surl)
                    .setfUrl(furl)
                    .setIsDebug(Boolean.parseBoolean(isDebug))
                    .setKey(merchantKey)
                    .setMerchantId(merchantId)
                    .build();
            paymentParam.setMerchantHash(hash);
            PayUmoneyFlowManager.startPayUMoneyFlow(
                    paymentParam,
                    activity,
                    R.style.AppTheme_default,
                    false
            );
        } catch (Exception e) {
            e.printStackTrace();
            showToast(activity, e.getMessage());
        }
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return false;
        }
        String payuResponse = bundle.getString("payuResponse");
        String transactionStatus = bundle.getString("transactionStatus");
        if (payuResponse != null) {
            if ("SUCCESSFUL".equals(transactionStatus)) {
                return true;
            }
            if ("CANCELLED".equals(transactionStatus)) {
                showToast(activity, activity.getString(R.string.payment_status_cancelled));
                return false;
            }
            if ("FAILED".equals(transactionStatus)) {
                showToast(activity, activity.getString(R.string.payment_status_failed));
                return false;
            }
        }
        return false;
    }

    @Override
    public void release() {
        Application application = mActivity.get().getApplication();
        if (PayUmoneyFlowManager.isUserLoggedIn(application)) {
            PayUmoneyFlowManager.logoutUser(application);
        }
    }

    private void showToast(Activity activity, String text) {
        Toast.makeText(activity, text, Toast.LENGTH_SHORT).show();
    }

}
