package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;

import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.entity.PaymentInfo;
import com.aitime.android.payment.payumoney.R;
import com.payumoney.core.PayUmoneySdkInitializer;
import com.payumoney.sdkui.ui.utils.PayUmoneyFlowManager;

import java.util.Map;

/**
 * PayUmoney payment
 */
@Payment(channel = PayChannel.PAYUMONEY)
public class PayUmoney implements IPayment {

    private final Activity mActivity;

    public PayUmoney(Activity activity) {
        this.mActivity = activity;
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {

    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code merchantId}、{@code merchantKey}、{@code hash}、{@code isDebug}、{@code surl} and {@code furl})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        String merchantId = getParameter(params, "merchantId");
        String merchantKey = getParameter(params, "merchantKey");
        String hash = getParameter(params, "hash");
        String isDebug = getParameter(params, "isDebug");
        String surl = getParameter(params, "surl");
        String furl = getParameter(params, "furl");
        int labelRes = 0;
        try {
            Context context = mActivity.getApplicationContext();
            PackageManager packageManager = context.getPackageManager();
            PackageInfo packageInfo = packageManager.getPackageInfo(context.getPackageName(), 0);
            labelRes = packageInfo.applicationInfo.labelRes;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        try {
            String payName = info.getPayName();
            PayUmoneySdkInitializer.PaymentParam paymentParam = new PayUmoneySdkInitializer.PaymentParam.Builder()
                    .setAmount(info.getPayMoney())
                    .setTxnId(info.getOrderId())
                    .setPhone(info.getContact())
                    .setFirstName(info.getName())
                    .setEmail(info.getEmail())
                    .setProductName(payName != null ? payName : mActivity.getString(labelRes))
                    .setsUrl(surl)
                    .setfUrl(furl)
                    .setIsDebug(Boolean.parseBoolean(isDebug))
                    .setKey(merchantKey)
                    .setMerchantId(merchantId)
                    .build();
            paymentParam.setMerchantHash(hash);
            PayUmoneyFlowManager.startPayUMoneyFlow(
                    paymentParam,
                    mActivity,
                    R.style.AppTheme_default,
                    false
            );
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        String payuResponse = bundle.getString("payuResponse");
        String transactionStatus = bundle.getString("transactionStatus");
        if (payuResponse != null) {
            if ("SUCCESSFUL".equals(transactionStatus)) {
                return true;
            }
            if ("CANCELLED".equals(transactionStatus)) {
                showToast(R.string.payment_status_cancelled);
                return false;
            }
            if ("FAILED".equals(transactionStatus)) {
                showToast(R.string.payment_status_failed);
                return false;
            }
        }
        return false;
    }

    @Override
    public void release() {
        Application application = mActivity.getApplication();
        if (PayUmoneyFlowManager.isUserLoggedIn(application)) {
            PayUmoneyFlowManager.logoutUser(application);
        }
    }

    private void showToast(@StringRes int resId) {
        Toast.makeText(mActivity, resId, Toast.LENGTH_SHORT).show();
    }

}
