package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.text.TextUtils;
import android.webkit.WebView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.entity.PaymentInfo;
import com.aitime.android.payment.payumobile.R;
import com.payu.base.models.ErrorResponse;
import com.payu.base.models.PayUPaymentParams;
import com.payu.checkoutpro.PayUCheckoutPro;
import com.payu.checkoutpro.utils.PayUCheckoutProConstants;
import com.payu.ui.model.listeners.PayUCheckoutProListener;
import com.payu.ui.model.listeners.PayUHashGenerationListener;

import java.lang.ref.WeakReference;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

/**
 * PayUmobile payment
 */
@Payment(channel = PayChannel.PAYUMOBILE)
public class PayUmobile implements IPayment {

    private final WeakReference<Activity> mActivity;
    private IPaymentCallback mCallback;

    public PayUmobile(Activity activity) {
        this.mActivity = new WeakReference<>(activity);
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {
        this.mCallback = callback;
    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code name}、{@code product}、{@code orderId}、{@code merchantKey}、{@code hash}、{@code isDebug}、{@code userCredentials}、{@code surl} and {@code furl})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return;
        }
        String name = getParameter(params, "name");
        String amount = getParameter(params, "amount");
        String product = getParameter(params, "product");
        String orderId = getParameter(params, "orderId");
        String merchantKey = getParameter(params, "merchantKey");
        String hash = getParameter(params, "hash");
        String isDebug = getParameter(params, "isDebug");
        String surl = getParameter(params, "surl");
        String furl = getParameter(params, "furl");
        String salt = getParameter(params, "salt");
        String userCredentials = getParameter(params, "userCredentials");
        PayUPaymentParams payUPaymentParams = new PayUPaymentParams.Builder()
                .setAmount(amount)
                .setIsProduction(!Boolean.parseBoolean(isDebug))
                .setProductInfo(product)
                .setKey(merchantKey)
                .setPhone(info.getContact())
                .setTransactionId(orderId)
                .setFirstName(name)
                .setEmail(info.getEmail())
                .setSurl(surl)
                .setFurl(furl)
                .setUserCredential(userCredentials)
                .build();
        PayUCheckoutPro.open(activity, payUPaymentParams, new PayUCheckoutProListener() {

            @Override
            public void onPaymentSuccess(@NonNull Object response) {
                Bundle bundle = new Bundle();
                if (response instanceof HashMap) {
                    HashMap<String, Object> result = (HashMap<String, Object>) response;
                    String payuResponse = (String) result.get(PayUCheckoutProConstants.CP_PAYU_RESPONSE);
                    String merchantResponse = (String) result.get(PayUCheckoutProConstants.CP_MERCHANT_RESPONSE);
                    bundle.putString("payuResponse", payuResponse);
                    bundle.putString("merchantResponse", merchantResponse);
                }
                mCallback.onPaymentResult("1", bundle);
            }

            @Override
            public void onPaymentFailure(@NonNull Object response) {
                Bundle bundle = new Bundle();
                if (response instanceof HashMap) {
                    HashMap<String, Object> result = (HashMap<String, Object>) response;
                    String payuResponse = (String) result.get(PayUCheckoutProConstants.CP_PAYU_RESPONSE);
                    String merchantResponse = (String) result.get(PayUCheckoutProConstants.CP_MERCHANT_RESPONSE);
                    bundle.putString("payuResponse", payuResponse);
                    bundle.putString("merchantResponse", merchantResponse);
                }
                mCallback.onPaymentResult("0", bundle);
            }

            @Override
            public void onPaymentCancel(boolean isTxnInitiated) {
                Bundle bundle = new Bundle();
                bundle.putBoolean("isTxnInitiated", isTxnInitiated);
                mCallback.onPaymentResult("-1", bundle);
            }

            @Override
            public void onError(@NonNull ErrorResponse errorResponse) {
                Bundle bundle = new Bundle();
                bundle.putInt("errorCode", errorResponse.getErrorCode());
                bundle.putString("errorMessage", errorResponse.getErrorMessage());
                mCallback.onPaymentResult("-2", bundle);
            }

            @Override
            public void setWebViewProperties(@Nullable WebView webView, @Nullable Object o) {
                // For setting webview properties, if any. Check Customized Integration section for more details on this
            }

            @Override
            public void generateHash(@NonNull HashMap<String, String> valueMap, @NonNull PayUHashGenerationListener hashGenerationListener) {
                String hashName = valueMap.get(PayUCheckoutProConstants.CP_HASH_NAME);
                String hashData = valueMap.get(PayUCheckoutProConstants.CP_HASH_STRING);
                if (!TextUtils.isEmpty(hashName) && !TextUtils.isEmpty(hashData)) {
                    HashMap<String, String> dataMap = new HashMap<>();
                    dataMap.put(hashName, calculateHash(hashData, salt));
                    hashGenerationListener.onHashGenerated(dataMap);
                }
            }

        });
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return false;
        }
        if ("1".equals(status)) {
            return true;
        }
        if ("-1".equals(status)) {
            showToast(activity, activity.getString(R.string.payment_status_cancelled));
        } else if ("-2".equals(status)) {
            int errorCode = bundle.getInt("errorCode");
            String errorMessage = bundle.getString("errorMessage");
            showToast(activity, activity.getString(R.string.payment_status_failed) + "[" + errorCode + "]" + errorMessage);
        } else {
            showToast(activity, activity.getString(R.string.payment_status_failed));
        }
        return false;
    }

    @Override
    public void release() {

    }

    private String calculateHash(String paymentParams, String salt) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA-512");
            messageDigest.update((paymentParams + salt).getBytes());
            byte[] digest = messageDigest.digest();
            return toHexString(digest);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return null;
    }

    private String toHexString(byte[] bytes) {
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String hex = Integer.toHexString(bytes[i] & 0xFF);
            if (hex.length() == 1) {
                hex = '0' + hex;
            }
            builder.append(hex);
        }
        return builder.toString();
    }

    private void showToast(Activity activity, String text) {
        Toast.makeText(activity, text, Toast.LENGTH_SHORT).show();
    }

}
