package com.aitime.android.payment;

import android.app.Activity;
import android.app.Application;
import android.os.Bundle;
import android.text.TextUtils;
import android.widget.Toast;

import com.aitime.android.payment.braintree.R;
import com.aitime.android.payment.core.IPayment;
import com.aitime.android.payment.core.IPaymentCallback;
import com.aitime.android.payment.core.PayChannel;
import com.aitime.android.payment.core.PayCode;
import com.aitime.android.payment.entity.PaymentInfo;
import com.braintreepayments.api.dropin.DropInRequest;
import com.braintreepayments.api.models.GooglePaymentRequest;
import com.braintreepayments.api.models.PayPalRequest;
import com.google.android.gms.wallet.TransactionInfo;
import com.google.android.gms.wallet.WalletConstants;

import java.lang.ref.WeakReference;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Braintree payment
 */
@Payment(channel = PayChannel.BRAINTREE)
public class Braintree implements IPayment {

    private final WeakReference<Activity> mActivity;

    public Braintree(Activity activity) {
        this.mActivity = new WeakReference<>(activity);
    }

    @Override
    public void callback(@NonNull IPaymentCallback callback) {

    }

    @Override
    public void preload(@NonNull Application application) {

    }

    /**
     * Initiate Payment
     *
     * @param info   payment information
     * @param params payment params(contains {@code token})
     */
    @Override
    public void payment(@NonNull PaymentInfo info, @NonNull Map<String, String> params) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return;
        }
        String token = getParameter(params, "token");
        DropInRequest dropInRequest = new DropInRequest()
                .clientToken(token)
                .collectDeviceData(true)
                .requestThreeDSecureVerification(true)
                .paypalRequest(getPayPalRequest(info))
                .googlePaymentRequest(getGooglePaymentRequest(info))
                .maskCardNumber(true)
                .maskSecurityCode(true);
        activity.startActivityForResult(dropInRequest.getIntent(activity), PayCode.BRAINTREE);
    }

    private PayPalRequest getPayPalRequest(@NonNull PaymentInfo info) {
        return new PayPalRequest(info.getPayMoney())
                .currencyCode(info.getCurrency())
                .displayName(info.getPayName());
    }

    private GooglePaymentRequest getGooglePaymentRequest(@NonNull PaymentInfo info) {
        return new GooglePaymentRequest()
                .emailRequired(true)
                .transactionInfo(TransactionInfo.newBuilder()
                        .setTotalPrice(info.getPayMoney())
                        .setCurrencyCode(info.getCurrency())
                        .setTotalPriceStatus(WalletConstants.TOTAL_PRICE_STATUS_FINAL)
                        .build()
                );
    }

    @Override
    public boolean isAvailable(@Nullable String status, @NonNull Bundle bundle) {
        Activity activity = mActivity.get();
        if (activity == null || activity.isDestroyed() || activity.isFinishing()) {
            return false;
        }
        if (String.valueOf(Activity.RESULT_CANCELED).equals(status)) {
            String message = bundle.getString("message");
            if (TextUtils.isEmpty(message)) {
                message = activity.getString(R.string.payment_status_failed);
            }
            showToast(activity, message);
            return false;
        }
        return true;
    }

    @Override
    public void release() {
        mActivity.clear();
    }

    private void showToast(Activity activity, String message) {
        Toast.makeText(activity, message, Toast.LENGTH_SHORT).show();
    }

}
