/*
 * Copyright (C) 2017 Baidu, Inc. All Rights Reserved.
 */
package com.aitime.android.ocr.camera;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;

import java.io.File;

import androidx.annotation.RequiresApi;
import androidx.core.content.ContextCompat;

public class MaskView extends View {

    public MaskView(Context context) {
        this(context, null);
    }

    public MaskView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public MaskView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        // 硬件加速不支持，图层混合。
        setLayerType(View.LAYER_TYPE_SOFTWARE, null);

        pen.setColor(Color.WHITE);
        pen.setStyle(Paint.Style.STROKE);
        pen.setStrokeWidth(6);

        eraser.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.CLEAR));
    }

    public void setMaskColor(int maskColor) {
        this.maskColor = maskColor;
    }

    private int ocrType = OCRType.NONE;
    private int maskColor = Color.argb(100, 0, 0, 0);

    private Rect frame = new Rect();
    private Rect framePassport = new Rect();
    private Paint eraser = new Paint(Paint.ANTI_ALIAS_FLAG);
    private Paint pen = new Paint(Paint.ANTI_ALIAS_FLAG);


    private Drawable locatorDrawable;

    public Rect getFrameRect() {
        if (ocrType == OCRType.NONE) {
            return new Rect(0, 0, getWidth(), getHeight());
        } else if (ocrType == OCRType.PASSPORT_FRONT || ocrType == OCRType.PASSPORT_BACK) {
            return new Rect(framePassport);
        } else {
            return new Rect(frame);
        }

    }

    public Rect getFrameRectExtend() {
        Rect rc = new Rect(frame);
        int widthExtend = (int) ((frame.right - frame.left) * 0.02f);
        int heightExtend = (int) ((frame.bottom - frame.top) * 0.02f);
        rc.left -= widthExtend;
        rc.right += widthExtend;
        rc.top -= heightExtend;
        rc.bottom += heightExtend;
        return rc;
    }

    public void setOCRType(@OCRType int ocrType) {
        this.ocrType = ocrType;
        if (ocrType == OCRType.ID_CARD_FRONT) {
            locatorDrawable = ContextCompat.getDrawable(getContext(), R.drawable.ic_ocr_id_card_locator_front);
        } else if (ocrType == OCRType.ID_CARD_BACK) {
            locatorDrawable = ContextCompat.getDrawable(getContext(), R.drawable.ic_ocr_id_card_locator_back);
        }
        invalidate();
    }

    public int getOcrType() {
        return ocrType;
    }

    public void setOrientation(@CameraView.Orientation int orientation) {
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        if (w > 0 && h > 0) {
            if (ocrType != OCRType.PASSPORT_FRONT && ocrType != OCRType.PASSPORT_BACK) {
                float ratio = h > w ? 0.9f : 0.72f;

                int width = (int) (w * ratio);
                int height = width * 400 / 620;

                int left = (w - width) / 2;
                int top = (h - height) / 2;
                int right = width + left;
                int bottom = height + top;

                frame.left = left;
                frame.top = top;
                frame.right = right;
                frame.bottom = bottom;
            } else {
                float ratio = 0.9f;

                int width = (int) (w * ratio);
                int height = width * 330 / 470;

                int left = (w - width) / 2;
                int top = (h - height) / 2;
                int right = width + left;
                int bottom = height + top;

                framePassport.left = left;
                framePassport.top = top;
                framePassport.right = right;
                framePassport.bottom = bottom;
            }
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        Rect frame = this.frame;
        if (ocrType == OCRType.PASSPORT_FRONT || ocrType == OCRType.PASSPORT_BACK) {
            frame = framePassport;
        }

        int width = frame.width();
        int height = frame.height();

        int left = frame.left;
        int top = frame.top;
        int right = frame.right;
        int bottom = frame.bottom;

        canvas.drawColor(maskColor);
        fillRectRound(left, top, right, bottom, 30, 30, false);
        canvas.drawPath(path, pen);
        canvas.drawPath(path, eraser);

        if (ocrType == OCRType.ID_CARD_FRONT) {
            locatorDrawable.setBounds(
                    (int) (left + 601f / 1006 * width),
                    (int) (top + (110f / 632) * height),
                    (int) (left + (963f / 1006) * width),
                    (int) (top + (476f / 632) * height));
        } else if (ocrType == OCRType.ID_CARD_BACK) {
            locatorDrawable.setBounds(
                    (int) (left + 51f / 1006 * width),
                    (int) (top + (48f / 632) * height),
                    (int) (left + (250f / 1006) * width),
                    (int) (top + (262f / 632) * height));
        }
        if (locatorDrawable != null) {
            locatorDrawable.draw(canvas);
        }
    }

    private Path path = new Path();

    private Path fillRectRound(float left, float top, float right, float bottom, float rx, float ry, boolean conformToOriginalPost) {
        path.reset();
        if (rx < 0) {
            rx = 0;
        }
        if (ry < 0) {
            ry = 0;
        }
        float width = right - left;
        float height = bottom - top;
        if (rx > width / 2) {
            rx = width / 2;
        }
        if (ry > height / 2) {
            ry = height / 2;
        }
        float widthMinusCorners = (width - (2 * rx));
        float heightMinusCorners = (height - (2 * ry));

        path.moveTo(right, top + ry);
        path.rQuadTo(0, -ry, -rx, -ry);
        path.rLineTo(-widthMinusCorners, 0);
        path.rQuadTo(-rx, 0, -rx, ry);
        path.rLineTo(0, heightMinusCorners);

        if (conformToOriginalPost) {
            path.rLineTo(0, ry);
            path.rLineTo(width, 0);
            path.rLineTo(0, -ry);
        } else {
            path.rQuadTo(0, ry, rx, ry);
            path.rLineTo(widthMinusCorners, 0);
            path.rQuadTo(rx, 0, rx, -ry);
        }

        path.rLineTo(0, -heightMinusCorners);
        path.close();
        return path;
    }

    private void capture(File file) {

    }

}
