package com.adobe.pdfservices.operation.pdfops.options.extractpdf;

import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * Parameters for extract PDF operation using {@link com.adobe.pdfservices.operation.pdfops.ExtractPDFOperation}.
 */
public class ExtractPDFOptions {

    private TableStructureType tableStructureType;

    private boolean addCharInfo;

    private boolean getStylingInfo;

    private List<ExtractRenditionsElementType> elementsToExtractRenditions;

    private List<ExtractElementType> elementsToExtract;


    private ExtractPDFOptions(Builder builder) {
        this.elementsToExtractRenditions = builder.elementsToExtractRenditions;
        this.elementsToExtract = builder.elementsToExtract;
        this.addCharInfo = builder.addCharInfo;
        this.tableStructureType = builder.tableStructureType;
        this.getStylingInfo = builder.getStylingInfo;
    }

    /**
     * Returns the table structure type of the resulting rendition
     *
     * @return the table structure type
     */
    public TableStructureType getTableStructureType(){
        return tableStructureType;
    }

    /**
     * Returns whether character level information was invoked for operation.
     *
     * @return the char info boolean
     */
    public Boolean getAddCharInfo(){ return addCharInfo; }

    /**
     * Returns whether styling information was invoked for operation.
     *
     * @return the include styling boolean
     */
    public Boolean getStylingInfo(){ return getStylingInfo; }

    /**
     * Returns the list of renditions (Tables and/or Figures) invoked for operation
     *
     * @return the list of renditions
     */
    public List<ExtractRenditionsElementType> getElementsToExtractRenditions(){
        return elementsToExtractRenditions;
    }

    /**
     * Returns the list of elements (Text and/or Tables) invoked for operation
     *
     * @return the list of elements to extract
     */
    public List<ExtractElementType> getElementsToExtract(){
        return elementsToExtract;
    }

    /**
     * Creates a new {@code ExtractPDFOptions} builder.
     *
     * @return a {@code ExtractPDFOptions.Builder} instance
     */
    public static ExtractPDFOptions.Builder extractPdfOptionsBuilder() {
        return new ExtractPDFOptions.Builder();
    }

    /**
     * Static Builder class for Extract PDF Options
     */
    public static class Builder {

        private TableStructureType tableStructureType;

        private boolean addCharInfo;

        private boolean getStylingInfo;

        private List<ExtractRenditionsElementType> elementsToExtractRenditions = new ArrayList<ExtractRenditionsElementType>();

        private List<ExtractElementType> elementsToExtract = new ArrayList<ExtractElementType>();


        /**
         * Constructs a {@code Builder} instance.
         */
        private Builder() {}

        /**
         * Adds an element to extract (Text or Tables)
         * @param elementToExtract element parameter for extract operation
         * @return this Builder instance to add any additional parameters
         */
        public Builder addElementToExtract(ExtractElementType elementToExtract) {
            if (elementToExtract != null) {
                this.elementsToExtract.add(elementToExtract);
            }
            return this;
        }

        /**
         * Adds elements to extract (Text and/or Tables) to the list of elements to be extracted.
         * @param elementsToExtract elements parameters for extract operation
         * @return this Builder instance to add any additional parameters
         */
        public Builder addElementsToExtract(List<ExtractElementType> elementsToExtract) {
            if (CollectionUtils.isNotEmpty(elementsToExtract)) {
                this.elementsToExtract.addAll(elementsToExtract);
            }
            return this;
        }

        /**
         * Adds a rendition to extract (Figures or Tables)
         * @param elementToExtractRenditions the rendition parameter for extract operation
         * @return this Builder instance to add any additional parameters
         */
        public Builder addElementToExtractRenditions(ExtractRenditionsElementType elementToExtractRenditions) {
            if (elementToExtractRenditions != null) {
                this.elementsToExtractRenditions.add(elementToExtractRenditions);
            }
            return this;
        }

        /**
         * Adds elements to extract (Figures and/or Tables) to the list of elements to be extracted.
         * @param elementsToExtractRenditions the renditions parameters for extract operation
         * @return this Builder instance to add any additional parameters
         */
        public Builder addElementsToExtractRenditions(List<ExtractRenditionsElementType> elementsToExtractRenditions) {
            if (CollectionUtils.isNotEmpty(elementsToExtractRenditions)) {
                this.elementsToExtractRenditions.addAll(elementsToExtractRenditions);
            }
            return this;
        }

        /**
         * Adds the Table Structure format (Csv or Xlsx) for table renditions
         * @param tableStructureType parameter for output type of table structure
         * @return this Builder instance to add any additional parameters
         */
        public Builder addTableStructureFormat(TableStructureType tableStructureType) {
            this.tableStructureType = tableStructureType;
            return this;
        }

        /**
         * Boolean specifying whether to add character level bounding boxes to output json
         * @param addCharInfo Boolean
         * @return this Builder instance to add any additional parameters
         */
        public Builder addCharInfo(Boolean addCharInfo) {
            this.addCharInfo = addCharInfo;
            return this;
        }

        /**
         * Boolean specifying whether to add styling information to output json
         * @param getStylingInfo Boolean
         * @return this Builder instance to add any additional parameters
         */
        public Builder addGetStylingInfo(Boolean getStylingInfo) {
            this.getStylingInfo = getStylingInfo;
            return this;
        }

        public ExtractPDFOptions build() { return new ExtractPDFOptions( this);}
    }
}
