/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops;

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.adobe.pdfservices.operation.exception.ServiceApiException;
import com.adobe.pdfservices.operation.exception.ServiceUsageException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.MediaType;
import com.adobe.pdfservices.operation.internal.api.FileDownloadApi;
import com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.CPFContentAnalyzerResponse;
import com.adobe.pdfservices.operation.internal.exception.OperationException;
import com.adobe.pdfservices.operation.internal.service.CompressPDFService;
import com.adobe.pdfservices.operation.internal.util.FileUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.adobe.pdfservices.operation.internal.util.ValidationUtil;
import com.adobe.pdfservices.operation.ExecutionContext;
import com.adobe.pdfservices.operation.Operation;
import com.adobe.pdfservices.operation.io.FileRef;
import com.adobe.pdfservices.operation.pdfops.options.compresspdf.CompressPDFOptions;
import com.adobe.pdfservices.operation.pdfops.options.compresspdf.CompressionLevel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An operation that reduces the size of a PDF file. Allows specifying compressionLevel({@link CompressionLevel})
 * for compressing pdf
 * <p>
 * Sample Usage:
 * <pre>{@code   CompressPDFOperation compressPDFOperation = CompressPDFOperation.createNew();
 *   compressPDFOperation.setInput(FileRef.createFromLocalFile("~/Documents/compressOperationInput.pdf",
 *                                                              CompressPDFOperation.SupportedSourceFormat.PDF.getMediaType()));
 *   Credentials credentials = Credentials.serviceAccountCredentialsBuilder().fromFile("pdfservices-api-credentials.json").build();
 *   FileRef result = compressPDFOperation.execute(ExecutionContext.create(credentials));
 *   result.saveAs("output/compressOperationOutput.pdf");
 * }</pre>
 */
public class CompressPDFOperation implements Operation {

    private static final Logger LOGGER = LoggerFactory.getLogger(CompressPDFOperation.class);

    /**
     * Supported media types for this operation
     */
    private static final Set<String> SUPPORTED_SOURCE_MEDIA_TYPES =
            new HashSet<>(Collections.singletonList(ExtensionMediaTypeMapping.PDF.getMediaType()));

    /**
     * Field representing the extension of the operation result
     */
    private static final String TARGET_FILE_EXTENSION = ExtensionMediaTypeMapping.PDF.getExtension();

    /**
     * Variable to check if the operation instance was invoked more than once
     */
    private boolean isInvoked = false;

    private FileRefImpl sourceFileRef;
    private CompressPDFOptions compressPDFOptions;

    private CompressPDFOperation() { }

    /**
     * Constructs a {@code CompressPDFOperation} instance.
     *
     * @return a new {@code CompressPDFOperation} instance
     */
    public static CompressPDFOperation createNew() {
        return new CompressPDFOperation();
    }

    /**
     * Sets an input file.
     *
     * @param sourceFileRef an input file
     */
    public void setInput(FileRef sourceFileRef) {
        Objects.requireNonNull(sourceFileRef, "No input was set for operation");
        this.sourceFileRef = (FileRefImpl) sourceFileRef;
    }

    /**
     * Sets the options for this operation. See {@link CompressPDFOptions} for how to specify the
     * options for the different compression levels.
     *
     * @param compressPDFOptions optional parameters.
     */
    public void setOptions(CompressPDFOptions compressPDFOptions) {
        this.compressPDFOptions = compressPDFOptions;
    }

    /**
     * Executes this operation synchronously using the supplied context and returns a new FileRef instance for the resulting PDF file.
     * <p>
     * The resulting file may be stored in the system temporary directory (per java.io.tmpdir System property).
     * See {@link FileRef} for how temporary resources are cleaned up.
     *
     * @param context the context in which to execute the operation
     * @return the resulting PDF file
     * @throws ServiceApiException   if an API call results in an error response
     * @throws IOException           if there is an error in reading either the input source or the resulting PDF file
     * @throws ServiceUsageException if service usage limits have been reached or credentials quota has been exhausted
     */
    public FileRef execute(ExecutionContext context) throws ServiceApiException, IOException, ServiceUsageException {
        validateInvocationCount();
        InternalExecutionContext internalExecutionContext = (InternalExecutionContext) context;
        this.validate(internalExecutionContext);

        try {
            LOGGER.info("All validations successfully done. Beginning Compress PDF operation execution");
            long startTimeMs = System.currentTimeMillis();

            String location = CompressPDFService.compressPDF(internalExecutionContext, sourceFileRef, compressPDFOptions, this.getClass().getSimpleName());

            String targetFileName = FileUtil.getRandomFileName(TARGET_FILE_EXTENSION);
            String temporaryDestinationPath = PathUtil.getTemporaryDestinationPath(targetFileName, TARGET_FILE_EXTENSION);

            FileDownloadApi.downloadAndSave(internalExecutionContext, location, temporaryDestinationPath, CPFContentAnalyzerResponse.class);
            LOGGER.info("Operation successfully completed. Stored requisite PDF at {}", temporaryDestinationPath);
            LOGGER.debug("Operation Success Info - Request ID: {}, Latency(ms): {}",
                    StringUtil.getRequestIdFromLocation(location), System.currentTimeMillis() - startTimeMs);

            isInvoked = true;
            return FileRef.createFromLocalFile(temporaryDestinationPath);
        } catch (OperationException oe) {
            throw new ServiceApiException(oe.getErrorMessage(), oe.getRequestTrackingId(), oe.getStatusCode(), oe.getReportErrorCode());
        }
    }

    private void validateInvocationCount() {
        if (isInvoked) {
            LOGGER.error("Operation instance must only be invoked once");
            throw new IllegalStateException("Operation instance must not be reused, can only be invoked once");
        }
    }

    private void validate(InternalExecutionContext context) {
        if (sourceFileRef == null) {
            throw new IllegalArgumentException("No input was set for operation");
        }
        if (sourceFileRef.getSourceURL() != null) {
            throw new IllegalArgumentException("Input for the Compress PDF Operation can not be sourced from a URL");
        }
        ValidationUtil.validateExecutionContext(context);
        ValidationUtil.validateMediaType(SUPPORTED_SOURCE_MEDIA_TYPES, this.sourceFileRef.getMediaType());
    }

    /**
     * Supported source file formats for {@link CompressPDFOperation}.
     */
    public enum SupportedSourceFormat implements MediaType {
        /**
         * Represents "application/pdf" media type
         */
        PDF;

        /**
         * Returns the corresponding media type for this format, intended to be used for {@code mediaType} parameter in
         * {@link FileRef} methods.
         *
         * @return the corresponding media type
         */
        public String getMediaType() {
            return ExtensionMediaTypeMapping.valueOf(name()).getMediaType().toLowerCase();
        }
    }
}
