/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.service;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.options.CombineOperationInput;
import com.adobe.pdfservices.operation.pdfops.options.PageRanges;

public class InsertPagesService extends CombinePDFService{

    private static final int INPUT_SOURCE_LIMIT = 20;

    private static InsertFilesInput getFilesToInsert(FileRefImpl baseFileRef,
                                                     Map<Integer, List<CombineOperationInput>> filesToInsert) throws IOException {
        List<FileRefImpl> fileRefList = new ArrayList<>();
        List<PageRanges> pageRangeList = new ArrayList<>();
        int baseFileStartIndex = 1;
        for (Map.Entry<Integer, List<CombineOperationInput>> entry : filesToInsert.entrySet()) {
            if (entry.getKey() != 1) {
                fileRefList.add(baseFileRef.getCopy());
                PageRanges pageRanges = new PageRanges();
                pageRanges.addRange(baseFileStartIndex,entry.getKey()-1);
                pageRangeList.add(pageRanges);
                baseFileStartIndex = entry.getKey();
            }

            for (CombineOperationInput combineOperationInput : entry.getValue()) {
                fileRefList.add(combineOperationInput.getSourceFileRef());
                pageRangeList.add(combineOperationInput.getPageRanges());
            }
        }
        fileRefList.add(baseFileRef);
        PageRanges basePageRanges = new PageRanges();
        basePageRanges.addAllFrom(baseFileStartIndex);
        pageRangeList.add(basePageRanges);
        return new InsertFilesInput(fileRefList, pageRangeList);
    }

    public static String insertPages(InternalExecutionContext context,
                                     FileRefImpl baseFileRef,
                                     Map<Integer, List<CombineOperationInput>> filesToInsert,
                                     String operation) throws IOException {
        try {

            InsertFilesInput insertFilesInput = getFilesToInsert(baseFileRef, filesToInsert);

            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = insertFilesInput.fileRefList;

            // Validate insert operation input source limit
            if (sourceFileRefList.size() > INPUT_SOURCE_LIMIT) {
                throw new IllegalArgumentException("Too many insertions specified for the operation");
            }

            return execute(context,operation,insertFilesInput.pageRangesList,sourceFileRefList,sourceFileRefList.size());
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }

    private static class InsertFilesInput {

        private List<FileRefImpl> fileRefList;
        private List<PageRanges> pageRangesList;

        InsertFilesInput(List<FileRefImpl> fileRefList, List<PageRanges> pageRangesList) {
            this.fileRefList = fileRefList;
            this.pageRangesList = pageRangesList;
        }
    }
}
