/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.service;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import com.adobe.pdfservices.operation.internal.cpf.constants.CPFConstants;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.CPFContentAnalyzerRequests;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.CombinePageRange;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.Inputs;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.MultiPartFormField;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.api.CPFApi;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.multipleInputsField.CPFMultipleInputsField;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.multipleInputsField.CombineCPFMultipleInputsField;
import com.adobe.pdfservices.operation.internal.http.DefaultRequestHeaders;
import com.adobe.pdfservices.operation.internal.http.HttpResponse;
import com.adobe.pdfservices.operation.internal.options.CombineOperationInput;
import com.adobe.pdfservices.operation.internal.util.InputStreamUtil;
import com.adobe.pdfservices.operation.pdfops.options.PageRanges;

public class CombinePDFService {

    private static final String IO_FORMAT_SPECIFIER = "application/pdf";

    private static List<PageRanges> generatePageRangesList(List<CombineOperationInput> filesToCombine) {
        List<PageRanges> pageRangesList = new ArrayList<>();
        for (CombineOperationInput fileToCombine: filesToCombine) {
            PageRanges pageRanges = fileToCombine.getPageRanges();
            if(pageRanges!=null){
                if(pageRanges.isEmpty()){
                    pageRanges.addAll();
                }
                pageRangesList.add(pageRanges);
            }
        }
        return  pageRangesList;
    }

    private static List<CPFMultipleInputsField> getCPFMultipleInputsFieldList(List<PageRanges> pageRangesList, int numOfInputs)
    {
        List<CPFMultipleInputsField> cpfMultipleInputsFieldList = new ArrayList<>();
        for (int partNumber = 1; partNumber <= numOfInputs; partNumber++)
        {
            cpfMultipleInputsFieldList.add(new CombineCPFMultipleInputsField(new MultiPartFormField(IO_FORMAT_SPECIFIER, InputStreamUtil.generateFileFormFieldName(partNumber)),
                    new CombinePageRange(pageRangesList.get(partNumber-1).getRanges())));
        }
        return cpfMultipleInputsFieldList;
    }

    public static String combinePdf(InternalExecutionContext context,
                                    List<CombineOperationInput> filesToCombine,
                                    String operation) throws FileNotFoundException{

        try {
            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = new ArrayList<>();
            for(CombineOperationInput fileToCombine : filesToCombine) {
                sourceFileRefList.add(fileToCombine.getSourceFileRef());
            }

            // Create PageRange Params
            List<PageRanges> pageRangesList = generatePageRangesList(filesToCombine);

            return execute(context,operation,pageRangesList,sourceFileRefList,filesToCombine.size());
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }

    protected static String execute(InternalExecutionContext context, String operation, List<PageRanges> pageRangesList, List<FileRefImpl> sourceFileRefList, int numOfInputs) throws FileNotFoundException {
        // Create CPFMultipleInputsFieldList
        List<CPFMultipleInputsField> cpfMultipleInputsFieldList = getCPFMultipleInputsFieldList(pageRangesList, numOfInputs);

        // Build cpf inputs
        Inputs inputs = new Inputs(cpfMultipleInputsFieldList);

        // Create CPFContentAnalyzerRequests
        CPFContentAnalyzerRequests cpfContentAnalyzerRequests
                = new CPFContentAnalyzerRequests(CPFConstants.CombinePDF.PAPI_ENGINE_REPO_ASSET_ID, inputs, IO_FORMAT_SPECIFIER);

        HttpResponse<String> response = CPFApi.cpfCreateOpsApi(context, cpfContentAnalyzerRequests, sourceFileRefList, String.class, operation);
        return response.getHeaders().get(DefaultRequestHeaders.LOCATION_HEADER_NAME);
    }



}
