/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.options;

import com.fasterxml.jackson.annotation.JsonInclude;

/**
 * Range of pages of a file where page numbering starts from 1
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class PageRange {

    private Integer start;
    private Integer end;

    /**
     * Creates a PageRange with start and end page num as its limit with both page num included.
     * Page Numbering starts from 1
     *
     * @param start Start page num of this page range
     * @param end   End page num of this page range
     */
    public PageRange(Integer start, Integer end) {
        this.start = start;
        this.end = end;
    }

    /**
     * Returns the start Page Num of Page Range
     */
    public Integer getStart() {
        return start;
    }

    /**
     * Returns the end Page Num of Page Range from source PDF file.
     */
    public Integer getEnd() {
        return end;
    }

    public void validate() {
        if ((end != null && start > end) || start <= 0) {
            throw new IllegalArgumentException("Invalid page range specified");
        }

    }

    /**
     * Returns the representation of file Page Ranges, as required by its corresponding API spec
     */
    @Override
    public String toString() {
        if (end != null && start == end) {
            return String.valueOf(start);
        } else {
            if (end == null) {
                return start + "-";
            } else {
                return start + "-" + end;
            }
        }

    }

}
