/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.http;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.adobe.pdfservices.operation.internal.cpf.constants.RequestKey;
import com.adobe.pdfservices.operation.internal.auth.AuthenticationMethod;
import com.adobe.pdfservices.operation.internal.auth.Authenticator;
import com.adobe.pdfservices.operation.internal.auth.SessionToken;

import com.github.hal4j.uritemplate.URITemplate;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.entity.StringEntity;


/**
 * Returns a new instance of the multipart request on every setter
 */
public class BaseMultipartRequest implements MultiPartRequest {


    private URITemplate uriTemplate;
    private HttpMethod httpMethod;
    private Map<String, String> headers;
    private AuthenticationMethod authenticationMethod;
    private RequestType requestType;
    private List<ByteArrayPart> byteArrayParts;
    private StringBodyPart stringBodyPart;
    private List<InputStreamPart> inputStreamParts;
    private Authenticator authenticator;
    private HttpRequestConfig requestConfig;
    private RequestKey requestKey;

    public BaseMultipartRequest(String uriTemplate) {
        this.headers = new HashMap<>();
        this.uriTemplate = new URITemplate(uriTemplate);
        this.authenticationMethod = AuthenticationMethod.AUTH_HEADER_PRIMARY;
        this.requestType = RequestType.MULTIPART;
        this.httpMethod = HttpMethod.POST;

    }

    @Override
    public MultiPartRequest withContentType(String contentType) {
        return withHeader(HttpHeaders.CONTENT_TYPE, contentType);
    }

    @Override
    public MultiPartRequest withHeader(String headerName, String headerValue) {
        headers.put(headerName, headerValue);
        return this;
    }

    @Override
    public MultiPartRequest withHeaders(Map<String, String> headers) {
        Map<String, String> headersCopy = getHeadersCopy(headers);
        this.headers.putAll(headersCopy);
        return this;
    }

    @Override
    public HttpRequest withUriParam(String name, String value) {
        return null;
    }

    @Override
    public MultiPartRequest withUrlEncodedFormParams(List<NameValuePair> formParams){
        // Not applicable for multipart request
        throw new UnsupportedOperationException("Operation not supported");
    }

    @Override
    public MultiPartRequest withBody(String body) {
        // Not applicable for multipart request
        throw new UnsupportedOperationException("Operation not supported");

    }

    @Override
    public MultiPartRequest withAuthenticationMethod(AuthenticationMethod authMethod) {
        this.authenticationMethod = authMethod;
        return this;
    }

    @Override
    public MultiPartRequest withTemplate(String uriTemplate) {
        this.uriTemplate = new URITemplate(uriTemplate);
        return this;
    }

    @Override
    public MultiPartRequest withAcceptType(String acceptHeader) {
        return withHeader(HttpHeaders.ACCEPT, acceptHeader);
    }

    @Override
    public MultiPartRequest withAuthenticator(Authenticator authenticator) {
        this.authenticator = authenticator;
        this.headers.put(DefaultRequestHeaders.X_API_KEY_HEADER_NAME, this.authenticator.getClientId());
        return this;
    }

    @Override
    public MultiPartRequest withConfig(HttpRequestConfig requestConfig) {
        this.requestConfig = requestConfig;
        return this;
    }

    @Override
    public HttpRequest withRequestKey(RequestKey requestKey) {
        this.requestKey = requestKey;
        return this;
    }

    /**
     * Only supports a single body part for now, based on the backend API. Internal implementation can be changed
     *
     * @param stringBodyPart
     * @return
     */
    @Override
    public MultiPartRequest withStringBodyPart(StringBodyPart stringBodyPart) {
        this.stringBodyPart = stringBodyPart;
        return this;
    }

    @Override
    public MultiPartRequest withInputStreamPart(InputStreamPart inputStreamPart) {
        if (this.inputStreamParts == null) {
            this.inputStreamParts = new ArrayList<InputStreamPart>();
         }
        this.inputStreamParts.add(inputStreamPart);
        return this;
    }

    @Override
    public MultiPartRequest withByteArrayPart(ByteArrayPart byteArrayPart) {
        if(this.byteArrayParts == null) {
            this.byteArrayParts = new ArrayList<>();
        }
        this.byteArrayParts.add(byteArrayPart);
        return this;
    }

    @Override
    public void authenticate() {
        if (authenticationMethod.isAuthRequired()) {
            if (authenticator == null) {
                throw new IllegalStateException("No authenticator provided for request. Cannot add authorization headers");
            }
            SessionToken sessionToken = this.authenticator.getSessionToken(requestConfig);
            withHeader(HttpHeaders.AUTHORIZATION, String.format("Bearer %s", sessionToken.getAccessToken()));
        }
    }

    @Override
    public void forceAuthenticate() {
        if (authenticationMethod.isAuthRequired()) {
            if (authenticator == null) {
                throw new IllegalStateException("No authenticator provided for request. Cannot add authorization headers");
            }
            SessionToken sessionToken = this.authenticator.refreshSessionToken(requestConfig);
            withHeader(HttpHeaders.AUTHORIZATION, String.format("Bearer %s", sessionToken.getAccessToken()));
        }
    }

    @Override
    public HttpRequestConfig getConfig() {
        return requestConfig;
    }

    @Override
    public StringEntity getEntity() {
        //not supported
        throw new UnsupportedOperationException("Operation not supported");
    }

    @Override
    public HttpMethod getHttpMethod() {
        return httpMethod;
    }


    @Override
    public AuthenticationMethod getAuthMethod() {
        return authenticationMethod;
    }

    @Override
    public Map<String, String> getHeaders() {
        return headers;
    }

    @Override
    public String getFinalUri() {
        return this.uriTemplate.expand().toString();
    }

    private Map<String, String> getHeadersCopy(Map<String, String> headersToCopy) {
        return headersToCopy.entrySet()
                .stream()
                .collect(Collectors.toMap(Map.Entry::getKey,
                        Map.Entry::getValue));
    }

    @Override
    public RequestType getRequestType() {
        return requestType;
    }

    @Override
    public Authenticator getAuthenticator() {
        return authenticator;
    }

    @Override
    public String getTemplateString() {
        return uriTemplate.toString();
    }

    @Override
    public List<ByteArrayPart> getByteArrayParts() {
        return byteArrayParts;
    }

    @Override
    public List<InputStreamPart> getInputStreams() {
      return this.inputStreamParts;
    }

    @Override
    public StringBodyPart getStringBodyPart() {
        return stringBodyPart;
    }

    @Override
    public RequestKey getRequestKey() {
        return requestKey;
    }
}
