/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.auth;

import java.security.PrivateKey;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import com.adobe.pdfservices.operation.internal.cpf.constants.RequestKey;
import com.adobe.pdfservices.operation.internal.dto.response.AuthenticationResponseDto;
import com.adobe.pdfservices.operation.internal.http.*;
import com.adobe.pdfservices.operation.internal.util.PrivateKeyParser;
import com.adobe.pdfservices.operation.auth.ServiceAccountCredentials;
import com.adobe.pdfservices.operation.internal.GlobalConfig;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSSigner;
import com.nimbusds.jose.crypto.RSASSASigner;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.pdfservices.operation.exception.SdkException;

/**
 * An authenticator implementation which deals with service account authentication through JWT
 */
public class JwtAuthenticator implements Authenticator {

    private static final Logger LOGGER = LoggerFactory.getLogger(JwtAuthenticator.class);
    private static final String CLIENT_ID = "client_id";
    private static final String CLIENT_SECRET = "client_secret";
    private static final String JWT_TOKEN = "jwt_token";
    private static final int expiryMs = 1000 * 60 * 60 * 24;
    private static String JWT_ENDPOINT;
    private static String JWT_AUDIENCE_BASE_URI;
    private ServiceAccountCredentials serviceAccountConfiguration;
    private SessionToken sessionToken;
    private PrivateKey privateKey;


    public JwtAuthenticator(ServiceAccountCredentials serviceAccountConfiguration) {
        this.serviceAccountConfiguration = serviceAccountConfiguration;
        this.privateKey = PrivateKeyParser.parsePrivateKey(serviceAccountConfiguration.getPrivateKey());
        JWT_ENDPOINT = String.format("%s/%s", ((ServiceAccountCredentialsWithUri)serviceAccountConfiguration).getImsBaseUri(), GlobalConfig.getJwtUriSuffix());
        JWT_AUDIENCE_BASE_URI = String.format("%s/%s", ((ServiceAccountCredentialsWithUri)serviceAccountConfiguration).getImsBaseUri(),
                GlobalConfig.getJwtAudienceSuffix());

    }

    @Override
    public String getClientId() {
        return serviceAccountConfiguration.getClientId();
    }

    @Override
    public synchronized SessionToken getSessionToken(HttpRequestConfig requestConfig) {
        if (sessionToken != null) {

//       Check expiration token with a 2 minute buffer
            Duration durationTillSessionExpiry = Duration.between(Instant.now(), sessionToken.getExpiresAt());
            if (durationTillSessionExpiry.getSeconds() > 0 && durationTillSessionExpiry.toMinutes() > 2) {
                LOGGER.debug("Session Token is valid, won't be refreshed. ");
                return sessionToken;
            } else {
                LOGGER.debug("Session token expired. Refreshing! ");
            }
        }
        return refreshSessionToken(requestConfig);
    }

    /**
     * Fetches a new Session Token from IMS
     */
    @Override
    public synchronized SessionToken refreshSessionToken(HttpRequestConfig requestConfig) {
        HttpClient client = HttpClientFactory.getDefaultHttpClient();

        // The auth request must never require any authentication
        HttpRequest httpRequest = new BaseHttpRequest(SESSION_TOKEN_REQUEST_GROUP_KEY, HttpMethod.POST, JWT_ENDPOINT)
                    .withAuthenticationMethod(AuthenticationMethod.UNAUTHENTICATED)
                    .withHeader(DefaultRequestHeaders.DC_REQUEST_ID_HEADER_KEY, UUID.randomUUID().toString())
                    .withHeader(DefaultRequestHeaders.DC_APP_INFO_HEADER_KEY, GlobalConfig.getAppInfo())
                    .withHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_FORM_URLENCODED.toString())
                    .withUrlEncodedFormParams(getFormParams())
                    .withRequestKey(RequestKey.AUTHN)
                    .withConfig(requestConfig);

        HttpResponse<AuthenticationResponseDto> sessionTokenResponse = client.send(httpRequest, AuthenticationResponseDto.class);
        AuthenticationResponseDto authResponseBody = sessionTokenResponse.getBody();


        sessionToken = new SessionToken(authResponseBody.getAccessToken(),
                Instant.now().plusMillis(authResponseBody.getExpiryInterval()));

        return sessionToken;
    }

    private List<NameValuePair> getFormParams(){
        List<NameValuePair> formParams = new ArrayList<>();
        formParams.add(new BasicNameValuePair(CLIENT_ID, serviceAccountConfiguration.getClientId()));
        formParams.add(new BasicNameValuePair(CLIENT_SECRET, serviceAccountConfiguration.getClientSecret()));
        formParams.add(new BasicNameValuePair(JWT_TOKEN, prepareJwt()));
        return formParams;
    }

    //Create SERVICE Account
    private String prepareJwt() {

        String audience = JWT_AUDIENCE_BASE_URI + serviceAccountConfiguration.getClientId();

        // 24 hours
        JWTClaimsSet claimsSet = new JWTClaimsSet.Builder()
                .subject(serviceAccountConfiguration.getAccountId())
                .issuer(serviceAccountConfiguration.getOrganizationId())
                .audience(audience)
                .expirationTime(new Date(System.currentTimeMillis() + expiryMs))
                .claim(((ServiceAccountCredentialsWithUri)serviceAccountConfiguration).getClaim(), true)
                .build();


        SignedJWT signedJWT = new SignedJWT(
                new JWSHeader.Builder(JWSAlgorithm.RS256).build(), claimsSet);
        JWSSigner signer = new RSASSASigner(privateKey);
        try {
            signedJWT.sign(signer);
        } catch (JOSEException e) {
            LOGGER.error("Exception encountered while signing JWT ", e);
            throw new SdkException("Exception while signing JWT", e);
        }

        return signedJWT.serialize();
    }
}
