/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;

import com.adobe.pdfservices.operation.internal.auth.AuthenticationType;
import com.adobe.pdfservices.operation.internal.http.HttpClientType;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GlobalConfig {

    private static final Logger LOGGER = LoggerFactory.getLogger(GlobalConfig.class);

    private static final String SYSTEM_TEMPORARY_DIRECTORY = "java.io.tmpdir";
    private static final String SERVICE_CONFIG_FILE = "dc-services-default-config.properties";
    private static final String HTTP_ERROR_CODE_CONFIG_FILE = "http-error-code.properties";
    private static final HttpClientType DEFAULT_HTTP_CLIENT_TYPE = HttpClientType.APACHE;
    private static final AuthenticationType DEFAULT_AUTHENTICATION_TYPE = AuthenticationType.SERVICE_ACCOUNT;
    private static Properties sdkConfigProperties = new Properties();
    /**
     * Contains http status codes which have custom error messages
     */
    private static Properties httpErrorCodeProperties = new Properties();

    static {

        InputStream resourceAsStream = GlobalConfig.class.getClassLoader().getResourceAsStream(SERVICE_CONFIG_FILE);
        InputStream errorCodeStream = GlobalConfig.class.getClassLoader().getResourceAsStream(HTTP_ERROR_CODE_CONFIG_FILE);

        try {
            sdkConfigProperties.load(resourceAsStream);
            httpErrorCodeProperties.load(errorCodeStream);

        } catch (IOException e) {
            LOGGER.warn("Cannot load config properties! Application will not function properly");
        }
    }

    public static int getConnectTimeout() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.CONNECT_TIMEOUT_KEY));
    }

    public static int getSocketTimeout() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.SOCKET_TIMEOUT_KEY));
    }

    public static int getMaxRetries() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.MAX_RETRIES_KEY));
    }

    public static int getRetryDelayFactor() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.RETRY_DELAY_FACTOR));
    }

    public static int getMaxRetryInterval() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.MAX_RETRY_INTERVAL));
    }

    public static int getRetryBackoffInterval() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.BACKOFF_INTERVAL_KEY));
    }

    public static Set<Integer> getSuccessResponseCodes() {
        String successResponseCodeString = sdkConfigProperties.getProperty(PropertyKeys.SUCCESS_RESPONSE_CODE_KEY);
        return Arrays.stream(successResponseCodeString.split(","))
                .map(Integer::parseInt).collect(Collectors.toSet());
    }

    public static Set<Integer> getRetriableResponseCodes() {
        String retriableResponseCodeString = sdkConfigProperties.getProperty(PropertyKeys.RETRIABLE_RESPONSE_CODE_KEY);
        return Arrays.stream(retriableResponseCodeString.split(","))
                .map(Integer::parseInt).collect(Collectors.toSet());
    }

    public static int getMaxApacheConnections() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.MAX_CONNECTION_KEY));
    }

    public static int getMaxApacheConnectionsPerRoute() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.MAX_CONNECTION_PER_ROUTE_KEY));
    }

    public static HttpClientType getDefaultHttpClientType() {
        return DEFAULT_HTTP_CLIENT_TYPE;
    }

    public static AuthenticationType getDefaultAuthenticationType() {
        return DEFAULT_AUTHENTICATION_TYPE;
    }

    public static String getSystemTemporaryDirectory() {
        return System.getProperty(SYSTEM_TEMPORARY_DIRECTORY);
    }

    public static String getAppInfo() {
        return String.join("/", sdkConfigProperties.getProperty(PropertyKeys.APP_INFO_KEY).split(","));

    }

    public static String getAppInfoPrependedToRequestId() {
        return sdkConfigProperties.getProperty(PropertyKeys.APP_INFO_PREPEND_REQUEST_ID);

    }

    public static String getImsBaseUri() {
        return sdkConfigProperties.getProperty(PropertyKeys.JWT_ENDPOINT_KEY);
    }

    public static String getClaim() {
        return sdkConfigProperties.getProperty(PropertyKeys.JWT_CLAIM);
    }

    public static String getJwtUriSuffix() {
        return sdkConfigProperties.getProperty(PropertyKeys.JWT_ENDPOINT_SUFFIX_KEY);
    }

    public static String getJwtAudienceSuffix() {
        return sdkConfigProperties.getProperty(PropertyKeys.JWT_AUDIENCE_SUFFIX_KEY);
    }

    public static String getTemporaryOperationResultDirectory() {
        return sdkConfigProperties.getProperty(PropertyKeys.TEMPORARY_OPERATION_RESULT_DIRECTORY_KEY);
    }

    public static Set<String> getAllowedUrlProtocols() {
        return new HashSet<>(Collections.singletonList("https"));
    }

    /**
     * Checks if a given http error status code is special i.e. we need some custom error messages corresponding to it
     *
     * @return
     */
    public static boolean isCustomErrorMessageRequired(int httpErrorCode) {
        String errorCode = httpErrorCodeProperties.getProperty(String.valueOf(httpErrorCode));
        return StringUtil.isNotBlank(errorCode);

    }

    public static String getErrorCodeForHttpStatusCode(int httpErrorCode) {
        String property = httpErrorCodeProperties.getProperty(Integer.toString(httpErrorCode));
        return property.split(":")[0];

    }

    public static String getErrorMessageForHttpStatusCode(int httpErrorCode) {
        String property = httpErrorCodeProperties.getProperty(Integer.toString(httpErrorCode));
        return property.split(":")[1];

    }

    public static String getOpsCreateUri() {
        return sdkConfigProperties.getProperty(PropertyKeys.CPF_OPS_CREATE_URI_KEY);
    }

    public static Integer getProxyPort() {
        return Integer.parseInt(sdkConfigProperties.getProperty(PropertyKeys.PROXY_PORT));
    }

    public static String getExtractOpsAnalyzerID() {
        return sdkConfigProperties.getProperty(PropertyKeys.CPF_OPS_EXTRACT_ANALYZER_ID);
    }

    private static class PropertyKeys {

        private static final String CONNECT_TIMEOUT_KEY = "http.connectTimeout";
        private static final String SOCKET_TIMEOUT_KEY = "http.socketTimeout";
        private static final String MAX_RETRIES_KEY = "http.maxRetries";
        private static final String RETRY_DELAY_FACTOR = "http.retryDelayFactor";
        private static final String MAX_RETRY_INTERVAL = "http.maxRetryInterval";
        private static final String BACKOFF_INTERVAL_KEY = "http.retryBackoffInterval";
        private static final String SUCCESS_RESPONSE_CODE_KEY = "http.successResponseCode";
        private static final String RETRIABLE_RESPONSE_CODE_KEY = "http.retriableResponseCode";

        private static final String PROXY_HOST = "http.proxyHost";
        private static final String PROXY_PORT = "http.proxyPort";
        private static final String PROXY_PROTOCOL = "http.proxyProtocol";

        private static final String MAX_CONNECTION_KEY = "apache.client.maxConnection";
        private static final String MAX_CONNECTION_PER_ROUTE_KEY = "apache.client.maxConnectionPerRoute";

        private static final String JWT_ENDPOINT_KEY = "jwt.base.uri";
        private static final String JWT_ENDPOINT_SUFFIX_KEY = "jwt.uri.suffix";
        private static final String JWT_AUDIENCE_SUFFIX_KEY = "jwt.audience.suffix";
        private static final String JWT_CLAIM = "jwt.claim";

        private static final String TEMPORARY_OPERATION_RESULT_DIRECTORY_KEY = "operation.result.tempdirectory";

        private static final String APP_INFO_KEY = "app.info";
        private static final String APP_INFO_PREPEND_REQUEST_ID = "app.info.prepend.request.id";

        private static final String CPF_OPS_CREATE_URI_KEY = "cpf.ops.create.uri";
        private static final String CPF_OPS_EXTRACT_ANALYZER_ID = "cpf.ops.extract.analyzerId";

    }


}
