/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import com.adobe.pdfservices.operation.internal.cpf.constants.RequestKey;
import com.adobe.pdfservices.operation.internal.http.BaseHttpRequest;
import com.adobe.pdfservices.operation.internal.http.BaseMultipartRequest;
import com.adobe.pdfservices.operation.internal.http.DefaultRequestHeaders;
import com.adobe.pdfservices.operation.internal.http.HttpMethod;
import com.adobe.pdfservices.operation.internal.http.HttpRequest;
import com.adobe.pdfservices.operation.internal.http.RequestType;
import org.apache.http.HttpHeaders;

public class CPFServiceRequestContext {

    private static final String ACCEPT_HEADER_VALUE = "application/json, text/plain, */*";
    private static final String RESPOND_WITH_QUERY_PARAM = "respondWith";

    private Map<RequestKey, HttpRequest> baseRequestMap;

    public CPFServiceRequestContext(String cpfServiceOpsCreateUri) {
        baseRequestMap = new HashMap<>();
        baseRequestMap.put(RequestKey.CREATE, new BaseMultipartRequest(cpfServiceOpsCreateUri)
                .withRequestKey(RequestKey.CREATE));
        baseRequestMap.put(RequestKey.STATUS, new BaseHttpRequest(HttpMethod.GET)
                .withRequestKey(RequestKey.STATUS));
    }

    public synchronized HttpRequest getBaseRequest(RequestKey requestKey) {
        HttpRequest baseRequest = baseRequestMap.get(requestKey);
        return copy(baseRequest);
    }

    private HttpRequest copy(HttpRequest httpRequest) {
        RequestType requestType = httpRequest.getRequestType();
        HttpRequest httpRequestCopy = null;
        switch (requestType) {
            case REGULAR:
                httpRequestCopy = new BaseHttpRequest(httpRequest.getHttpMethod())
                        .withRequestKey(httpRequest.getRequestKey())
                        .withHeaders(getDefaultHeaders());
                break;
            case MULTIPART:
                try {
                    httpRequestCopy = new BaseMultipartRequest(httpRequest.getTemplateString() + String.format("?%s=%s",RESPOND_WITH_QUERY_PARAM,URLEncoder.encode("{\"reltype\": \"http://ns.adobe.com/rel/primary\"}", StandardCharsets.UTF_8.toString())))
                            .withRequestKey(httpRequest.getRequestKey())
                            .withHeaders(getDefaultHeaders());
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
                break;
        }

        return httpRequestCopy;
    }

    private Map<String, String> getDefaultHeaders() {
        Map<String, String> headers = new HashMap<>();
        headers.put(HttpHeaders.ACCEPT, ACCEPT_HEADER_VALUE);
        headers.put(DefaultRequestHeaders.DC_APP_INFO_HEADER_KEY, GlobalConfig.getAppInfo());
        headers.put(DefaultRequestHeaders.PREFER_HEADER_NAME,DefaultRequestHeaders.PREFER_HEADER_VALUE);
        return headers;
    }
}
