/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.exception;

import java.util.Objects;

/**
 * ServiceApiException is thrown when an underlying service API call results in an error.
 *
 * @see SdkException
 * @see ServiceUsageException
 */
public class ServiceApiException extends Exception {

    /**
     * the default value of status code if there is no status code for this service failure.
     */
    public final int DEFAULT_STATUS_CODE = 0;

    /**
     * the default value of error code if there is no error code for this service failure.
     */
    public final String DEFAULT_ERROR_CODE = "UNKNOWN";

    private String requestTrackingId;
    private int statusCode;
    private String errorCode;

    /**
     * Constructs a {@code ServiceApiException} with the specified detail message.
     *
     * @param message the detail message
     */
    public ServiceApiException(String message) {
        super(message);
        this.statusCode = DEFAULT_STATUS_CODE;
        this.errorCode = DEFAULT_ERROR_CODE;
    }

    /**
     * Constructs a {@code ServiceApiException} with the specified detail message, requestId and returned HTTP status code.
     *
     * @param message           the detail message
     * @param requestTrackingId X-Request-Id header value to track the request
     * @param statusCode        HTTP Status code returned in API response
     */
    public ServiceApiException(String message, String requestTrackingId, int statusCode) {
        super(message);
        this.requestTrackingId = requestTrackingId;
        this.statusCode = statusCode;
        this.errorCode = DEFAULT_ERROR_CODE;
    }

    /**
     * Constructs a {@code ServiceApiException} with the specified detail message, requestId, returned HTTP status code and error code.
     *
     * @param message           the detail message
     * @param requestTrackingId X-Request-Id header value to track the request
     * @param statusCode        HTTP Status code returned in API response
     * @param errorCode         Error code returned in API response
     */
    public ServiceApiException(String message, String requestTrackingId, int statusCode, String errorCode) {
        this(message, requestTrackingId, statusCode);
        this.errorCode = Objects.nonNull(errorCode) ? errorCode : DEFAULT_ERROR_CODE;
    }

    /**
     * Returns the HTTP Status code or {@code DEFAULT_STATUS_CODE} if the status code doesn't adequately represent the error.
     *
     * @return the HTTP Status code or {@code DEFAULT_STATUS_CODE} if the status code doesn't adequately represent the error.
     */
    public int getStatusCode() {
        return statusCode;
    }

    /**
     * Returns the Error code or {@code DEFAULT_ERROR_CODE} if the error code doesn't adequately represent the error.
     *
     * @return the Error code or {@code DEFAULT_ERROR_CODE} if the error code doesn't adequately represent the error.
     */
    public String getErrorCode() {
        return errorCode;
    }

    /**
     * Returns the detail message of this exception.
     *
     * @return the detail message
     */
    @Override
    public String getMessage() {
        return String.format("description ='%s'; requestTrackingId='%s'; statusCode=%d; errorCode=%s",
                super.getMessage(), this.getRequestTrackingId(), this.getStatusCode(), this.getErrorCode());
    }

    /**
     * Returns the Request ID (the value of the X-Request-ID header).
     *
     * @return Request ID
     */
    public String getRequestTrackingId() {
        return requestTrackingId;
    }
}

