/*
 * Decompiled with CFR 0.152.
 */
package ca.uhn.hl7v2.validation.impl;

import ca.uhn.hl7v2.Version;
import ca.uhn.hl7v2.util.XMLUtils;
import ca.uhn.hl7v2.validation.ValidationException;
import ca.uhn.hl7v2.validation.impl.AbstractEncodingRule;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.DOMError;
import org.w3c.dom.DOMErrorHandler;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class XMLSchemaRule
extends AbstractEncodingRule {
    private static final String SECTION_REFERENCE = "http://www.hl7.org/Special/committees/xml/drafts/v2xml.html";
    private static final String DESCRIPTION = "Checks that an encoded XML message validates against a declared or default schema (it is recommended to use the standard HL7 schema, but this is not enforced here).";
    private static final Logger log = LoggerFactory.getLogger(XMLSchemaRule.class);
    private static final String DEFAULT_NS = "urn:hl7-org:v2xml";
    private Map<String, String> locations;

    @Override
    public ValidationException[] apply(String msg) {
        ArrayList<ValidationException> validationErrors = new ArrayList<ValidationException>();
        try {
            Document doc = XMLUtils.parse(msg);
            if (this.hasCorrectNamespace(doc, validationErrors)) {
                XMLUtils.validate(doc, this.getSchemaLocation(doc), new ErrorHandler(validationErrors));
            }
        }
        catch (Exception e) {
            log.error("Unable to validate message: {}", (Object)e.getMessage(), (Object)e);
            validationErrors.add(new ValidationException("Unable to validate message " + e.getMessage(), e));
        }
        return validationErrors.toArray(new ValidationException[validationErrors.size()]);
    }

    private String getSchemaLocation(Document doc) throws IOException {
        String schemaFilename = this.extractSchemaLocation(doc);
        if (schemaFilename == null && (schemaFilename = this.staticSchema(doc)) == null) {
            throw new IOException("Unable to retrieve a valid schema to use for message validation");
        }
        return schemaFilename;
    }

    private String extractSchemaLocation(Document doc) {
        String schemaFileName = null;
        log.debug("Trying to retrieve the schema defined in the xml document");
        Element element = doc.getDocumentElement();
        String schemaLocation = element.getAttributeNS("http://www.w3.org/2001/XMLSchema-instance", "schemaLocation");
        if (schemaLocation.length() > 0) {
            log.debug("Schema defined in document: {}", (Object)schemaLocation);
            String[] schemaItems = schemaLocation.split(" ");
            if (schemaItems.length == 2) {
                File f = new File(schemaItems[1]);
                if (f.exists()) {
                    schemaFileName = schemaItems[1];
                    log.debug("Schema defined in document points to a valid file");
                } else {
                    log.warn("Schema file defined in xml document not found on disk: {}", (Object)schemaItems[1]);
                }
            }
        } else {
            log.debug("No schema location defined in the xml document");
        }
        return schemaFileName;
    }

    private String staticSchema(Document doc) {
        String schemaFilename = null;
        log.debug("Lookup HL7 version in MSH-12 to know which default schema to use");
        NodeList nodeList = doc.getElementsByTagNameNS(DEFAULT_NS, "VID.1");
        if (nodeList.getLength() == 1) {
            Node versionNode = nodeList.item(0);
            Version version = Version.versionOf(versionNode.getFirstChild().getNodeValue());
            String schemaLocation = this.locations.get(version.getVersion());
            schemaFilename = schemaLocation + "/" + doc.getDocumentElement().getNodeName() + ".xsd";
            File myFile = new File(schemaFilename);
            if (myFile.exists()) {
                log.debug("Valid schema file present: {}", (Object)schemaFilename);
            } else {
                log.warn("Schema file not found on disk: {}", (Object)schemaFilename);
                schemaFilename = null;
            }
        } else {
            log.error("HL7 version node MSH-12 not present - unable to determine default schema");
        }
        return schemaFilename;
    }

    private boolean hasCorrectNamespace(Document domDocumentToValidate, List<ValidationException> validationErrors) {
        String nsUri = domDocumentToValidate.getDocumentElement().getNamespaceURI();
        boolean ok = DEFAULT_NS.equals(nsUri);
        if (!ok) {
            ValidationException e = new ValidationException("The default namespace of the XML document is incorrect - should be urn:hl7-org:v2xml but was " + nsUri);
            validationErrors.add(e);
            log.error(e.getMessage());
        }
        return ok;
    }

    public void setSchemaLocations(Map<String, String> locations) {
        this.locations = locations;
    }

    Map<String, String> getSchemaLocations() {
        return this.locations;
    }

    @Override
    public String getDescription() {
        return DESCRIPTION;
    }

    @Override
    public String getSectionReference() {
        return SECTION_REFERENCE;
    }

    private static class ErrorHandler
    implements DOMErrorHandler {
        private List<ValidationException> validationErrors;

        public ErrorHandler(List<ValidationException> validationErrors) {
            this.validationErrors = validationErrors;
        }

        @Override
        public boolean handleError(DOMError error) {
            this.validationErrors.add(new ValidationException(this.getSeverity(error) + error.getMessage()));
            return true;
        }

        private String getSeverity(DOMError error) {
            switch (error.getSeverity()) {
                case 1: {
                    return "WARNING: ";
                }
                case 2: {
                    return "ERROR: ";
                }
            }
            return "FATAL ERROR: ";
        }
    }
}

