package app.raybritton.tokenstorage

import app.raybritton.tokenstorage.crypto.Crypto
import app.raybritton.tokenstorage.keyCrypto.KeyCrypto
import app.raybritton.tokenstorage.keyCrypto.NoKeyCrypto
import app.raybritton.tokenstorage.persistence.Persistence
import io.reactivex.BackpressureStrategy
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.subjects.ReplaySubject

class RxTokenStorage(crypto: Crypto,
                     persistence: Persistence,
                     keyCrypto: KeyCrypto = NoKeyCrypto()) {
    private val storage = TokenStorage(crypto, persistence, keyCrypto)

    private val loadSubjects = mutableMapOf<String, ReplaySubject<Optional<String>>>()

    private val keysSubject = ReplaySubject.create<List<String>>(1)

    fun clearAll(): Completable {
        return Completable.fromCallable {
            storage.clearAll()
            loadSubjects.values.forEach {
                it.onNext(Optional.None)
            }
            keysSubject.onNext(listOf())
        }
    }

    fun clear(key: String): Completable {
        return Completable.fromCallable {
            storage.clear(key)
            reload(key)
        }
    }

    fun save(key: String, plaintext: String): Completable {
        return Completable.fromCallable {
            storage.save(key, plaintext)
            reload(key)
        }
    }

    fun load(key: String): Flowable<Optional<String>> {
        storage.crypto.verify()
        if (loadSubjects.containsKey(key)) {
            return loadSubjects[key]!!.share().toFlowable(BackpressureStrategy.LATEST)
        } else {
            val subject = ReplaySubject.create<Optional<String>>(1)
            loadSubjects[key] = subject
            subject.startWith { loadString(key) }
            return subject.share().toFlowable(BackpressureStrategy.LATEST)
        }
    }

    fun keys(): Flowable<List<String>> {
        storage.crypto.verify()
        return keysSubject.share().toFlowable(BackpressureStrategy.LATEST)
                .startWith(storage.keys())
    }

    private fun loadString(key: String): String? {
        return storage.load(key)
    }
    private fun reload(key: String) {
        loadSubjects[key]?.onNext(Optional.auto(loadString(key)))
        keysSubject.onNext(storage.keys())
    }

}