/*
 * Decompiled with CFR 0.152.
 */
package androidx.security.identity;

import android.content.Context;
import android.icu.util.Calendar;
import android.security.keystore.KeyGenParameterSpec;
import android.util.AtomicFile;
import android.util.Log;
import android.util.Pair;
import androidx.annotation.NonNull;
import androidx.security.identity.AccessControlProfile;
import androidx.security.identity.AccessControlProfileId;
import androidx.security.identity.PersonalizationData;
import androidx.security.identity.UnknownAuthenticationKeyException;
import androidx.security.identity.Util;
import co.nstant.in.cbor.CborBuilder;
import co.nstant.in.cbor.CborDecoder;
import co.nstant.in.cbor.CborEncoder;
import co.nstant.in.cbor.CborException;
import co.nstant.in.cbor.builder.ArrayBuilder;
import co.nstant.in.cbor.builder.MapBuilder;
import co.nstant.in.cbor.model.Array;
import co.nstant.in.cbor.model.ByteString;
import co.nstant.in.cbor.model.DataItem;
import co.nstant.in.cbor.model.Map;
import co.nstant.in.cbor.model.Number;
import co.nstant.in.cbor.model.UnicodeString;
import co.nstant.in.cbor.model.UnsignedInteger;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.ByteBuffer;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.AlgorithmParameterSpec;
import java.util.AbstractList;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;

class CredentialData {
    private static final String TAG = "CredentialData";
    private Context mContext;
    private String mCredentialName;
    private String mDocType = "";
    private String mCredentialKeyAlias = "";
    private Collection<X509Certificate> mCertificateChain = null;
    private byte[] mProofOfProvisioningSha256 = null;
    private AbstractList<AccessControlProfile> mAccessControlProfiles = new ArrayList<AccessControlProfile>();
    private AbstractMap<Integer, AccessControlProfile> mProfileIdToAcpMap = new HashMap<Integer, AccessControlProfile>();
    private AbstractList<PersonalizationData.NamespaceData> mNamespaceDatas = new ArrayList<PersonalizationData.NamespaceData>();
    private int mAuthKeyCount = 0;
    private int mAuthMaxUsesPerKey = 1;
    private String mPerReaderSessionKeyAlias = "";
    private AbstractMap<Integer, String> mAcpTimeoutKeyAliases;
    private AbstractList<AuthKeyData> mAuthKeyDatas = new ArrayList<AuthKeyData>();

    private CredentialData(Context context, String credentialName) {
        this.mContext = context;
        this.mCredentialName = credentialName;
    }

    void deleteKeysForReplacement() {
        KeyStore ks;
        try {
            ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new RuntimeException("Error loading keystore", e);
        }
        try {
            if (!this.mPerReaderSessionKeyAlias.isEmpty()) {
                ks.deleteEntry(this.mPerReaderSessionKeyAlias);
            }
            for (String alias : this.mAcpTimeoutKeyAliases.values()) {
                ks.deleteEntry(alias);
            }
            for (AuthKeyData authKeyData : this.mAuthKeyDatas) {
                if (!authKeyData.mAlias.isEmpty()) {
                    ks.deleteEntry(authKeyData.mAlias);
                }
                if (authKeyData.mPendingAlias.isEmpty()) continue;
                ks.deleteEntry(authKeyData.mPendingAlias);
            }
        }
        catch (KeyStoreException e) {
            throw new RuntimeException("Error deleting key", e);
        }
    }

    static CredentialData createCredentialData(Context context, String docType, String credentialName, String credentialKeyAlias, Collection<X509Certificate> certificateChain, PersonalizationData personalizationData, byte[] proofOfProvisioningSha256, boolean isReplacement) {
        if (!isReplacement && CredentialData.credentialAlreadyExists(context, credentialName)) {
            throw new RuntimeException("Credential with given name already exists");
        }
        CredentialData data = new CredentialData(context, credentialName);
        data.mDocType = docType;
        data.mCredentialKeyAlias = credentialKeyAlias;
        data.mCertificateChain = certificateChain;
        data.mProofOfProvisioningSha256 = proofOfProvisioningSha256;
        data.mAccessControlProfiles = new ArrayList<AccessControlProfile>();
        data.mProfileIdToAcpMap = new HashMap<Integer, AccessControlProfile>();
        for (AccessControlProfile item : personalizationData.getAccessControlProfiles()) {
            data.mAccessControlProfiles.add(item);
            data.mProfileIdToAcpMap.put(item.getAccessControlProfileId().getId(), item);
        }
        data.mNamespaceDatas = new ArrayList<PersonalizationData.NamespaceData>();
        data.mNamespaceDatas.addAll(personalizationData.getNamespaceDatas());
        data.mAcpTimeoutKeyAliases = new HashMap<Integer, String>();
        for (AccessControlProfile profile : personalizationData.getAccessControlProfiles()) {
            boolean isAuthRequired = profile.isUserAuthenticationRequired();
            long timeoutSeconds = profile.getUserAuthenticationTimeout();
            if (!isAuthRequired) continue;
            CredentialData.ensurePerReaderSessionKey(credentialName, data);
            CredentialData.ensureAcpTimoutKeyForProfile(credentialName, data, profile, timeoutSeconds);
        }
        data.createDataEncryptionKey();
        data.saveToDisk();
        return data;
    }

    static boolean credentialAlreadyExists(Context context, String credentialName) {
        String filename = CredentialData.getFilenameForCredentialData(credentialName);
        AtomicFile file = new AtomicFile(context.getFileStreamPath(filename));
        try {
            file.openRead();
        }
        catch (FileNotFoundException e) {
            return false;
        }
        return true;
    }

    private static void ensurePerReaderSessionKey(String credentialName, CredentialData data) {
        if (!data.mPerReaderSessionKeyAlias.isEmpty()) {
            return;
        }
        data.mPerReaderSessionKeyAlias = CredentialData.getAcpKeyAliasFromCredentialName(credentialName);
        try {
            KeyGenerator kg = KeyGenerator.getInstance("AES", "AndroidKeyStore");
            KeyGenParameterSpec.Builder builder = new KeyGenParameterSpec.Builder(data.mPerReaderSessionKeyAlias, 3).setBlockModes(new String[]{"GCM"}).setEncryptionPaddings(new String[]{"NoPadding"}).setKeySize(128).setUserAuthenticationRequired(true).setUserAuthenticationValidityDurationSeconds(-1);
            kg.init((AlgorithmParameterSpec)builder.build());
            kg.generateKey();
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchProviderException e) {
            throw new RuntimeException("Error creating ACP auth-bound key", e);
        }
    }

    private static void ensureAcpTimoutKeyForProfile(String credentialName, CredentialData data, AccessControlProfile profile, long timeoutMilliSeconds) {
        if (timeoutMilliSeconds > 0L) {
            int profileId = profile.getAccessControlProfileId().getId();
            String acpAlias = CredentialData.getAcpTimeoutKeyAliasFromCredentialName(credentialName, profileId);
            try {
                int timeoutSeconds = (int)(timeoutMilliSeconds / 1000L);
                KeyGenerator kg = KeyGenerator.getInstance("AES", "AndroidKeyStore");
                KeyGenParameterSpec.Builder builder = new KeyGenParameterSpec.Builder(acpAlias, 3).setBlockModes(new String[]{"GCM"}).setEncryptionPaddings(new String[]{"NoPadding"}).setUserAuthenticationRequired(true).setUserAuthenticationValidityDurationSeconds(timeoutSeconds).setKeySize(128);
                kg.init((AlgorithmParameterSpec)builder.build());
                kg.generateKey();
            }
            catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchProviderException e) {
                throw new RuntimeException("Error creating ACP auth-bound timeout key", e);
            }
            data.mAcpTimeoutKeyAliases.put(profileId, acpAlias);
        }
    }

    static CredentialData loadCredentialData(Context context, String credentialName) {
        CredentialData data = new CredentialData(context, credentialName);
        String dataKeyAlias = CredentialData.getDataKeyAliasFromCredentialName(credentialName);
        if (!data.loadFromDisk(dataKeyAlias)) {
            return null;
        }
        return data;
    }

    static String escapeCredentialName(String componentName, String credentialName) {
        try {
            return "identity_credential_" + componentName + "_" + URLEncoder.encode(credentialName, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unexpected UnsupportedEncodingException", e);
        }
    }

    static String getFilenameForCredentialData(String credentialName) {
        return CredentialData.escapeCredentialName("data", credentialName);
    }

    static String getAliasFromCredentialName(String credentialName) {
        return CredentialData.escapeCredentialName("credkey", credentialName);
    }

    static String getDataKeyAliasFromCredentialName(String credentialName) {
        return CredentialData.escapeCredentialName("datakey", credentialName);
    }

    static String getAcpTimeoutKeyAliasFromCredentialName(String credentialName, int acpProfileId) {
        return CredentialData.escapeCredentialName("acp_timeout_for_id" + acpProfileId, credentialName);
    }

    static String getAcpKeyAliasFromCredentialName(String credentialName) {
        return CredentialData.escapeCredentialName("acp", credentialName);
    }

    PrivateKey getCredentialKeyPrivate() {
        KeyStore.Entry entry;
        try {
            KeyStore ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
            entry = ks.getEntry(this.mCredentialKeyAlias, null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException e) {
            throw new RuntimeException("Error loading keystore", e);
        }
        return ((KeyStore.PrivateKeyEntry)entry).getPrivateKey();
    }

    @NonNull
    byte[] proveOwnership(@NonNull byte[] challenge) {
        byte[] signatureBytes;
        PrivateKey key = this.getCredentialKeyPrivate();
        CborBuilder signedDataBuilder = new CborBuilder();
        signedDataBuilder.addArray().add("ProofOfOwnership").add(this.mDocType).add(challenge).add(false);
        try {
            ByteArrayOutputStream dtsBaos = new ByteArrayOutputStream();
            CborEncoder dtsEncoder = new CborEncoder((OutputStream)dtsBaos);
            dtsEncoder.encode((DataItem)signedDataBuilder.build().get(0));
            byte[] dataToSign = dtsBaos.toByteArray();
            signatureBytes = Util.coseSign1Sign(key, dataToSign, null, null);
        }
        catch (CborException | InvalidKeyException | NoSuchAlgorithmException | CertificateEncodingException e) {
            throw new RuntimeException("Error building ProofOfOwnership", e);
        }
        return signatureBytes;
    }

    static byte[] buildProofOfDeletionSignature(String docType, PrivateKey key, byte[] challenge) {
        byte[] signatureBytes;
        CborBuilder signedDataBuilder = new CborBuilder();
        ArrayBuilder arrayBuilder = signedDataBuilder.addArray();
        arrayBuilder.add("ProofOfDeletion").add(docType);
        if (challenge != null) {
            arrayBuilder.add(challenge);
        }
        arrayBuilder.add(false);
        try {
            ByteArrayOutputStream dtsBaos = new ByteArrayOutputStream();
            CborEncoder dtsEncoder = new CborEncoder((OutputStream)dtsBaos);
            dtsEncoder.encode((DataItem)signedDataBuilder.build().get(0));
            byte[] dataToSign = dtsBaos.toByteArray();
            signatureBytes = Util.coseSign1Sign(key, dataToSign, null, null);
        }
        catch (CborException | InvalidKeyException | NoSuchAlgorithmException | CertificateEncodingException e) {
            throw new RuntimeException("Error building ProofOfDeletion", e);
        }
        return signatureBytes;
    }

    static byte[] delete(Context context, String credentialName, byte[] challenge) {
        KeyStore.Entry entry;
        KeyStore ks;
        String filename = CredentialData.getFilenameForCredentialData(credentialName);
        AtomicFile file = new AtomicFile(context.getFileStreamPath(filename));
        try {
            file.openRead();
        }
        catch (FileNotFoundException e) {
            return null;
        }
        CredentialData data = new CredentialData(context, credentialName);
        String dataKeyAlias = CredentialData.getDataKeyAliasFromCredentialName(credentialName);
        try {
            data.loadFromDisk(dataKeyAlias);
        }
        catch (RuntimeException e) {
            Log.e((String)TAG, (String)"Error parsing file on disk (old version?). Deleting anyway.");
            file.delete();
            return null;
        }
        try {
            ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
            entry = ks.getEntry(data.mCredentialKeyAlias, null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException e) {
            throw new RuntimeException("Error loading keystore", e);
        }
        byte[] signature = CredentialData.buildProofOfDeletionSignature(data.mDocType, ((KeyStore.PrivateKeyEntry)entry).getPrivateKey(), challenge);
        file.delete();
        try {
            ks.deleteEntry(data.mCredentialKeyAlias);
            if (!data.mPerReaderSessionKeyAlias.isEmpty()) {
                ks.deleteEntry(data.mPerReaderSessionKeyAlias);
            }
            for (String alias : data.mAcpTimeoutKeyAliases.values()) {
                ks.deleteEntry(alias);
            }
            for (AuthKeyData authKeyData : data.mAuthKeyDatas) {
                if (!authKeyData.mAlias.isEmpty()) {
                    ks.deleteEntry(authKeyData.mAlias);
                }
                if (authKeyData.mPendingAlias.isEmpty()) continue;
                ks.deleteEntry(authKeyData.mPendingAlias);
            }
        }
        catch (KeyStoreException e) {
            throw new RuntimeException("Error deleting key", e);
        }
        return signature;
    }

    private void createDataEncryptionKey() {
        try {
            String dataKeyAlias = CredentialData.getDataKeyAliasFromCredentialName(this.mCredentialName);
            KeyGenerator kg = KeyGenerator.getInstance("AES", "AndroidKeyStore");
            KeyGenParameterSpec.Builder builder = new KeyGenParameterSpec.Builder(dataKeyAlias, 3).setBlockModes(new String[]{"GCM"}).setEncryptionPaddings(new String[]{"NoPadding"}).setKeySize(128);
            kg.init((AlgorithmParameterSpec)builder.build());
            kg.generateKey();
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchProviderException e) {
            throw new RuntimeException("Error creating data encryption key", e);
        }
    }

    private void saveToDisk() {
        CborBuilder builder = new CborBuilder();
        MapBuilder map = builder.addMap();
        this.saveToDiskBasic((MapBuilder<CborBuilder>)map);
        this.saveToDiskAuthDatas((MapBuilder<CborBuilder>)map);
        this.saveToDiskACPs((MapBuilder<CborBuilder>)map);
        this.saveToDiskNamespaceDatas((MapBuilder<CborBuilder>)map);
        this.saveToDiskAuthKeys((MapBuilder<CborBuilder>)map);
        byte[] cleartextDataToSaveBytes = this.saveToDiskEncode(builder);
        byte[] dataToSaveBytes = this.saveToDiskEncrypt(cleartextDataToSaveBytes);
        String filename = CredentialData.getFilenameForCredentialData(this.mCredentialName);
        AtomicFile file = new AtomicFile(this.mContext.getFileStreamPath(filename));
        FileOutputStream outputStream = null;
        try {
            outputStream = file.startWrite();
            outputStream.write(dataToSaveBytes);
            outputStream.close();
            file.finishWrite(outputStream);
        }
        catch (IOException e) {
            if (outputStream != null) {
                file.failWrite(outputStream);
            }
            throw new RuntimeException("Error writing data", e);
        }
    }

    private byte[] saveToDiskEncode(CborBuilder map) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        CborEncoder encoder = new CborEncoder((OutputStream)baos);
        try {
            encoder.encode(map.build());
        }
        catch (CborException e) {
            throw new RuntimeException("Error encoding data", e);
        }
        return baos.toByteArray();
    }

    private byte[] saveToDiskEncrypt(byte[] cleartextDataToSaveBytes) {
        byte[] dataToSaveBytes;
        try {
            KeyStore ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
            String dataKeyAlias = CredentialData.getDataKeyAliasFromCredentialName(this.mCredentialName);
            KeyStore.Entry entry = ks.getEntry(dataKeyAlias, null);
            SecretKey secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
            Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
            cipher.init(1, secretKey);
            byte[] cipherText = cipher.doFinal(cleartextDataToSaveBytes);
            ByteBuffer byteBuffer = ByteBuffer.allocate(12 + cipherText.length);
            byteBuffer.put(cipher.getIV());
            byteBuffer.put(cipherText);
            dataToSaveBytes = byteBuffer.array();
        }
        catch (IOException | InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException | BadPaddingException | IllegalBlockSizeException | NoSuchPaddingException e) {
            throw new RuntimeException("Error encrypting CBOR for saving to disk", e);
        }
        return dataToSaveBytes;
    }

    private void saveToDiskAuthKeys(MapBuilder<CborBuilder> map) {
        map.put("perReaderSessionKeyAlias", this.mPerReaderSessionKeyAlias);
        MapBuilder acpTimeoutKeyMapBuilder = map.putMap("acpTimeoutKeyMap");
        for (Map.Entry<Integer, String> entry : this.mAcpTimeoutKeyAliases.entrySet()) {
            int profileId = entry.getKey();
            String acpAlias = entry.getValue();
            acpTimeoutKeyMapBuilder.put((DataItem)new UnsignedInteger((long)profileId), (DataItem)new UnicodeString(acpAlias));
        }
    }

    private void saveToDiskNamespaceDatas(MapBuilder<CborBuilder> map) {
        MapBuilder ensArrayBuilder = map.putMap("namespaceDatas");
        for (PersonalizationData.NamespaceData namespaceData : this.mNamespaceDatas) {
            ensArrayBuilder.put((DataItem)new UnicodeString(namespaceData.getNamespaceName()), Util.namespaceDataToCbor(namespaceData));
        }
    }

    private void saveToDiskACPs(MapBuilder<CborBuilder> map) {
        ArrayBuilder acpArrayBuilder = map.putArray("accessControlProfiles");
        for (AccessControlProfile profile : this.mAccessControlProfiles) {
            acpArrayBuilder.add(Util.accessControlProfileToCbor(profile));
        }
    }

    private void saveToDiskAuthDatas(MapBuilder<CborBuilder> map) {
        ArrayBuilder authKeyDataArrayBuilder = map.putArray("authKeyDatas");
        for (AuthKeyData data : this.mAuthKeyDatas) {
            long expirationDateMillis = Long.MAX_VALUE;
            if (data.mExpirationDate != null) {
                expirationDateMillis = data.mExpirationDate.getTimeInMillis();
            }
            authKeyDataArrayBuilder.addMap().put("alias", data.mAlias).put("useCount", (long)data.mUseCount).put("certificate", data.mCertificate).put("staticAuthenticationData", data.mStaticAuthenticationData).put("pendingAlias", data.mPendingAlias).put("pendingCertificate", data.mPendingCertificate).put("expirationDateMillis", expirationDateMillis).end();
        }
    }

    private void saveToDiskBasic(MapBuilder<CborBuilder> map) {
        map.put("docType", this.mDocType);
        map.put("credentialKeyAlias", this.mCredentialKeyAlias);
        ArrayBuilder credentialKeyCertChainBuilder = map.putArray("credentialKeyCertChain");
        for (X509Certificate certificate : this.mCertificateChain) {
            try {
                credentialKeyCertChainBuilder.add(certificate.getEncoded());
            }
            catch (CertificateEncodingException e) {
                throw new RuntimeException("Error encoding certificate", e);
            }
        }
        map.put("proofOfProvisioningSha256", this.mProofOfProvisioningSha256);
        map.put("authKeyCount", (long)this.mAuthKeyCount);
        map.put("authKeyMaxUses", (long)this.mAuthMaxUsesPerKey);
    }

    private boolean loadFromDisk(String dataKeyAlias) {
        String filename = CredentialData.getFilenameForCredentialData(this.mCredentialName);
        byte[] encryptedFileData = new byte[]{};
        try {
            AtomicFile file = new AtomicFile(this.mContext.getFileStreamPath(filename));
            encryptedFileData = file.readFully();
        }
        catch (Exception e) {
            return false;
        }
        byte[] fileData = this.loadFromDiskDecrypt(dataKeyAlias, encryptedFileData);
        try {
            ByteArrayInputStream bais = new ByteArrayInputStream(fileData);
            List dataItems = new CborDecoder((InputStream)bais).decode();
            if (dataItems.size() != 1) {
                throw new RuntimeException("Expected 1 item, found " + dataItems.size());
            }
            if (!(dataItems.get(0) instanceof Map)) {
                throw new RuntimeException("Item is not a map");
            }
            Map map = (Map)dataItems.get(0);
            this.loadBasic(map);
            this.loadCredentialKeyCertChain(map);
            this.loadProofOfProvisioningSha256(map);
            this.loadAccessControlProfiles(map);
            this.loadNamespaceDatas(map);
            this.loadAuthKey(map);
        }
        catch (CborException e) {
            throw new RuntimeException("Error decoding data", e);
        }
        return true;
    }

    private byte[] loadFromDiskDecrypt(String dataKeyAlias, byte[] encryptedFileData) {
        byte[] fileData = null;
        try {
            KeyStore ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
            KeyStore.Entry entry = ks.getEntry(dataKeyAlias, null);
            SecretKey secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
            if (encryptedFileData.length < 12) {
                throw new RuntimeException("Encrypted CBOR on disk is too small");
            }
            ByteBuffer byteBuffer = ByteBuffer.wrap(encryptedFileData);
            byte[] iv = new byte[12];
            byteBuffer.get(iv);
            byte[] cipherText = new byte[encryptedFileData.length - 12];
            byteBuffer.get(cipherText);
            Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
            cipher.init(2, (Key)secretKey, new GCMParameterSpec(128, iv));
            fileData = cipher.doFinal(cipherText);
        }
        catch (IOException | InvalidAlgorithmParameterException | InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException | BadPaddingException | IllegalBlockSizeException | NoSuchPaddingException e) {
            throw new RuntimeException("Error decrypting CBOR", e);
        }
        return fileData;
    }

    private void loadBasic(Map map) {
        this.mDocType = ((UnicodeString)map.get((DataItem)new UnicodeString("docType"))).getString();
        this.mCredentialKeyAlias = ((UnicodeString)map.get((DataItem)new UnicodeString("credentialKeyAlias"))).getString();
    }

    private void loadAuthKey(Map map) {
        this.mPerReaderSessionKeyAlias = ((UnicodeString)map.get((DataItem)new UnicodeString("perReaderSessionKeyAlias"))).getString();
        DataItem userAuthKeyAliases = map.get((DataItem)new UnicodeString("acpTimeoutKeyMap"));
        if (!(userAuthKeyAliases instanceof Map)) {
            throw new RuntimeException("acpTimeoutKeyMap not found or not map");
        }
        this.mAcpTimeoutKeyAliases = new HashMap<Integer, String>();
        for (DataItem key : ((Map)userAuthKeyAliases).getKeys()) {
            if (!(key instanceof UnsignedInteger)) {
                throw new RuntimeException("Key in acpTimeoutKeyMap is not an integer");
            }
            int profileId = ((UnsignedInteger)key).getValue().intValue();
            DataItem item = ((Map)userAuthKeyAliases).get(key);
            if (!(item instanceof UnicodeString)) {
                throw new RuntimeException("Item in acpTimeoutKeyMap is not a string");
            }
            String acpAlias = ((UnicodeString)item).getString();
            this.mAcpTimeoutKeyAliases.put(profileId, acpAlias);
        }
        this.mAuthKeyCount = ((Number)map.get((DataItem)new UnicodeString("authKeyCount"))).getValue().intValue();
        this.mAuthMaxUsesPerKey = ((Number)map.get((DataItem)new UnicodeString("authKeyMaxUses"))).getValue().intValue();
        DataItem authKeyDatas = map.get((DataItem)new UnicodeString("authKeyDatas"));
        if (!(authKeyDatas instanceof Array)) {
            throw new RuntimeException("authKeyDatas not found or not array");
        }
        this.mAuthKeyDatas = new ArrayList<AuthKeyData>();
        for (DataItem item : ((Array)authKeyDatas).getDataItems()) {
            AuthKeyData data = new AuthKeyData();
            Map im = (Map)item;
            data.mAlias = ((UnicodeString)im.get((DataItem)new UnicodeString("alias"))).getString();
            data.mUseCount = ((Number)im.get((DataItem)new UnicodeString("useCount"))).getValue().intValue();
            data.mCertificate = ((ByteString)im.get((DataItem)new UnicodeString("certificate"))).getBytes();
            data.mStaticAuthenticationData = ((ByteString)im.get((DataItem)new UnicodeString("staticAuthenticationData"))).getBytes();
            data.mPendingAlias = ((UnicodeString)im.get((DataItem)new UnicodeString("pendingAlias"))).getString();
            data.mPendingCertificate = ((ByteString)im.get((DataItem)new UnicodeString("pendingCertificate"))).getBytes();
            long expirationDateMillis = Long.MAX_VALUE;
            DataItem expirationDateMillisItem = im.get((DataItem)new UnicodeString("expirationDateMillis"));
            if (expirationDateMillisItem != null) {
                if (!(expirationDateMillisItem instanceof Number)) {
                    throw new RuntimeException("expirationDateMillis not a number");
                }
                expirationDateMillis = ((Number)expirationDateMillisItem).getValue().longValue();
            }
            Calendar expirationDate = Calendar.getInstance();
            expirationDate.setTimeInMillis(expirationDateMillis);
            data.mExpirationDate = expirationDate;
            this.mAuthKeyDatas.add(data);
        }
    }

    private void loadNamespaceDatas(Map map) {
        DataItem namespaceDatas = map.get((DataItem)new UnicodeString("namespaceDatas"));
        if (!(namespaceDatas instanceof Map)) {
            throw new RuntimeException("namespaceDatas not found or not map");
        }
        this.mNamespaceDatas = new ArrayList<PersonalizationData.NamespaceData>();
        for (DataItem key : ((Map)namespaceDatas).getKeys()) {
            if (!(key instanceof UnicodeString)) {
                throw new RuntimeException("Key in namespaceDatas is not a string");
            }
            String namespaceName = ((UnicodeString)key).getString();
            DataItem item = ((Map)namespaceDatas).get(key);
            this.mNamespaceDatas.add(Util.namespaceDataFromCbor(namespaceName, item));
        }
    }

    private void loadAccessControlProfiles(Map map) {
        DataItem accessControlProfiles = map.get((DataItem)new UnicodeString("accessControlProfiles"));
        if (!(accessControlProfiles instanceof Array)) {
            throw new RuntimeException("accessControlProfiles not found or not array");
        }
        this.mAccessControlProfiles = new ArrayList<AccessControlProfile>();
        this.mProfileIdToAcpMap = new HashMap<Integer, AccessControlProfile>();
        for (DataItem item : ((Array)accessControlProfiles).getDataItems()) {
            AccessControlProfile profile = Util.accessControlProfileFromCbor(item);
            this.mAccessControlProfiles.add(profile);
            this.mProfileIdToAcpMap.put(profile.getAccessControlProfileId().getId(), profile);
        }
    }

    private void loadProofOfProvisioningSha256(Map map) {
        DataItem proofOfProvisioningSha256 = map.get((DataItem)new UnicodeString("proofOfProvisioningSha256"));
        if (!(proofOfProvisioningSha256 instanceof ByteString)) {
            throw new RuntimeException("proofOfProvisioningSha256 not found or not bstr");
        }
        this.mProofOfProvisioningSha256 = ((ByteString)proofOfProvisioningSha256).getBytes();
    }

    private void loadCredentialKeyCertChain(Map map) {
        DataItem credentialKeyCertChain = map.get((DataItem)new UnicodeString("credentialKeyCertChain"));
        if (!(credentialKeyCertChain instanceof Array)) {
            throw new RuntimeException("credentialKeyCertChain not found or not array");
        }
        this.mCertificateChain = new ArrayList<X509Certificate>();
        for (DataItem item : ((Array)credentialKeyCertChain).getDataItems()) {
            byte[] encodedCert = ((ByteString)item).getBytes();
            try {
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                ByteArrayInputStream certBais = new ByteArrayInputStream(encodedCert);
                this.mCertificateChain.add((X509Certificate)cf.generateCertificate(certBais));
            }
            catch (CertificateException e) {
                throw new RuntimeException("Error decoding certificate blob", e);
            }
        }
    }

    Collection<AccessControlProfile> getAccessControlProfiles() {
        return this.mAccessControlProfiles;
    }

    Collection<PersonalizationData.NamespaceData> getNamespaceDatas() {
        return this.mNamespaceDatas;
    }

    PersonalizationData.NamespaceData lookupNamespaceData(String nameSpace) {
        for (PersonalizationData.NamespaceData namespaceData : this.mNamespaceDatas) {
            if (!namespaceData.getNamespaceName().equals(nameSpace)) continue;
            return namespaceData;
        }
        return null;
    }

    String getCredentialKeyAlias() {
        return this.mCredentialKeyAlias;
    }

    String getPerReaderSessionKeyAlias() {
        return this.mPerReaderSessionKeyAlias;
    }

    int getAuthKeyCount() {
        return this.mAuthKeyCount;
    }

    int getAuthMaxUsesPerKey() {
        return this.mAuthMaxUsesPerKey;
    }

    int[] getAuthKeyUseCounts() {
        int[] result = new int[this.mAuthKeyCount];
        int n = 0;
        for (AuthKeyData data : this.mAuthKeyDatas) {
            result[n++] = data.mUseCount;
        }
        return result;
    }

    void setAvailableAuthenticationKeys(int keyCount, int maxUsesPerKey) {
        int prevAuthKeyCount = this.mAuthKeyCount;
        this.mAuthKeyCount = keyCount;
        this.mAuthMaxUsesPerKey = maxUsesPerKey;
        if (prevAuthKeyCount < this.mAuthKeyCount) {
            for (int n = prevAuthKeyCount; n < this.mAuthKeyCount; ++n) {
                this.mAuthKeyDatas.add(new AuthKeyData());
            }
        } else if (prevAuthKeyCount > this.mAuthKeyCount) {
            KeyStore ks = null;
            try {
                ks = KeyStore.getInstance("AndroidKeyStore");
                ks.load(null);
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                throw new RuntimeException("Error loading keystore", e);
            }
            int numKeysToDelete = prevAuthKeyCount - this.mAuthKeyCount;
            for (int n = 0; n < numKeysToDelete; ++n) {
                AuthKeyData data = this.mAuthKeyDatas.get(0);
                if (!data.mAlias.isEmpty()) {
                    try {
                        if (ks.containsAlias(data.mAlias)) {
                            ks.deleteEntry(data.mAlias);
                        }
                    }
                    catch (KeyStoreException e) {
                        throw new RuntimeException("Error deleting auth key with mAlias " + data.mAlias, e);
                    }
                }
                if (!data.mPendingAlias.isEmpty()) {
                    try {
                        if (ks.containsAlias(data.mPendingAlias)) {
                            ks.deleteEntry(data.mPendingAlias);
                        }
                    }
                    catch (KeyStoreException e) {
                        throw new RuntimeException("Error deleting auth key with mPendingAlias " + data.mPendingAlias, e);
                    }
                }
                this.mAuthKeyDatas.remove(0);
            }
        }
        this.saveToDisk();
    }

    Collection<X509Certificate> getAuthKeysNeedingCertification() {
        KeyStore ks = null;
        try {
            ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new RuntimeException("Error loading keystore", e);
        }
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>();
        Calendar now = Calendar.getInstance();
        for (int n = 0; n < this.mAuthKeyCount; ++n) {
            boolean certificationPending;
            AuthKeyData data = this.mAuthKeyDatas.get(n);
            boolean keyExceededUseCount = data.mUseCount >= this.mAuthMaxUsesPerKey;
            boolean keyBeyondExpirationDate = false;
            if (data.mExpirationDate != null) {
                keyBeyondExpirationDate = now.after((Object)data.mExpirationDate);
            }
            boolean newKeyNeeded = data.mAlias.isEmpty() || keyExceededUseCount || keyBeyondExpirationDate;
            boolean bl = certificationPending = !data.mPendingAlias.isEmpty();
            if (newKeyNeeded && !certificationPending) {
                try {
                    String aliasForAuthKey = this.mCredentialKeyAlias + String.format("_auth_%d", n);
                    if (aliasForAuthKey.equals(data.mAlias)) {
                        aliasForAuthKey = aliasForAuthKey + "_";
                    }
                    KeyPairGenerator kpg = KeyPairGenerator.getInstance("EC", "AndroidKeyStore");
                    KeyGenParameterSpec.Builder builder = new KeyGenParameterSpec.Builder(aliasForAuthKey, 12).setDigests(new String[]{"SHA-256", "SHA-512"});
                    kpg.initialize((AlgorithmParameterSpec)builder.build());
                    kpg.generateKeyPair();
                    X509Certificate certificate = Util.generateAuthenticationKeyCert(aliasForAuthKey, this.mCredentialKeyAlias, this.mProofOfProvisioningSha256);
                    data.mPendingAlias = aliasForAuthKey;
                    data.mPendingCertificate = certificate.getEncoded();
                    certificationPending = true;
                }
                catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | NoSuchProviderException | CertificateEncodingException e) {
                    throw new RuntimeException("Error creating auth key", e);
                }
            }
            if (!certificationPending) continue;
            try {
                CertificateFactory cf = CertificateFactory.getInstance("X.509");
                ByteArrayInputStream bais = new ByteArrayInputStream(data.mPendingCertificate);
                certificates.add((X509Certificate)cf.generateCertificate(bais));
                continue;
            }
            catch (CertificateException e) {
                throw new RuntimeException("Error creating certificate for auth key", e);
            }
        }
        this.saveToDisk();
        return certificates;
    }

    void storeStaticAuthenticationData(X509Certificate authenticationKey, Calendar expirationDate, byte[] staticAuthData) throws UnknownAuthenticationKeyException {
        AuthKeyData dataForAuthKey = null;
        CertificateFactory cf = null;
        try {
            cf = CertificateFactory.getInstance("X.509");
            for (AuthKeyData data : this.mAuthKeyDatas) {
                ByteArrayInputStream bais;
                X509Certificate certificate;
                if (data.mPendingCertificate.length <= 0 || !(certificate = (X509Certificate)cf.generateCertificate(bais = new ByteArrayInputStream(data.mPendingCertificate))).equals(authenticationKey)) continue;
                dataForAuthKey = data;
                break;
            }
        }
        catch (CertificateException e) {
            throw new RuntimeException("Error encoding certificate", e);
        }
        if (dataForAuthKey == null) {
            throw new UnknownAuthenticationKeyException("No such authentication key");
        }
        if (!dataForAuthKey.mAlias.isEmpty()) {
            KeyStore ks = null;
            try {
                ks = KeyStore.getInstance("AndroidKeyStore");
                ks.load(null);
                if (ks.containsAlias(dataForAuthKey.mAlias)) {
                    ks.deleteEntry(dataForAuthKey.mAlias);
                }
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                throw new RuntimeException("Error deleting old authentication key", e);
            }
        }
        dataForAuthKey.mAlias = dataForAuthKey.mPendingAlias;
        dataForAuthKey.mCertificate = dataForAuthKey.mPendingCertificate;
        dataForAuthKey.mStaticAuthenticationData = staticAuthData;
        dataForAuthKey.mUseCount = 0;
        dataForAuthKey.mPendingAlias = "";
        dataForAuthKey.mPendingCertificate = new byte[0];
        dataForAuthKey.mExpirationDate = expirationDate;
        this.saveToDisk();
    }

    Pair<PrivateKey, byte[]> selectAuthenticationKey(boolean allowUsingExhaustedKeys, boolean allowUsingExpiredKeys) {
        Pair<PrivateKey, byte[]> keyAndStaticData = this.selectAuthenticationKeyHelper(allowUsingExhaustedKeys, false);
        if (keyAndStaticData != null) {
            return keyAndStaticData;
        }
        if (!allowUsingExpiredKeys) {
            return null;
        }
        return this.selectAuthenticationKeyHelper(allowUsingExhaustedKeys, true);
    }

    Pair<PrivateKey, byte[]> selectAuthenticationKeyHelper(boolean allowUsingExhaustedKeys, boolean allowUsingExpiredKeys) {
        AuthKeyData candidate = null;
        Calendar now = Calendar.getInstance();
        for (int n = 0; n < this.mAuthKeyCount; ++n) {
            AuthKeyData data = this.mAuthKeyDatas.get(n);
            if (data.mAlias.isEmpty() || data.mExpirationDate != null && now.after((Object)data.mExpirationDate) && !allowUsingExpiredKeys || candidate != null && data.mUseCount >= candidate.mUseCount) continue;
            candidate = data;
        }
        if (candidate == null) {
            return null;
        }
        if (candidate.mUseCount >= this.mAuthMaxUsesPerKey && !allowUsingExhaustedKeys) {
            return null;
        }
        KeyStore.Entry entry = null;
        try {
            KeyStore ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
            entry = ks.getEntry(candidate.mAlias, null);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException e) {
            throw new RuntimeException("Error loading keystore", e);
        }
        Pair result = new Pair((Object)((KeyStore.PrivateKeyEntry)entry).getPrivateKey(), (Object)candidate.mStaticAuthenticationData);
        ++candidate.mUseCount;
        this.saveToDisk();
        return result;
    }

    String getDocType() {
        return this.mDocType;
    }

    Collection<X509Certificate> getCredentialKeyCertificateChain() {
        return this.mCertificateChain;
    }

    AccessControlProfile getAccessControlProfile(AccessControlProfileId profileId) {
        AccessControlProfile profile = this.mProfileIdToAcpMap.get(profileId.getId());
        if (profile == null) {
            throw new RuntimeException("No profile with id " + profileId.getId());
        }
        return profile;
    }

    private boolean checkUserAuthenticationTimeout(String acpAlias) {
        try {
            KeyStore ks = KeyStore.getInstance("AndroidKeyStore");
            ks.load(null);
            KeyStore.Entry entry = ks.getEntry(acpAlias, null);
            SecretKey secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
            Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
            cipher.init(1, secretKey);
            byte[] clearText = new byte[]{1, 2};
            cipher.doFinal(clearText);
        }
        catch (IOException | InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException | BadPaddingException | IllegalBlockSizeException | NoSuchPaddingException e) {
            return false;
        }
        return true;
    }

    boolean checkUserAuthentication(AccessControlProfileId accessControlProfileId, boolean didUserAuth) {
        AccessControlProfile profile = this.getAccessControlProfile(accessControlProfileId);
        if (profile.getUserAuthenticationTimeout() == 0L) {
            return didUserAuth;
        }
        String acpAlias = this.mAcpTimeoutKeyAliases.get(accessControlProfileId.getId());
        if (acpAlias == null) {
            throw new RuntimeException("No key alias for ACP with ID " + accessControlProfileId.getId());
        }
        return this.checkUserAuthenticationTimeout(acpAlias);
    }

    private static class AuthKeyData {
        String mAlias = "";
        byte[] mCertificate = new byte[0];
        byte[] mStaticAuthenticationData = new byte[0];
        int mUseCount = 0;
        String mPendingAlias = "";
        byte[] mPendingCertificate = new byte[0];
        Calendar mExpirationDate = null;

        AuthKeyData() {
        }
    }
}

