/*
 * Decompiled with CFR 0.152.
 */
package ai.z.openapi.service.realtime;

import ai.z.openapi.service.realtime.JasonUtil;
import ai.z.openapi.service.realtime.RealtimeClientEvent;
import ai.z.openapi.service.realtime.RealtimeServerEvent;
import java.io.Closeable;
import java.io.IOException;
import java.util.EnumSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class OkHttpRealtimeClient
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(OkHttpRealtimeClient.class);
    private final OkHttpClient client;
    private final CommunicationProvider communicationProvider;
    private final AtomicBoolean isDisposed = new AtomicBoolean(false);
    private final AtomicReference<WebSocket> webSocketRef = new AtomicReference();
    private final Consumer<RealtimeServerEvent> serverEventHandler;
    private final ConnectivityMonitor connectivityMonitor = new ConnectivityMonitor();
    private final boolean closeClientOnClose;

    public OkHttpRealtimeClient(CommunicationProvider communicationProvider, Consumer<RealtimeServerEvent> serverEventHandler, OkHttpClient client) {
        this.client = client;
        this.communicationProvider = communicationProvider;
        this.serverEventHandler = serverEventHandler;
        this.closeClientOnClose = false;
    }

    public OkHttpRealtimeClient(CommunicationProvider communicationProvider, Consumer<RealtimeServerEvent> serverEventHandler, OkHttpClient client, boolean closeClientOnClose) {
        this.client = client;
        this.communicationProvider = communicationProvider;
        this.serverEventHandler = serverEventHandler;
        this.closeClientOnClose = false;
    }

    public OkHttpRealtimeClient(CommunicationProvider communicationProvider, Consumer<RealtimeServerEvent> serverEventHandler) {
        ExecutorService executorService = Executors.newFixedThreadPool(4);
        Dispatcher dispatcher = new Dispatcher(executorService);
        dispatcher.setMaxRequests(4);
        dispatcher.setMaxRequestsPerHost(2);
        this.client = new OkHttpClient.Builder().connectTimeout(5L, TimeUnit.SECONDS).readTimeout(20L, TimeUnit.SECONDS).writeTimeout(20L, TimeUnit.SECONDS).callTimeout(40L, TimeUnit.SECONDS).pingInterval(10L, TimeUnit.SECONDS).dispatcher(dispatcher).build();
        this.communicationProvider = communicationProvider;
        this.serverEventHandler = serverEventHandler;
        this.closeClientOnClose = true;
    }

    public void start() {
        if (this.isDisposed.get()) {
            throw new IllegalStateException("Client is closed");
        }
        EnumSet<ConnectivityState> allowedStates = EnumSet.of(ConnectivityState.STOPPED, ConnectivityState.DISCONNECTED);
        if (!this.connectivityMonitor.changeStateOnAnyOf(allowedStates, ConnectivityState.CONNECTING)) {
            throw new IllegalStateException("Cannot start connection in state " + (Object)((Object)this.connectivityMonitor.get()));
        }
        WebSocketListener listener = new WebSocketListener(){

            public void onOpen(WebSocket webSocket, Response response) {
                logger.info("WebSocket connection established");
                OkHttpRealtimeClient.this.webSocketRef.set(webSocket);
                OkHttpRealtimeClient.this.connectivityMonitor.changeState(ConnectivityState.CONNECTED);
            }

            public void onMessage(WebSocket webSocket, String text) {
                logger.debug("Received message: {}", (Object)text);
                RealtimeServerEvent serverEvent = JasonUtil.fromJsonToServerEvent(text);
                if (serverEvent == null) {
                    logger.error("Unable to parse server event: {}", (Object)text);
                    return;
                }
                OkHttpRealtimeClient.this.serverEventHandler.accept(serverEvent);
            }

            public void onClosed(WebSocket webSocket, int code, String reason) {
                logger.info("Connection closed normally, reason: {}", (Object)reason);
                OkHttpRealtimeClient.this.webSocketRef.set(null);
                if (!this.isStoppingState()) {
                    OkHttpRealtimeClient.this.connectivityMonitor.changeState(ConnectivityState.DISCONNECTED);
                }
            }

            public void onFailure(WebSocket webSocket, Throwable t, Response response) {
                logger.error("Connection error", t);
                if (response != null) {
                    logger.error("Error response code: {}, response content: {}", (Object)response.code(), (Object)(response.body() != null ? response.body().toString() : "Empty content"));
                }
                OkHttpRealtimeClient.this.webSocketRef.set(null);
                if (!this.isStoppingState()) {
                    OkHttpRealtimeClient.this.connectivityMonitor.changeState(ConnectivityState.DISCONNECTED);
                }
            }

            private boolean isStoppingState() {
                ConnectivityState state = OkHttpRealtimeClient.this.connectivityMonitor.get();
                return state == ConnectivityState.STOPPING || state == ConnectivityState.CLOSED;
            }
        };
        Request request = new Request.Builder().url(this.communicationProvider.getWebSocketUrl()).addHeader("Authorization", "Bearer " + this.communicationProvider.getAuthToken()).build();
        request.url().redact();
        this.client.newWebSocket(request, listener);
        this.client.dispatcher().executorService().submit(() -> logger.info("WebSocket connection thread started"));
    }

    public void stop() {
        ConnectivityState currentState = this.connectivityMonitor.get();
        if (currentState == ConnectivityState.CLOSED) {
            throw new IllegalStateException("Client is closed");
        }
        if (this.connectivityMonitor.changeStateOn(ConnectivityState.CONNECTED, ConnectivityState.STOPPING)) {
            WebSocket webSocket = this.webSocketRef.get();
            if (webSocket != null) {
                webSocket.close(1000, "Normal close");
            }
        } else {
            logger.warn("Stop failed, current state: {}", (Object)this.connectivityMonitor.get());
        }
    }

    public void waitForConnection() throws InterruptedException {
        while (this.connectivityMonitor.get() != ConnectivityState.CONNECTED) {
            Thread.sleep(100L);
        }
    }

    public void sendMessage(RealtimeClientEvent clientEvent) {
        ConnectivityState state = this.connectivityMonitor.get();
        if (state != ConnectivityState.CONNECTED) {
            throw new IllegalStateException("Connection not ready, current state: " + (Object)((Object)state));
        }
        WebSocket webSocket = this.webSocketRef.get();
        if (webSocket == null) {
            throw new IllegalStateException("WebSocket connection not established");
        }
        String jsonMessage = JasonUtil.toJsonFromClientEvent(clientEvent);
        if (jsonMessage == null) {
            logger.error("Unable to serialize client event: type={}, event_id={}", (Object)clientEvent.getType(), (Object)clientEvent.getEventId());
            return;
        }
        webSocket.send(jsonMessage);
    }

    @Override
    public void close() throws IOException {
        if (this.isDisposed.compareAndSet(false, true)) {
            this.connectivityMonitor.changeState(ConnectivityState.CLOSED);
            WebSocket webSocket = this.webSocketRef.get();
            if (webSocket != null) {
                webSocket.close(1000, "Client close");
                this.webSocketRef.set(null);
            }
            if (this.closeClientOnClose) {
                this.client.dispatcher().executorService().shutdown();
            }
        }
    }

    public static final class ConnectivityMonitor {
        private final AtomicReference<ConnectivityState> clientState = new AtomicReference<ConnectivityState>(ConnectivityState.STOPPED);

        public ConnectivityState get() {
            return this.clientState.get();
        }

        public boolean changeStateOn(ConnectivityState expected, ConnectivityState newState) {
            boolean updated = this.clientState.compareAndSet(expected, newState);
            if (updated) {
                logger.info("State change: {} -> {}", (Object)expected, (Object)newState);
            }
            return updated;
        }

        public boolean changeStateOnAnyOf(EnumSet<ConnectivityState> expecteds, ConnectivityState newState) {
            ConnectivityState current;
            do {
                if (expecteds.contains((Object)(current = this.clientState.get()))) continue;
                return false;
            } while (!this.clientState.compareAndSet(current, newState));
            logger.info("State change: {} -> {}", (Object)current, (Object)newState);
            return true;
        }

        public void changeState(ConnectivityState newState) {
            ConnectivityState prev = this.clientState.getAndSet(newState);
            logger.info("State change: {} -> {}", (Object)prev, (Object)newState);
        }
    }

    public static interface CommunicationProvider {
        public String getWebSocketUrl();

        public String getAuthToken();
    }

    public static enum ConnectivityState {
        STOPPED,
        CONNECTING,
        CONNECTED,
        DISCONNECTED,
        STOPPING,
        CLOSED;

    }
}

