package com.particles.mes.android.data.mapper

import com.google.protobuf.ByteString
import com.particles.mes.android.data.MesAdClickEvent
import com.particles.mes.android.data.MesAdHideEvent
import com.particles.mes.android.data.MesAdImpressionEvent
import com.particles.mes.android.data.MesAdReportEvent
import com.particles.mes.android.data.MesAdRequest
import com.particles.mes.android.data.MesAdRequestEvent
import com.particles.mes.android.data.MesAdRequestExt
import com.particles.mes.android.data.MesAdResponse
import com.particles.mes.android.data.MesAdResponseEvent
import com.particles.mes.android.data.MesAdType
import com.particles.mes.android.data.MesGetAdEvent
import com.particles.mes.android.data.MesLoadAdEvent
import com.particles.mes.android.data.MesSdkInitEvent
import com.particles.mes.protos.Ad
import com.particles.mes.protos.AdClickEvent
import com.particles.mes.protos.AdHideEvent
import com.particles.mes.protos.AdImpressionEvent
import com.particles.mes.protos.AdReportEvent
import com.particles.mes.protos.AdRequest
import com.particles.mes.protos.AdResponse
import com.particles.mes.protos.AdType
import com.particles.mes.protos.GetAdEvent
import com.particles.mes.protos.LoadAd
import com.particles.mes.protos.OsType
import com.particles.mes.protos.RequestContext
import com.particles.mes.protos.RequestContextExt
import com.particles.mes.protos.SdkInitEvent
import com.particles.mes.protos.openrtb.BidResponse
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid

object EventMapper {
    fun map(source: MesAdClickEvent): AdClickEvent {
        return AdClickEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdImpressionEvent): AdImpressionEvent {
        return AdImpressionEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesSdkInitEvent): SdkInitEvent {
        return SdkInitEvent.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                os = OsType.OS_TYPE_ANDROID
                org = source.org
                app = source.app
                latency = source.latency
                totalCompleteTime = source.totalCompleteTime
                putAllCompleteTimeByAdNetwork(source.completeTimeByAdNetwork)
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesLoadAdEvent): LoadAd {
        return LoadAd.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                source.adRequest?.let { requestContext = map(it) }
                source.adResponse?.let { ad = map(it) }
                os = OsType.OS_TYPE_ANDROID
                org = source.org
                app = source.app
                latency = source.latency
                errorCode = source.errorCode
                source.errorMessage?.let { errorMessage = it }
                filledFromCache = source.filledFromCache
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesGetAdEvent): GetAdEvent {
        return GetAdEvent.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                errorCode = source.errorCode
                source.errorMessage?.let { errorMessage = it }
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdRequestEvent): AdRequest {
        return AdRequest.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                requestContext = map(source.adRequest)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdResponseEvent): AdResponse {
        return AdResponse.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                requestContext = map(source.adRequest)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                source.adResponse?.let {ad = map(it) }
                latency = source.latency
                errorCode = source.errorCode
                source.errorMessage?.let { errorMessage = it }
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdHideEvent): AdHideEvent {
        return AdHideEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                reason = source.reason
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdReportEvent): AdReportEvent {
        return AdReportEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                reason = source.reason
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                source.description?.let { description = it }
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    private fun map(source: MesAdRequest): RequestContext {
        return RequestContext.newBuilder()
            .apply {
                tsMs = source.sentRequestAtMillis
                source.bidRequest?.let { bidRequest = BidRequestMapper.map(it) }
                isOpenRtbRequest = source.isOpenRtb
                ext = map(source.ext)
            }
            .build()
    }

    private fun map(source: MesAdRequestExt): RequestContextExt {
        return RequestContextExt.newBuilder()
            .apply {
                source.docId?.let { docId = it }
                source.sessionId?.let { sessionId = it }
                source.source?.let { this.source = it }
                position = source.position
                placementId = source.placementId
                source.buckets.forEach { addBuckets(it) }
                adSlotId = source.adSlotId
                userId = source.userId
            }
            .build()
    }

    private fun map(source: MesAdResponse): Ad {
        return Ad.newBuilder()
            .apply {
                tsMs = source.receivedResponseAtMillis
                type = map(source.adType)
                source.bid?.let {
                    seatBid = BidResponse.SeatBid.newBuilder()
                        .setSeat(getSeat(it))
                        .addBid(BidResponseMapper.map(it))
                        .build()
                }
                source.title?.let { title = it }
                source.body?.let { body = it }
                source.advertiser?.let { advertiser = it }
                source.adScreenshot?.let { adScreenshot = ByteString.copyFrom(it) }
                source.fullScreenshot?.let { fullScreenshot = ByteString.copyFrom(it) }
            }
            .build()
    }

    private fun getSeat(bid: Bid): String {
        return try {
            JSONObject(bid.jsonString).getJSONObject("ext")
                .getJSONObject("prebid")
                .getJSONObject("meta")
                .getString("adaptercode")
        } catch (ignored: JSONException) {
            ""
        }
    }

    private fun map(source: MesAdType): AdType {
        return when (source) {
            MesAdType.AD_TYPE_UNSPECIFIED -> AdType.AD_TYPE_UNSPECIFIED
            MesAdType.AD_TYPE_NATIVE -> AdType.AD_TYPE_NATIVE
            MesAdType.AD_TYPE_DISPLAY -> AdType.AD_TYPE_DISPLAY
            MesAdType.AD_TYPE_INTERSTITIAL -> AdType.AD_TYPE_INTERSTITIAL
            MesAdType.AD_TYPE_VIDEO -> AdType.AD_TYPE_VIDEO
        }
    }
}
