package ai.connectif.sdk.model

import ai.connectif.sdk.Utils.isValidEmail
import ai.connectif.sdk.Utils.isValidPhone
import ai.connectif.sdk.Utils.truncate
import ai.connectif.sdk.data.model.event.RegisterInternal
import ai.connectif.sdk.manager.Logger
import java.util.Date

data class Register(val email: String, val contactInfo: RegisterContactInfo? = null) {
    internal fun mapToRegisterEventInternal(): RegisterInternal {
        require(isValidEmail(this.email)) { "Invalid email ${this.email}" }
        val mobilePhone = if (this.contactInfo?.mobilePhone == null ||
            isValidPhone(this.contactInfo.mobilePhone)
        ) {
            this.contactInfo?.mobilePhone
        } else {
            Logger.e("Error occurred: '${this.contactInfo.mobilePhone}' is not a valid phone")
            ""
        }

        return RegisterInternal(
            email = this.email,
            contactInfo = RegisterContactInfo(
                name = truncate(this.contactInfo?.name, 150),
                surname = truncate(this.contactInfo?.surname, 150),
                birthdate = this.contactInfo?.birthdate,
                mobilePhone = mobilePhone,
                newsletterSubscriptionStatus = this.contactInfo?.newsletterSubscriptionStatus,
                smsSubscriptionStatus = this.contactInfo?.smsSubscriptionStatus
            )
        )
    }
}

/**
 * Represents the contact information used during contact registration.
 *
 * @property name The contact's first name (up to 150 characters).
 * @property surname The contact's surname (up to 150 characters).
 * @property birthdate The contact's birthdate.
 * @property mobilePhone The contact's mobile phone number, which must include a valid `+XX` prefix
 *   (otherwise it will be sent as an empty string).
 * @property newsletterSubscriptionStatus The contact's subscription status for newsletters (can be `subscribed` or `none`).
 * @property smsSubscriptionStatus The contact's subscription status for SMS (can be `subscribed` or `none`).
 */
data class RegisterContactInfo(
    val name: String? = null,
    val surname: String? = null,
    val birthdate: Date? = null,
    val mobilePhone: String? = null,
    val newsletterSubscriptionStatus: SubscriptionStatus? = null,
    val smsSubscriptionStatus: SubscriptionStatus? = null
)
