package ai.connectif.sdk.manager.event

import ai.connectif.sdk.data.model.event.CustomInternal
import ai.connectif.sdk.data.model.event.Event
import ai.connectif.sdk.data.model.event.LoginInternal
import ai.connectif.sdk.data.model.event.RegisterInternal
import ai.connectif.sdk.data.model.event.SubscribeNewsletterInternal
import ai.connectif.sdk.data.repository.EventRepository
import ai.connectif.sdk.data.repository.SettingRepository
import ai.connectif.sdk.data.source.model.response.SendEventResponse
import ai.connectif.sdk.manager.DeviceInfoManager
import ai.connectif.sdk.model.Cart
import ai.connectif.sdk.model.ContactInfo
import ai.connectif.sdk.model.Login
import ai.connectif.sdk.model.PageVisit
import ai.connectif.sdk.model.ProductVisit
import ai.connectif.sdk.model.Purchase
import ai.connectif.sdk.model.Register
import ai.connectif.sdk.model.Search
import ai.connectif.sdk.model.customevent.CustomEvent

internal class EventManagerImpl(
    private val eventRepository: EventRepository,
    private val settingRepository: SettingRepository,
    private val deviceInfoManager: DeviceInfoManager
) : EventManager {
    internal suspend fun <T : Event> sendEvent(
        event: T,
        sendFunction: suspend (T) -> SendEventResponse?
    ) {
        if (event is LoginInternal || event is RegisterInternal) {
            settingRepository.email = event.email
        } else {
            event.email = settingRepository.email
        }

        event.trackerId = settingRepository.trackerId
        event.deviceInfo = deviceInfoManager.getDeviceInfo()
        val result = sendFunction(event)
        handleSendEventResult(result)
    }

    fun handleSendEventResult(result: SendEventResponse?) {
        if (result?.trackerId != null) {
            settingRepository.trackerId = result.trackerId
        }
    }

    override suspend fun sendPageVisit(event: PageVisit) {
        sendEvent(event.mapToPageVisitEventInternal(), eventRepository::sendPageVisit)
    }

    override suspend fun sendSearch(event: Search) {
        sendEvent(event.mapToSearchEventInternal(), eventRepository::sendSearch)
    }

    override suspend fun sendProductVisit(event: ProductVisit) {
        sendEvent(
            event.mapToProductVisitEventInternal(),
            eventRepository::sendProductVisit
        )
    }

    override suspend fun sendLogin(event: Login) {
        sendEvent(event.mapToLoginEventInternal(), eventRepository::sendLogin)
    }

    override suspend fun sendRegister(event: Register) {
        sendEvent(event.mapToRegisterEventInternal(), eventRepository::sendRegister)
    }

    override suspend fun sendCart(event: Cart) {
        sendEvent(
            event.mapToCartInternal(),
            eventRepository::sendCart
        )
    }

    override suspend fun sendPurchase(event: Purchase) {
        sendEvent(event.mapToPurchaseInternal(), eventRepository::sendPurchase)
    }

    override suspend fun subscribeToNewsletter() {
        sendEvent(
            SubscribeNewsletterInternal(),
            eventRepository::subscribeToNewsletter
        )
    }

    override suspend fun sendContactInfo(event: ContactInfo) {
        sendEvent(event.mapToContactInfoInternal(), eventRepository::sendContactInfo)
    }

    override suspend fun sendCustomEventById(eventId: String, event: CustomEvent) {
        sendEvent(
            CustomInternal(payload = event, eventId = eventId),
            eventRepository::sendCustomEvent
        )
    }

    override suspend fun sendCustomEventById(eventId: String, payload: Any) {
        sendEvent(
            CustomInternal(payload = payload, eventId = eventId),
            eventRepository::sendCustomEvent
        )
    }

    override suspend fun sendCustomEventByAlias(eventAlias: String, event: CustomEvent) {
        sendEvent(
            CustomInternal(payload = event, eventAlias = eventAlias),
            eventRepository::sendCustomEvent

        )
    }

    override suspend fun sendCustomEventByAlias(eventAlias: String, payload: Any) {
        sendEvent(
            CustomInternal(payload = payload, eventAlias = eventAlias),
            eventRepository::sendCustomEvent

        )
    }
}
