package ai.connectif.sdk.data.source

import ai.connectif.sdk.Utils
import ai.connectif.sdk.data.error.ConnectifApiException
import ai.connectif.sdk.data.error.UnauthorizedApiException
import ai.connectif.sdk.data.error.ValidationException
import ai.connectif.sdk.data.model.event.CartInternal
import ai.connectif.sdk.data.model.event.ContactInfoInternal
import ai.connectif.sdk.data.model.event.CustomInternal
import ai.connectif.sdk.data.model.event.LoginInternal
import ai.connectif.sdk.data.model.event.PageVisitInternal
import ai.connectif.sdk.data.model.event.ProductVisitInternal
import ai.connectif.sdk.data.model.event.PurchaseInternal
import ai.connectif.sdk.data.model.event.RegisterInternal
import ai.connectif.sdk.data.model.event.SearchInternal
import ai.connectif.sdk.data.model.event.SubscribeNewsletterInternal
import ai.connectif.sdk.data.model.event.mapToApiModel
import ai.connectif.sdk.data.repository.SettingRepository
import ai.connectif.sdk.data.source.model.response.SendEventResponse
import retrofit2.HttpException

internal class EventDataSourceImpl(private val settingRepository: SettingRepository) :
    EventDataSource {

    private suspend fun <T> safeApiCall(apiCall: suspend () -> T): T? = try {
        apiCall()
    } catch (e: HttpException) {
        when (e.code()) {
            422 -> throw ValidationException.fromHttpException(e)
            401 -> throw UnauthorizedApiException("Invalid api key, please check it.", e)
            else -> throw ConnectifApiException(e.message(), e)
        }
    }

    override suspend fun sendPageVisit(event: PageVisitInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postPageVisit(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun sendSearch(event: SearchInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postSearch(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun sendProductVisit(event: ProductVisitInternal): SendEventResponse? =
        safeApiCall {
            RetrofitClient.getInstance().create(ConnectifApiService::class.java)
                .postProductVisit(
                    Utils.formatApiKeyToHeader(settingRepository.apiKey),
                    settingRepository.deviceId,
                    event.mapToApiModel()
                )
        }

    override suspend fun sendLogin(event: LoginInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postLogin(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun sendRegister(event: RegisterInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postRegister(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun sendCart(event: CartInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postCartProductAdded(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun sendPurchase(event: PurchaseInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postPurchase(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun sendContactInfo(event: ContactInfoInternal): SendEventResponse? =
        safeApiCall {
            RetrofitClient.getInstance().create(ConnectifApiService::class.java)
                .postContactInfo(
                    Utils.formatApiKeyToHeader(settingRepository.apiKey),
                    settingRepository.deviceId,
                    event.mapToApiModel()
                )
        }

    override suspend fun sendCustomEvent(event: CustomInternal): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postCustom(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }

    override suspend fun subscribeToNewsletter(
        event: SubscribeNewsletterInternal
    ): SendEventResponse? = safeApiCall {
        RetrofitClient.getInstance().create(ConnectifApiService::class.java)
            .postSubscribeToNewsletter(
                Utils.formatApiKeyToHeader(settingRepository.apiKey),
                settingRepository.deviceId,
                event.mapToApiModel()
            )
    }
}
