/*
 * Decompiled with CFR 0.152.
 */
package uk.co.lucasweb.aws.v4.signer;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import uk.co.lucasweb.aws.v4.signer.CanonicalHeaders;
import uk.co.lucasweb.aws.v4.signer.CanonicalRequest;
import uk.co.lucasweb.aws.v4.signer.Header;
import uk.co.lucasweb.aws.v4.signer.HttpRequest;
import uk.co.lucasweb.aws.v4.signer.SigningException;
import uk.co.lucasweb.aws.v4.signer.credentials.AwsCredentials;
import uk.co.lucasweb.aws.v4.signer.credentials.AwsCredentialsProviderChain;
import uk.co.lucasweb.aws.v4.signer.functional.Throwables;
import uk.co.lucasweb.aws.v4.signer.hash.Base16;
import uk.co.lucasweb.aws.v4.signer.hash.Sha256;

public class Signer {
    private static final String AUTH_TAG = "AWS4";
    private static final String ALGORITHM = "AWS4-HMAC-SHA256";
    private static final String TERMINATION_STRING = "aws4_request";
    private static final Charset UTF_8 = Throwables.returnableInstance(() -> Charset.forName("UTF-8"), SigningException::new);
    private static final String X_AMZ_DATE = "X-Amz-Date";
    private static final String HMAC_SHA256 = "HmacSHA256";
    private final CanonicalRequest request;
    private final AwsCredentials awsCredentials;
    private final String service;
    private final String region;

    private Signer(CanonicalRequest request, AwsCredentials awsCredentials, String service, String region) {
        this.request = request;
        this.awsCredentials = awsCredentials;
        this.service = service;
        this.region = region;
    }

    public String getSignature() {
        String date = this.request.getHeaders().getFirstValue(X_AMZ_DATE).orElseThrow(() -> new SigningException("headers missing 'X-Amz-Date' header"));
        String dateWithoutTimestamp = Signer.formatDateWithoutTimestamp(date);
        String credentialScope = Signer.buildCredentialScope(dateWithoutTimestamp, this.service, this.region);
        String hashedCanonicalRequest = Sha256.get(this.request.get(), UTF_8);
        String stringToSign = Signer.buildStringToSign(date, credentialScope, hashedCanonicalRequest);
        String signature = Signer.buildSignature(this.awsCredentials.getSecretKey(), dateWithoutTimestamp, stringToSign, this.service, this.region);
        return Signer.buildAuthHeader(this.awsCredentials.getAccessKey(), credentialScope, this.request.getHeaders().getNames(), signature);
    }

    public static Builder builder() {
        return new Builder();
    }

    private static String formatDateWithoutTimestamp(String date) {
        return date.substring(0, 8);
    }

    private static String buildStringToSign(String date, String credentialScope, String hashedCanonicalRequest) {
        return "AWS4-HMAC-SHA256\n" + date + "\n" + credentialScope + "\n" + hashedCanonicalRequest;
    }

    private static String buildCredentialScope(String dateWithoutTimeStamp, String service, String region) {
        return dateWithoutTimeStamp + "/" + region + "/" + service + "/" + TERMINATION_STRING;
    }

    private static String buildAuthHeader(String accessKey, String credentialScope, String signedHeaders, String signature) {
        return "AWS4-HMAC-SHA256 Credential=" + accessKey + "/" + credentialScope + ", SignedHeaders=" + signedHeaders + ", Signature=" + signature;
    }

    private static byte[] hmacSha256(byte[] key, String value) {
        try {
            String algorithm = HMAC_SHA256;
            Mac mac = Mac.getInstance(algorithm);
            SecretKeySpec signingKey = new SecretKeySpec(key, algorithm);
            mac.init(signingKey);
            return mac.doFinal(value.getBytes(UTF_8));
        }
        catch (Exception e) {
            throw new SigningException("Error signing request", e);
        }
    }

    private static String buildSignature(String secretKey, String dateWithoutTimestamp, String stringToSign, String service, String region) {
        byte[] kSecret = (AUTH_TAG + secretKey).getBytes(UTF_8);
        byte[] kDate = Signer.hmacSha256(kSecret, dateWithoutTimestamp);
        byte[] kRegion = Signer.hmacSha256(kDate, region);
        byte[] kService = Signer.hmacSha256(kRegion, service);
        byte[] kSigning = Signer.hmacSha256(kService, TERMINATION_STRING);
        return Base16.encode(Signer.hmacSha256(kSigning, stringToSign)).toLowerCase();
    }

    public static class Builder {
        private static final String DEFAULT_REGION = "us-east-1";
        private static final String S3 = "s3";
        private static final String GLACIER = "glacier";
        private AwsCredentials awsCredentials;
        private String region = "us-east-1";
        private List<Header> headersList = new ArrayList<Header>();

        public Builder awsCredentials(AwsCredentials awsCredentials) {
            this.awsCredentials = awsCredentials;
            return this;
        }

        public Builder region(String region) {
            this.region = region;
            return this;
        }

        public Builder header(String name, String value) {
            this.headersList.add(new Header(name, value));
            return this;
        }

        public Builder header(Header header) {
            this.headersList.add(header);
            return this;
        }

        public Builder headers(Header ... headers) {
            Arrays.stream(headers).forEach(this.headersList::add);
            return this;
        }

        public Signer build(HttpRequest request, String service, String contentSha256) {
            return new Signer(new CanonicalRequest(request, this.getCanonicalHeaders(), contentSha256), this.getAwsCredentials(), service, this.region);
        }

        public Signer buildS3(HttpRequest request, String contentSha256) {
            return this.build(request, S3, contentSha256);
        }

        public Signer buildGlacier(HttpRequest request, String contentSha256) {
            return this.build(request, GLACIER, contentSha256);
        }

        private AwsCredentials getAwsCredentials() {
            return Optional.ofNullable(this.awsCredentials).orElseGet(() -> new AwsCredentialsProviderChain().getCredentials());
        }

        private CanonicalHeaders getCanonicalHeaders() {
            CanonicalHeaders.Builder builder = CanonicalHeaders.builder();
            this.headersList.forEach(h -> builder.add(h.getName(), h.getValue()));
            return builder.build();
        }
    }
}

