/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.utils;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.regex.Pattern;
import software.amazon.smithy.utils.CodeWriter;
import software.amazon.smithy.utils.SetUtils;
import software.amazon.smithy.utils.StringUtils;

final class CodeFormatter {
    private static final Pattern NAME_PATTERN = Pattern.compile("^[a-z]+[a-zA-Z0-9_.#$]*$");
    private static final Set<Character> VALID_FORMATTER_CHARS = SetUtils.of(Character.valueOf('!'), Character.valueOf('#'), Character.valueOf('%'), Character.valueOf('&'), Character.valueOf('\''), Character.valueOf('('), Character.valueOf(')'), Character.valueOf('*'), Character.valueOf('+'), Character.valueOf(','), Character.valueOf('-'), Character.valueOf('.'), Character.valueOf('/'), Character.valueOf(':'), Character.valueOf(';'), Character.valueOf('<'), Character.valueOf('='), Character.valueOf('>'), Character.valueOf('?'), Character.valueOf('@'), Character.valueOf('A'), Character.valueOf('B'), Character.valueOf('C'), Character.valueOf('D'), Character.valueOf('E'), Character.valueOf('F'), Character.valueOf('G'), Character.valueOf('H'), Character.valueOf('I'), Character.valueOf('J'), Character.valueOf('K'), Character.valueOf('L'), Character.valueOf('M'), Character.valueOf('N'), Character.valueOf('O'), Character.valueOf('P'), Character.valueOf('Q'), Character.valueOf('R'), Character.valueOf('S'), Character.valueOf('T'), Character.valueOf('U'), Character.valueOf('V'), Character.valueOf('W'), Character.valueOf('X'), Character.valueOf('Y'), Character.valueOf('Z'), Character.valueOf('['), Character.valueOf(']'), Character.valueOf('^'), Character.valueOf('_'), Character.valueOf('`'), Character.valueOf('{'), Character.valueOf('|'), Character.valueOf('}'), Character.valueOf('~'));
    private final Map<Character, BiFunction<Object, String, String>> formatters = new HashMap<Character, BiFunction<Object, String, String>>();
    private final CodeFormatter parentFormatter;

    CodeFormatter() {
        this(null);
    }

    CodeFormatter(CodeFormatter parentFormatter) {
        this.parentFormatter = parentFormatter;
    }

    void putFormatter(Character identifier, BiFunction<Object, String, String> formatFunction) {
        if (!VALID_FORMATTER_CHARS.contains(identifier)) {
            throw new IllegalArgumentException("Invalid formatter identifier: " + identifier);
        }
        this.formatters.put(identifier, formatFunction);
    }

    BiFunction<Object, String, String> getFormatter(char identifier) {
        BiFunction<Object, String, String> result = this.formatters.get(Character.valueOf(identifier));
        if (result == null && this.parentFormatter != null) {
            result = this.parentFormatter.getFormatter(identifier);
        }
        return result;
    }

    String format(char expressionStart, Object content, String indent, CodeWriter writer, Object ... args) {
        String expression = String.valueOf(content);
        if (args.length == 0 && expression.indexOf(expressionStart) == -1) {
            return expression;
        }
        return this.parse(expressionStart, new State(content, indent, writer, args));
    }

    private String parse(char expressionStart, State state) {
        while (!state.eof()) {
            char c = state.c();
            state.next();
            if (c == expressionStart) {
                this.parseArgumentWrapper(expressionStart, state);
                continue;
            }
            state.append(c);
        }
        if (state.relativeIndex == -1) {
            CodeFormatter.ensureAllPositionalArgumentsWereUsed(state.expression, state.positionals);
        } else if (state.relativeIndex < state.args.length) {
            throw new IllegalArgumentException(String.format("Found %d unused relative format arguments: %s", state.args.length - state.relativeIndex, state.expression));
        }
        return state.result.toString();
    }

    private void parseArgumentWrapper(char expressionStart, State state) {
        if (state.eof()) {
            throw new IllegalArgumentException("Invalid format string: " + state);
        }
        char c = state.c();
        if (c == expressionStart) {
            state.append(expressionStart);
            state.next();
        } else if (c == '{') {
            this.parseBracedArgument(state);
        } else {
            this.parseArgument(state, -1);
        }
    }

    private void parseBracedArgument(State state) {
        int startingBraceColumn = state.column;
        state.next();
        this.parseArgument(state, startingBraceColumn);
        if (state.eof() || state.c() != '}') {
            throw new IllegalArgumentException("Unclosed expression argument: " + state);
        }
        state.next();
    }

    private void parseArgument(State state, int startingBraceColumn) {
        if (state.eof()) {
            throw new IllegalArgumentException("Invalid format string: " + state);
        }
        char c = state.c();
        if (Character.isLowerCase(c)) {
            this.parseNamedArgument(state, startingBraceColumn);
        } else if (Character.isDigit(c)) {
            this.parsePositionalArgument(state, startingBraceColumn);
        } else {
            this.parseRelativeArgument(state, startingBraceColumn);
        }
    }

    private void parseNamedArgument(State state, int startingBraceColumn) {
        String name = this.parseNameUntil(state, ':');
        state.next();
        if (state.eof()) {
            throw new IllegalArgumentException("Expected an identifier after the ':' in a named argument: " + state);
        }
        char identifier = this.consumeFormatterIdentifier(state);
        state.append(this.applyFormatter(state, identifier, state.writer.getContext(name), startingBraceColumn));
    }

    private char consumeFormatterIdentifier(State state) {
        char identifier = state.c();
        state.next();
        return identifier;
    }

    private void parsePositionalArgument(State state, int startingBraceColumn) {
        CodeFormatter.expectConsistentRelativePositionals(state, state.relativeIndex <= 0);
        state.relativeIndex = -1;
        int startPosition = state.position;
        while (state.next() && Character.isDigit(state.c())) {
        }
        int index = Integer.parseInt(state.expression.substring(startPosition, state.position)) - 1;
        if (index < 0 || index >= state.args.length) {
            throw new IllegalArgumentException(String.format("Positional argument index %d out of range of provided %d arguments in format string: %s", index, state.args.length, state));
        }
        Object arg = this.getPositionalArgument(state.expression, index, state.args);
        state.positionals[index] = true;
        char identifier = this.consumeFormatterIdentifier(state);
        state.append(this.applyFormatter(state, identifier, arg, startingBraceColumn));
    }

    private void parseRelativeArgument(State state, int startingBraceColumn) {
        CodeFormatter.expectConsistentRelativePositionals(state, state.relativeIndex > -1);
        ++state.relativeIndex;
        Object argument = this.getPositionalArgument(state.expression, state.relativeIndex - 1, state.args);
        char identifier = this.consumeFormatterIdentifier(state);
        state.append(this.applyFormatter(state, identifier, argument, startingBraceColumn));
    }

    private String parseNameUntil(State state, char endToken) {
        int endIndex = state.expression.indexOf(endToken, state.position);
        if (endIndex == -1) {
            throw new IllegalArgumentException("Invalid named format argument: " + state);
        }
        String name = state.expression.substring(state.position, endIndex);
        CodeFormatter.ensureNameIsValid(state, name);
        state.position = endIndex;
        return name;
    }

    private static void expectConsistentRelativePositionals(State state, boolean expectation) {
        if (!expectation) {
            throw new IllegalArgumentException("Cannot mix positional and relative arguments: " + state);
        }
    }

    private static void ensureAllPositionalArgumentsWereUsed(String expression, boolean[] positionals) {
        int unused = 0;
        for (boolean b : positionals) {
            if (b) continue;
            ++unused;
        }
        if (unused > 0) {
            throw new IllegalArgumentException(String.format("Found %d unused positional format arguments: %s", unused, expression));
        }
    }

    private Object getPositionalArgument(String content, int index, Object[] args) {
        if (index >= args.length) {
            throw new IllegalArgumentException(String.format("Given %d arguments but attempted to format index %d: %s", args.length, index, content));
        }
        return args[index];
    }

    private String applyFormatter(State state, char formatter, Object argument, int startingBraceColumn) {
        BiFunction<Object, String, String> formatFunction = this.getFormatter(formatter);
        if (formatFunction == null) {
            throw new IllegalArgumentException(String.format("Unknown formatter `%s` found in format string: %s", Character.valueOf(formatter), state));
        }
        String result = formatFunction.apply(argument, state.indent);
        if (!state.eof() && state.c() == '@') {
            if (startingBraceColumn == -1) {
                throw new IllegalArgumentException("Inline blocks can only be created inside braces: " + state);
            }
            result = this.expandInlineSection(state, result);
        }
        if (startingBraceColumn != -1 && !state.eof() && state.c() == '|') {
            state.next();
            String repeated = StringUtils.repeat(' ', startingBraceColumn);
            StringBuilder aligned = new StringBuilder();
            for (int i = 0; i < result.length(); ++i) {
                char c = result.charAt(i);
                if (c == '\n') {
                    aligned.append('\n').append(repeated);
                    continue;
                }
                if (c == '\r') {
                    aligned.append('\r');
                    if (i + 1 < result.length() && result.charAt(i + 1) == '\n') {
                        aligned.append('\n');
                        ++i;
                    }
                    aligned.append(repeated);
                    continue;
                }
                aligned.append(c);
            }
            result = aligned.toString();
        }
        return result;
    }

    private String expandInlineSection(State state, String argument) {
        state.next();
        String sectionName = this.parseNameUntil(state, '}');
        CodeFormatter.ensureNameIsValid(state, sectionName);
        return state.writer.expandSection(sectionName, argument, s -> state.writer.write(s, new Object[0]));
    }

    private static void ensureNameIsValid(State state, String name) {
        if (!NAME_PATTERN.matcher(name).matches()) {
            throw new IllegalArgumentException(String.format("Invalid format expression name `%s` at position %d of: %s", name, state.position + 1, state));
        }
    }

    private static final class State {
        StringBuilder result = new StringBuilder();
        int position = 0;
        int relativeIndex = 0;
        CodeWriter writer;
        String expression;
        String indent;
        Object[] args;
        boolean[] positionals;
        int column = 0;

        State(Object expression, String indent, CodeWriter writer, Object[] args) {
            this.expression = String.valueOf(expression);
            this.indent = indent;
            this.writer = writer;
            this.args = args;
            this.positionals = new boolean[args.length];
        }

        char c() {
            return this.expression.charAt(this.position);
        }

        boolean eof() {
            return this.position >= this.expression.length();
        }

        boolean next() {
            return ++this.position < this.expression.length() - 1;
        }

        void append(char c) {
            this.column = c == '\r' || c == '\n' ? 0 : ++this.column;
            this.result.append(c);
        }

        void append(String string) {
            int lastNewline = string.lastIndexOf(10);
            if (lastNewline == -1) {
                lastNewline = string.lastIndexOf(13);
            }
            this.column = lastNewline == -1 ? string.length() : string.length() - lastNewline;
            this.result.append(string);
        }

        public String toString() {
            return this.expression;
        }
    }
}

