/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.node;

import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URI;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.IdentityClassCache;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.NodeMapper;
import software.amazon.smithy.model.node.NodeSerializationException;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.node.ToNode;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.StringUtils;

final class DefaultNodeSerializers {
    private static final Logger LOGGER = Logger.getLogger(DefaultNodeSerializers.class.getName());
    private static final NodeMapper.Serializer<ToNode> TO_NODE_SERIALIZER = new NodeMapper.Serializer<ToNode>(){

        @Override
        public Class<ToNode> getType() {
            return ToNode.class;
        }

        @Override
        public Node serialize(ToNode value, Set<Object> serializedObjects, NodeMapper mapper) {
            if (mapper.getDisableToNode().contains(value.getClass())) {
                return null;
            }
            return value.toNode();
        }
    };
    private static final NodeMapper.Serializer<Optional> OPTIONAL_SERIALIZER = new NodeMapper.Serializer<Optional>(){

        @Override
        public Class<Optional> getType() {
            return Optional.class;
        }

        @Override
        public Node serialize(Optional value, Set<Object> serializedObjects, NodeMapper mapper) {
            return (Node)value.map(v -> mapper.serialize(v, serializedObjects)).orElse(Node.nullNode());
        }
    };
    private static final NodeMapper.Serializer<Number> NUMBER_SERIALIZER = new NodeMapper.Serializer<Number>(){

        @Override
        public Class<Number> getType() {
            return Number.class;
        }

        @Override
        public Node serialize(Number value, Set<Object> serializedObjects, NodeMapper mapper) {
            return Node.from(value);
        }
    };
    private static final NodeMapper.Serializer<String> STRING_SERIALIZER = new NodeMapper.Serializer<String>(){

        @Override
        public Class<String> getType() {
            return String.class;
        }

        @Override
        public Node serialize(String value, Set<Object> serializedObjects, NodeMapper mapper) {
            return Node.from(value);
        }
    };
    private static final NodeMapper.Serializer<File> FILE_SERIALIZER = new NodeMapper.Serializer<File>(){

        @Override
        public Class<File> getType() {
            return File.class;
        }

        @Override
        public Node serialize(File value, Set<Object> serializedObjects, NodeMapper mapper) {
            return Node.from(value.getAbsolutePath());
        }
    };
    private static final NodeMapper.Serializer<Path> PATH_SERIALIZER = new NodeMapper.Serializer<Path>(){

        @Override
        public Class<Path> getType() {
            return Path.class;
        }

        @Override
        public Node serialize(Path value, Set<Object> serializedObjects, NodeMapper mapper) {
            return Node.from(value.toUri().toString());
        }
    };
    private static final NodeMapper.Serializer<ShapeId> SHAPE_ID_SERIALIZER = new ToStringSerializer<ShapeId>(ShapeId.class);
    private static final NodeMapper.Serializer<Enum> ENUM_SERIALIZER = new ToStringSerializer<Enum>(Enum.class);
    private static final NodeMapper.Serializer<URL> URL_SERIALIZER = new ToStringSerializer<URL>(URL.class);
    private static final NodeMapper.Serializer<URI> URI_SERIALIZER = new ToStringSerializer<URI>(URI.class);
    private static final NodeMapper.Serializer<Pattern> PATTERN_SERIALIZER = new ToStringSerializer<Pattern>(Pattern.class);
    private static final NodeMapper.Serializer<Boolean> BOOLEAN_SERIALIZER = new NodeMapper.Serializer<Boolean>(){

        @Override
        public Class<Boolean> getType() {
            return Boolean.class;
        }

        @Override
        public Node serialize(Boolean value, Set<Object> serializedObjects, NodeMapper mapper) {
            return Node.from(value);
        }
    };
    private static final NodeMapper.Serializer<Map> MAP_SERIALIZER = new NodeMapper.Serializer<Map>(){

        @Override
        public Class<Map> getType() {
            return Map.class;
        }

        @Override
        public Node serialize(Map value, Set<Object> serializedObjects, NodeMapper mapper) {
            LinkedHashMap<StringNode, Node> mappings = new LinkedHashMap<StringNode, Node>();
            Set entries = value.entrySet();
            for (Map.Entry entry : entries) {
                Node key = mapper.serialize(entry.getKey(), serializedObjects);
                if (key instanceof StringNode) {
                    mappings.put((StringNode)key, mapper.serialize(entry.getValue(), serializedObjects));
                    continue;
                }
                throw new NodeSerializationException("Unable to write Map key because it was not serialized as a string: " + entry.getKey() + " -> " + Node.printJson(key));
            }
            return new ObjectNode(mappings, SourceLocation.NONE);
        }
    };
    private static final NodeMapper.Serializer<Iterable> ITERABLE_SERIALIZER = new NodeMapper.Serializer<Iterable>(){

        @Override
        public Class<Iterable> getType() {
            return Iterable.class;
        }

        @Override
        public Node serialize(Iterable value, Set<Object> serializedObjects, NodeMapper mapper) {
            ArrayList<Node> nodes = new ArrayList<Node>();
            for (Object item : value) {
                nodes.add(mapper.serialize(item, serializedObjects));
            }
            return new ArrayNode(nodes, SourceLocation.NONE);
        }
    };
    private static final NodeMapper.Serializer<Object[]> ARRAY_SERIALIZER = new NodeMapper.Serializer<Object[]>(){

        @Override
        public Class<Object[]> getType() {
            return Object[].class;
        }

        @Override
        public Node serialize(Object[] value, Set<Object> serializedObjects, NodeMapper mapper) {
            ArrayList<Node> nodes = new ArrayList<Node>();
            for (Object item : value) {
                nodes.add(mapper.serialize(item, serializedObjects));
            }
            return new ArrayNode(nodes, SourceLocation.NONE);
        }
    };
    static final NodeMapper.Serializer<Object> FROM_BEAN = new NodeMapper.Serializer<Object>(){

        @Override
        public Class<Object> getType() {
            return Object.class;
        }

        @Override
        public Node serialize(Object value, Set<Object> serializedObjects, NodeMapper mapper) {
            if (serializedObjects.contains(value)) {
                return Node.nullNode();
            }
            serializedObjects.add(value);
            TreeMap<StringNode, Node> mappings = new TreeMap<StringNode, Node>(Comparator.comparing(StringNode::getValue));
            ClassInfo info = ClassInfo.fromClass(value.getClass());
            for (Map.Entry<String, Method> entry : info.getters.entrySet()) {
                try {
                    Object getterResult = entry.getValue().invoke(value, new Object[0]);
                    Node result = mapper.serialize(getterResult, serializedObjects);
                    if (!this.canSerialize(mapper, result)) continue;
                    mappings.put(Node.from(entry.getKey()), result);
                }
                catch (ReflectiveOperationException e) {
                    String causeMessage = e.getCause() != null ? e.getCause().getMessage() : e.getMessage();
                    String message = String.format("Error serializing `%s` field of %s using %s(): %s", entry.getKey(), value.getClass().getName(), entry.getValue().getName(), causeMessage);
                    throw new NodeSerializationException(message, e);
                }
            }
            serializedObjects.remove(value);
            SourceLocation sourceLocation = SourceLocation.NONE;
            if (value instanceof FromSourceLocation) {
                sourceLocation = ((FromSourceLocation)value).getSourceLocation();
            }
            return new ObjectNode(mappings, sourceLocation);
        }

        private boolean canSerialize(NodeMapper mapper, Node value) {
            if (!mapper.getSerializeNullValues() && value.isNullNode()) {
                return false;
            }
            if (mapper.getOmitEmptyValues()) {
                if (value.isObjectNode() && value.expectObjectNode().isEmpty()) {
                    return false;
                }
                if (value.isArrayNode() && value.expectArrayNode().isEmpty()) {
                    return false;
                }
                if (value.isBooleanNode() && !value.expectBooleanNode().getValue()) {
                    return false;
                }
            }
            return true;
        }
    };
    static final List<NodeMapper.Serializer> SERIALIZERS = ListUtils.of((Object[])new NodeMapper.Serializer[]{TO_NODE_SERIALIZER, OPTIONAL_SERIALIZER, STRING_SERIALIZER, BOOLEAN_SERIALIZER, NUMBER_SERIALIZER, MAP_SERIALIZER, ARRAY_SERIALIZER, SHAPE_ID_SERIALIZER, ENUM_SERIALIZER, URL_SERIALIZER, URI_SERIALIZER, PATTERN_SERIALIZER, PATH_SERIALIZER, FILE_SERIALIZER, ITERABLE_SERIALIZER});

    private DefaultNodeSerializers() {
    }

    private static final class ClassInfo {
        private static final IdentityClassCache<Class, ClassInfo> CACHE = new IdentityClassCache();
        final Map<String, Method> getters = new TreeMap<String, Method>();

        private ClassInfo() {
        }

        static ClassInfo fromClass(Class<?> klass) {
            return CACHE.getForClass(klass, klass, () -> {
                ClassInfo info = new ClassInfo();
                Set<String> transientFields = ClassInfo.getTransientFields(klass);
                for (Method method : klass.getMethods()) {
                    int fieldPrefixChars;
                    if (ClassInfo.isIgnoredMethod(klass, method) || (fieldPrefixChars = ClassInfo.getGetterPrefixCharCount(method)) <= 0 || fieldPrefixChars == method.getName().length()) continue;
                    String lowerFieldName = StringUtils.uncapitalize((String)method.getName().substring(fieldPrefixChars));
                    if (!transientFields.contains(lowerFieldName)) {
                        info.getters.put(lowerFieldName, method);
                        continue;
                    }
                    LOGGER.fine(klass.getName() + " getter " + method.getName() + " is transient");
                }
                LOGGER.fine(() -> "Detected the following getters for " + klass.getName() + ": " + info.getters);
                return info;
            });
        }

        private static boolean isIgnoredMethod(Class<?> klass, Method method) {
            if (method.getDeclaringClass() == Object.class) {
                return true;
            }
            return FromSourceLocation.class.isAssignableFrom(klass) && method.getName().equals("getSourceLocation");
        }

        private static Set<String> getTransientFields(Class klass) {
            HashSet<String> transientFields = new HashSet<String>();
            for (Field field : klass.getDeclaredFields()) {
                if (!Modifier.isTransient(field.getModifiers())) continue;
                transientFields.add(StringUtils.uncapitalize((String)field.getName()));
            }
            return transientFields;
        }

        private static int getGetterPrefixCharCount(Method method) {
            if (!Modifier.isStatic(method.getModifiers()) && method.getParameterCount() == 0) {
                if (method.getName().startsWith("get")) {
                    return 3;
                }
                if (method.getName().startsWith("is") && method.getReturnType() == Boolean.TYPE) {
                    return 2;
                }
            }
            return 0;
        }
    }

    private static final class ToStringSerializer<T>
    implements NodeMapper.Serializer<T> {
        private Class<T> type;

        ToStringSerializer(Class<T> type) {
            this.type = type;
        }

        @Override
        public Class<T> getType() {
            return this.type;
        }

        @Override
        public Node serialize(T value, Set<Object> serializedObjects, NodeMapper mapper) {
            return Node.from(value.toString());
        }
    }
}

