/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.logging.Logger;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.AstModelLoader;
import software.amazon.smithy.model.loader.IdlModelParser;
import software.amazon.smithy.model.loader.LoadOperation;
import software.amazon.smithy.model.loader.ModelDiscovery;
import software.amazon.smithy.model.loader.ModelImportException;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.Version;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.utils.IoUtils;

final class ModelLoader {
    private static final Logger LOGGER = Logger.getLogger(ModelLoader.class.getName());

    private ModelLoader() {
    }

    static void load(TraitFactory traitFactory, Map<String, Object> properties, String filename, Consumer<LoadOperation> operationConsumer, Supplier<InputStream> contentSupplier) {
        try (InputStream inputStream = contentSupplier.get();){
            if (filename.endsWith(".smithy")) {
                String contents = IoUtils.toUtf8String((InputStream)inputStream);
                new IdlModelParser(filename, contents).parse(operationConsumer);
            } else if (filename.endsWith(".jar")) {
                ModelLoader.loadJar(traitFactory, properties, filename, operationConsumer);
            } else if (filename.endsWith(".json") || filename.equals(SourceLocation.NONE.getFilename())) {
                ModelLoader.loadParsedNode(Node.parse(inputStream, filename), operationConsumer);
            } else {
                LOGGER.warning(() -> "No ModelLoader was able to load " + filename);
            }
        }
        catch (IOException e) {
            throw new ModelImportException("Error loading " + filename + ": " + e.getMessage(), e);
        }
    }

    static void loadParsedNode(Node node, Consumer<LoadOperation> operationConsumer) {
        ObjectNode model = node.expectObjectNode("Smithy documents must be an object. Found {type}.");
        StringNode versionNode = model.expectStringMember("smithy");
        Version version = Version.fromString(versionNode.getValue());
        if (version == null) {
            throw new ModelSyntaxException("Unsupported Smithy version number: " + versionNode.getValue(), versionNode);
        }
        new AstModelLoader(version, model).parse(operationConsumer);
    }

    private static void loadJar(TraitFactory traitFactory, Map<String, Object> properties, String filename, Consumer<LoadOperation> operationConsumer) {
        URL manifestUrl = ModelDiscovery.createSmithyJarManifestUrl(filename);
        LOGGER.fine(() -> "Loading Smithy model imports from JAR: " + manifestUrl);
        for (URL model : ModelDiscovery.findModels(manifestUrl)) {
            try {
                URLConnection connection = model.openConnection();
                if (properties.containsKey("assembler.disableJarCache")) {
                    connection.setUseCaches(false);
                }
                ModelLoader.load(traitFactory, properties, model.toExternalForm(), operationConsumer, () -> {
                    try {
                        return connection.getInputStream();
                    }
                    catch (IOException e) {
                        throw ModelLoader.throwIoJarException(model, e);
                    }
                });
            }
            catch (IOException e) {
                throw ModelLoader.throwIoJarException(model, e);
            }
        }
    }

    private static ModelImportException throwIoJarException(URL model, Throwable e) {
        return new ModelImportException(String.format("Error loading Smithy model from URL `%s`: %s", model, e.getMessage()), e);
    }
}

