/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.validation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.knowledge.BoxIndex;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.NodeType;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeVisitor;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.node.NodeValidatorPlugin;
import software.amazon.smithy.model.validation.node.TimestampValidationStrategy;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.SmithyBuilder;

public final class NodeValidationVisitor
implements ShapeVisitor<List<ValidationEvent>> {
    private static final List<NodeValidatorPlugin> BUILTIN = NodeValidatorPlugin.getBuiltins();
    private final String eventId;
    private final Node value;
    private final Model model;
    private final String context;
    private final ShapeId eventShapeId;
    private final TimestampValidationStrategy timestampValidationStrategy;
    private final boolean allowBoxedNull;

    private NodeValidationVisitor(Builder builder) {
        this.value = (Node)SmithyBuilder.requiredState((String)"value", (Object)builder.value);
        this.model = (Model)SmithyBuilder.requiredState((String)"model", (Object)builder.model);
        this.context = builder.context;
        this.eventId = builder.eventId;
        this.eventShapeId = builder.eventShapeId;
        this.timestampValidationStrategy = builder.timestampValidationStrategy;
        this.allowBoxedNull = builder.allowBoxedNull;
    }

    public static Builder builder() {
        return new Builder();
    }

    private NodeValidationVisitor withNode(String segment, Node node) {
        Builder builder = NodeValidationVisitor.builder();
        builder.eventShapeId(this.eventShapeId);
        builder.eventId(this.eventId);
        builder.value(node);
        builder.model(this.model);
        builder.startingContext(this.context.isEmpty() ? segment : this.context + "." + segment);
        builder.timestampValidationStrategy(this.timestampValidationStrategy);
        builder.allowBoxedNull(this.allowBoxedNull);
        return new NodeValidationVisitor(builder);
    }

    @Override
    public List<ValidationEvent> blobShape(BlobShape shape) {
        return this.value.asStringNode().map(stringNode -> this.applyPlugins(shape)).orElseGet(() -> this.invalidShape(shape, NodeType.STRING));
    }

    @Override
    public List<ValidationEvent> booleanShape(BooleanShape shape) {
        return this.value.isBooleanNode() ? this.applyPlugins(shape) : this.invalidShape(shape, NodeType.BOOLEAN);
    }

    @Override
    public List<ValidationEvent> byteShape(ByteShape shape) {
        return this.validateNaturalNumber(shape, -128L, 127L);
    }

    @Override
    public List<ValidationEvent> shortShape(ShortShape shape) {
        return this.validateNaturalNumber(shape, -32768L, 32767L);
    }

    @Override
    public List<ValidationEvent> integerShape(IntegerShape shape) {
        return this.validateNaturalNumber(shape, (Long)Integer.MIN_VALUE, (Long)Integer.MAX_VALUE);
    }

    @Override
    public List<ValidationEvent> longShape(LongShape shape) {
        return this.validateNaturalNumber(shape, Long.MIN_VALUE, Long.MAX_VALUE);
    }

    @Override
    public List<ValidationEvent> bigIntegerShape(BigIntegerShape shape) {
        return this.validateNaturalNumber(shape, null, null);
    }

    private List<ValidationEvent> validateNaturalNumber(Shape shape, Long min, Long max) {
        return this.value.asNumberNode().map(number -> {
            if (!number.isNaturalNumber()) {
                return ListUtils.of((Object)this.event(String.format("%s shapes must not have floating point values, but found `%s` provided for `%s`", new Object[]{shape.getType(), number.getValue(), shape.getId()})));
            }
            Long numberValue = number.getValue().longValue();
            if (min != null && numberValue < min) {
                return ListUtils.of((Object)this.event(String.format("%s value must be > %d, but found %d", new Object[]{shape.getType(), min, numberValue})));
            }
            if (max != null && numberValue > max) {
                return ListUtils.of((Object)this.event(String.format("%s value must be < %d, but found %d", new Object[]{shape.getType(), max, numberValue})));
            }
            return this.applyPlugins(shape);
        }).orElseGet(() -> this.invalidShape(shape, NodeType.NUMBER));
    }

    @Override
    public List<ValidationEvent> floatShape(FloatShape shape) {
        return this.value.isNumberNode() ? this.applyPlugins(shape) : this.invalidShape(shape, NodeType.NUMBER);
    }

    @Override
    public List<ValidationEvent> documentShape(DocumentShape shape) {
        return Collections.emptyList();
    }

    @Override
    public List<ValidationEvent> doubleShape(DoubleShape shape) {
        return this.value.isNumberNode() ? this.applyPlugins(shape) : this.invalidShape(shape, NodeType.NUMBER);
    }

    @Override
    public List<ValidationEvent> bigDecimalShape(BigDecimalShape shape) {
        return this.value.isNumberNode() ? this.applyPlugins(shape) : this.invalidShape(shape, NodeType.NUMBER);
    }

    @Override
    public List<ValidationEvent> stringShape(StringShape shape) {
        return this.value.asStringNode().map(string -> this.applyPlugins(shape)).orElseGet(() -> this.invalidShape(shape, NodeType.STRING));
    }

    @Override
    public List<ValidationEvent> timestampShape(TimestampShape shape) {
        return this.applyPlugins(shape);
    }

    @Override
    public List<ValidationEvent> listShape(ListShape shape) {
        return this.processCollection(shape, shape.getMember());
    }

    @Override
    public List<ValidationEvent> setShape(SetShape shape) {
        return this.processCollection(shape, shape.getMember());
    }

    private List<ValidationEvent> processCollection(CollectionShape shape, MemberShape member) {
        return this.value.asArrayNode().map(array -> {
            List<ValidationEvent> events = this.applyPlugins(shape);
            for (int i = 0; i < array.getElements().size(); ++i) {
                events.addAll((Collection<ValidationEvent>)member.accept(this.withNode(String.valueOf(i), array.getElements().get(i))));
            }
            return events;
        }).orElseGet(() -> this.invalidShape(shape, NodeType.ARRAY));
    }

    @Override
    public List<ValidationEvent> mapShape(MapShape shape) {
        return this.value.asObjectNode().map(object -> {
            List<ValidationEvent> events = this.applyPlugins(shape);
            for (Map.Entry<StringNode, Node> entry : object.getMembers().entrySet()) {
                String key = entry.getKey().getValue();
                events.addAll((Collection<ValidationEvent>)this.withNode(key + " (map-key)", entry.getKey()).memberShape(shape.getKey()));
                events.addAll((Collection<ValidationEvent>)this.withNode(key, entry.getValue()).memberShape(shape.getValue()));
            }
            return events;
        }).orElseGet(() -> this.invalidShape(shape, NodeType.OBJECT));
    }

    @Override
    public List<ValidationEvent> structureShape(StructureShape shape) {
        return this.value.asObjectNode().map(object -> {
            List<ValidationEvent> events = this.applyPlugins(shape);
            Map members = shape.getAllMembers();
            object.getMembers().forEach((keyNode, value) -> {
                String key = keyNode.getValue();
                if (!members.containsKey(key)) {
                    String message = String.format("Invalid structure member `%s` found for `%s`", key, shape.getId());
                    events.add(this.event(message, Severity.WARNING));
                } else {
                    events.addAll((Collection<ValidationEvent>)this.withNode(key, (Node)value).memberShape((MemberShape)members.get(key)));
                }
            });
            members.forEach((memberName, member) -> {
                if (member.isRequired() && !object.getMember((String)memberName).isPresent() && !this.isMemberPrimitive((MemberShape)member)) {
                    events.add(this.event(String.format("Missing required structure member `%s` for `%s`", memberName, shape.getId())));
                }
            });
            return events;
        }).orElseGet(() -> this.invalidShape(shape, NodeType.OBJECT));
    }

    private boolean isMemberPrimitive(MemberShape member) {
        return !BoxIndex.of(this.model).isBoxed(member);
    }

    @Override
    public List<ValidationEvent> unionShape(UnionShape shape) {
        return this.value.asObjectNode().map(object -> {
            List<ValidationEvent> events = this.applyPlugins(shape);
            if (object.size() > 1) {
                events.add(this.event("union values can contain a value for only a single member"));
            } else {
                Map members = shape.getAllMembers();
                object.getMembers().forEach((keyNode, value) -> {
                    String key = keyNode.getValue();
                    if (!members.containsKey(key)) {
                        events.add(this.event(String.format("Invalid union member `%s` found for `%s`", key, shape.getId())));
                    } else {
                        events.addAll((Collection<ValidationEvent>)this.withNode(key, (Node)value).memberShape((MemberShape)members.get(key)));
                    }
                });
            }
            return events;
        }).orElseGet(() -> this.invalidShape(shape, NodeType.OBJECT));
    }

    @Override
    public List<ValidationEvent> memberShape(MemberShape shape) {
        List<ValidationEvent> events = this.applyPlugins(shape);
        events.addAll(this.model.getShape(shape.getTarget()).map(member -> member.accept(this)).orElse(ListUtils.of()));
        return events;
    }

    @Override
    public List<ValidationEvent> operationShape(OperationShape shape) {
        return this.invalidSchema(shape);
    }

    @Override
    public List<ValidationEvent> resourceShape(ResourceShape shape) {
        return this.invalidSchema(shape);
    }

    @Override
    public List<ValidationEvent> serviceShape(ServiceShape shape) {
        return this.invalidSchema(shape);
    }

    private List<ValidationEvent> invalidShape(Shape shape, NodeType expectedType) {
        if (this.allowBoxedNull && this.value.isNullNode() && BoxIndex.of(this.model).isBoxed(shape)) {
            return Collections.emptyList();
        }
        String message = String.format("Expected %s value for %s shape, `%s`; found %s value", new Object[]{expectedType, shape.getType(), shape.getId(), this.value.getType()});
        if (this.value.isStringNode()) {
            message = message + ", `" + this.value.expectStringNode().getValue() + "`";
        } else if (this.value.isNumberNode()) {
            message = message + ", `" + this.value.expectNumberNode().getValue() + "`";
        } else if (this.value.isBooleanNode()) {
            message = message + ", `" + this.value.expectBooleanNode().getValue() + "`";
        }
        return ListUtils.of((Object)this.event(message));
    }

    private List<ValidationEvent> invalidSchema(Shape shape) {
        return ListUtils.of((Object)this.event("Encountered invalid shape type: " + (Object)((Object)shape.getType())));
    }

    private ValidationEvent event(String message) {
        return this.event(message, Severity.ERROR);
    }

    private ValidationEvent event(String message, Severity severity) {
        return this.event(message, severity, this.value.getSourceLocation());
    }

    private ValidationEvent event(String message, Severity severity, SourceLocation sourceLocation) {
        return ValidationEvent.builder().id(this.eventId).severity(severity).sourceLocation(sourceLocation).shapeId(this.eventShapeId).message(this.context.isEmpty() ? message : this.context + ": " + message).build();
    }

    private List<ValidationEvent> applyPlugins(Shape shape) {
        ArrayList<ValidationEvent> events = new ArrayList<ValidationEvent>();
        this.timestampValidationStrategy.apply(shape, this.value, this.model, (location, message) -> events.add(this.event((String)message, Severity.ERROR, location.getSourceLocation())));
        for (NodeValidatorPlugin plugin : BUILTIN) {
            plugin.apply(shape, this.value, this.model, (location, message) -> events.add(this.event((String)message, Severity.ERROR, location.getSourceLocation())));
        }
        return events;
    }

    public static final class Builder
    implements SmithyBuilder<NodeValidationVisitor> {
        private String eventId = "Model";
        private String context = "";
        private ShapeId eventShapeId;
        private Node value;
        private Model model;
        private TimestampValidationStrategy timestampValidationStrategy = TimestampValidationStrategy.FORMAT;
        private boolean allowBoxedNull;

        Builder() {
        }

        public Builder model(Model model) {
            this.model = model;
            return this;
        }

        public Builder value(Node value) {
            this.value = Objects.requireNonNull(value);
            return this;
        }

        public Builder eventId(String id) {
            this.eventId = Objects.requireNonNull(id);
            return this;
        }

        public Builder startingContext(String context) {
            this.context = Objects.requireNonNull(context);
            return this;
        }

        public Builder eventShapeId(ShapeId eventShapeId) {
            this.eventShapeId = eventShapeId;
            return this;
        }

        public Builder timestampValidationStrategy(TimestampValidationStrategy timestampValidationStrategy) {
            this.timestampValidationStrategy = timestampValidationStrategy;
            return this;
        }

        public Builder allowBoxedNull(boolean allowBoxedNull) {
            this.allowBoxedNull = allowBoxedNull;
            return this;
        }

        public NodeValidationVisitor build() {
            return new NodeValidationVisitor(this);
        }
    }
}

