/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.logging.Logger;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.Prelude;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.ExpectationNotMetException;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.NullNode;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.traits.DynamicTrait;
import software.amazon.smithy.model.traits.Trait;
import software.amazon.smithy.model.traits.TraitDefinition;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.SmithyBuilder;

final class LoaderVisitor {
    private static final Logger LOGGER = Logger.getLogger(LoaderVisitor.class.getName());
    private static final TraitDefinition.Provider TRAIT_DEF_PROVIDER = new TraitDefinition.Provider();
    private final TraitFactory traitFactory;
    private final Map<String, Object> properties;
    private final List<ValidationEvent> events = new ArrayList<ValidationEvent>();
    private final Map<String, Node> metadata = new HashMap<String, Node>();
    private final Map<ShapeId, List<PendingTrait>> pendingTraits = new HashMap<ShapeId, List<PendingTrait>>();
    private final List<ForwardReferenceResolver> forwardReferenceResolvers = new ArrayList<ForwardReferenceResolver>();
    private final Map<ShapeId, AbstractShapeBuilder> pendingShapes = new HashMap<ShapeId, AbstractShapeBuilder>();
    private final Map<ShapeId, Shape> builtShapes = new HashMap<ShapeId, Shape>();
    private final Map<ShapeId, TraitDefinition> builtTraitDefinitions = new HashMap<ShapeId, TraitDefinition>();
    private ValidatedResult<Model> result;

    LoaderVisitor(TraitFactory traitFactory) {
        this(traitFactory, Collections.emptyMap());
    }

    LoaderVisitor(TraitFactory traitFactory, Map<String, Object> properties) {
        this.traitFactory = traitFactory;
        this.properties = properties;
    }

    public boolean hasDefinedShape(ShapeId id) {
        return this.builtShapes.containsKey(id) || this.pendingShapes.containsKey(id);
    }

    public boolean hasProperty(String property) {
        return this.properties.containsKey(property);
    }

    public Optional<Object> getProperty(String property) {
        return Optional.ofNullable(this.properties.get(property));
    }

    public <T> Optional<T> getProperty(String property, Class<T> type) {
        return this.getProperty(property).map(value -> {
            if (!type.isInstance(value)) {
                throw new ClassCastException(String.format("Expected `%s` property of the LoaderVisitor to be a `%s`, but found a `%s`", property, type.getName(), value.getClass().getName()));
            }
            return value;
        });
    }

    public void onError(ValidationEvent event) {
        this.events.add(Objects.requireNonNull(event));
    }

    public void onShape(AbstractShapeBuilder shapeBuilder) {
        ShapeId id = (ShapeId)SmithyBuilder.requiredState((String)"id", (Object)shapeBuilder.getId());
        if (this.validateOnShape(id, shapeBuilder)) {
            this.pendingShapes.put(id, shapeBuilder);
        }
    }

    public void onShape(Shape shape) {
        if (this.validateOnShape(shape.getId(), shape)) {
            this.builtShapes.put(shape.getId(), shape);
        }
        shape.getTrait(TraitDefinition.class).ifPresent(def -> this.onTraitDefinition(shape.getId(), (TraitDefinition)def));
    }

    private void onTraitDefinition(ShapeId target, TraitDefinition definition) {
        this.builtTraitDefinitions.put(target, definition);
    }

    private boolean validateOnShape(ShapeId id, FromSourceLocation source) {
        if (!this.hasDefinedShape(id)) {
            return true;
        }
        if (!Prelude.isPreludeShape(id)) {
            boolean canIgnore;
            SourceLocation previous = Optional.ofNullable((FromSourceLocation)this.pendingShapes.get(id)).orElseGet(() -> this.builtShapes.get(id)).getSourceLocation();
            boolean bl = canIgnore = !id.getMember().isPresent() && previous != SourceLocation.NONE && previous.equals(source.getSourceLocation());
            if (canIgnore) {
                LOGGER.warning(() -> "Ignoring duplicate shape definition defined in the same file: " + id + " defined at " + source.getSourceLocation());
            } else {
                String message = String.format("Duplicate shape definition for `%s` found at `%s` and `%s`", id, previous.getSourceLocation(), source.getSourceLocation());
                throw new SourceException(message, source);
            }
        }
        return false;
    }

    public void onTrait(ShapeId target, ShapeId trait, Node traitValue) {
        this.onTrait(target, trait, traitValue, false);
    }

    public void onAnnotationTrait(ShapeId target, ShapeId trait, NullNode traitValue) {
        this.onTrait(target, trait, traitValue, true);
    }

    private void onTrait(ShapeId target, ShapeId trait, Node traitValue, boolean isAnnotation) {
        if (trait.equals(TraitDefinition.ID)) {
            TraitDefinition traitDef = TRAIT_DEF_PROVIDER.createTrait(target, traitValue);
            this.onTraitDefinition(target, traitDef);
            this.onTrait(target, traitDef);
        } else {
            PendingTrait pendingTrait = new PendingTrait(trait, traitValue, isAnnotation);
            this.addPendingTrait(target, traitValue.getSourceLocation(), trait, pendingTrait);
        }
    }

    public void onTrait(ShapeId target, Trait trait) {
        PendingTrait pending = new PendingTrait(target, trait);
        this.addPendingTrait(target, trait.getSourceLocation(), trait.toShapeId(), pending);
    }

    private void addPendingTrait(ShapeId target, SourceLocation sourceLocation, ShapeId trait, PendingTrait pending) {
        if (Prelude.isImmutablePublicPreludeShape(target)) {
            this.onError(ValidationEvent.builder().severity(Severity.ERROR).eventId("Model").sourceLocation(sourceLocation).shapeId(target).message(String.format("Cannot apply `%s` to an immutable prelude shape defined in `smithy.api`.", trait)).build());
        } else {
            this.pendingTraits.computeIfAbsent(target, targetId -> new ArrayList()).add(pending);
        }
    }

    void addForwardReference(ShapeId expectedId, Consumer<ShapeId> consumer) {
        this.forwardReferenceResolvers.add(new ForwardReferenceResolver(expectedId, consumer));
    }

    public void onMetadata(String key, Node value) {
        if (!this.metadata.containsKey(key)) {
            this.metadata.put(key, value);
        } else if (this.metadata.get(key).isArrayNode() && value.isArrayNode()) {
            ArrayNode previous = this.metadata.get(key).expectArrayNode();
            ArrayList<Node> merged = new ArrayList<Node>(previous.getElements());
            merged.addAll(value.expectArrayNode().getElements());
            ArrayNode mergedArray = new ArrayNode(merged, value.getSourceLocation());
            this.metadata.put(key, mergedArray);
        } else if (!this.metadata.get(key).equals(value)) {
            this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(value).message(String.format("Metadata conflict for key `%s`. Defined in both `%s` and `%s`", key, value.getSourceLocation(), this.metadata.get(key).getSourceLocation())).build());
        } else {
            LOGGER.fine(() -> "Ignoring duplicate metadata definition of " + key);
        }
    }

    public ValidatedResult<Model> onEnd() {
        if (this.result != null) {
            return this.result;
        }
        Model.Builder modelBuilder = Model.builder().metadata(this.metadata);
        this.finalizeShapeTargets();
        this.finalizePendingTraits();
        ArrayList<ShapeId> needsConversion = new ArrayList<ShapeId>();
        for (AbstractShapeBuilder builder : this.pendingShapes.values()) {
            if (!(builder instanceof MemberShape.Builder)) continue;
            needsConversion.add(builder.getId().withoutMember());
        }
        needsConversion.forEach(this::resolveShapeBuilder);
        for (AbstractShapeBuilder shape : this.pendingShapes.values()) {
            MemberShape member;
            if (shape.getClass() != MemberShape.Builder.class || (member = (MemberShape)this.buildShape(modelBuilder, shape)) == null) continue;
            AbstractShapeBuilder container = this.pendingShapes.get(shape.getId().withoutMember());
            if (container == null) {
                LOGGER.warning(String.format("Member shape `%s` added to non-existent shape: %s", member.getId(), member.getSourceLocation()));
                continue;
            }
            container.addMember(member);
        }
        for (AbstractShapeBuilder shape : this.pendingShapes.values()) {
            if (shape.getClass() == MemberShape.Builder.class) continue;
            this.buildShape(modelBuilder, shape);
        }
        modelBuilder.addShapes(this.builtShapes.values());
        this.result = new ValidatedResult<Model>(modelBuilder.build(), this.events);
        return this.result;
    }

    private void finalizeShapeTargets() {
        for (ForwardReferenceResolver resolver : this.forwardReferenceResolvers) {
            ShapeId preludeId;
            if (!this.hasDefinedShape(resolver.expectedId) && Prelude.isPublicPreludeShape(preludeId = resolver.expectedId.withNamespace("smithy.api"))) {
                resolver.consumer.accept(preludeId);
                continue;
            }
            resolver.consumer.accept(resolver.expectedId);
        }
        this.forwardReferenceResolvers.clear();
    }

    private void finalizePendingTraits() {
        for (Map.Entry<ShapeId, List<PendingTrait>> entry : this.pendingTraits.entrySet()) {
            ShapeId target = entry.getKey();
            List<PendingTrait> pendingTraits = entry.getValue();
            AbstractShapeBuilder builder = this.resolveShapeBuilder(target);
            if (builder == null) {
                this.emitErrorsForEachInvalidTraitTarget(target, pendingTraits);
                continue;
            }
            for (PendingTrait pendingTrait : pendingTraits) {
                if (pendingTrait.trait == null) continue;
                builder.addTrait(pendingTrait.trait);
            }
            for (Map.Entry entry2 : this.computeTraits(builder, pendingTraits).entrySet()) {
                this.createAndApplyTraitToShape(builder, (ShapeId)entry2.getKey(), (Node)entry2.getValue());
            }
        }
    }

    private AbstractShapeBuilder resolveShapeBuilder(ShapeId id) {
        if (this.pendingShapes.containsKey(id)) {
            return this.pendingShapes.get(id);
        }
        if (this.builtShapes.containsKey(id)) {
            Object builder = Shape.shapeToBuilder(this.builtShapes.remove(id));
            this.pendingShapes.put(id, (AbstractShapeBuilder)builder);
            return builder;
        }
        return null;
    }

    private void emitErrorsForEachInvalidTraitTarget(ShapeId target, List<PendingTrait> pendingTraits) {
        for (PendingTrait pendingTrait : pendingTraits) {
            this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(pendingTrait.value.getSourceLocation()).message(String.format("Trait `%s` applied to unknown shape `%s`", Trait.getIdiomaticTraitName(pendingTrait.id), target)).build());
        }
    }

    private Shape buildShape(Model.Builder modelBuilder, AbstractShapeBuilder shapeBuilder) {
        try {
            Shape result = (Shape)shapeBuilder.build();
            modelBuilder.addShape(result);
            return result;
        }
        catch (SourceException e) {
            this.onError(ValidationEvent.fromSourceException(e).toBuilder().shapeId(shapeBuilder.getId()).build());
            return null;
        }
    }

    private Map<ShapeId, Node> computeTraits(AbstractShapeBuilder shapeBuilder, List<PendingTrait> pending) {
        HashMap<ShapeId, Node> traits = new HashMap<ShapeId, Node>();
        for (PendingTrait trait : pending) {
            if (trait.trait != null) continue;
            TraitDefinition definition = this.builtTraitDefinitions.get(trait.id);
            if (definition == null) {
                this.onUnresolvedTraitName(shapeBuilder, trait);
                continue;
            }
            ShapeId traitId = trait.id;
            Node value = this.coerceTraitValue(trait);
            Node previous = (Node)traits.get(traitId);
            if (previous == null) {
                traits.put(traitId, value);
                continue;
            }
            if (previous.isArrayNode() && value.isArrayNode()) {
                traits.put(traitId, value.asArrayNode().get().merge(previous.asArrayNode().get()));
                continue;
            }
            if (previous.equals(value)) {
                LOGGER.fine(() -> String.format("Ignoring duplicate %s trait value on %s", traitId, shapeBuilder.getId()));
                continue;
            }
            this.onDuplicateTrait(shapeBuilder.getId(), traitId, previous, value);
        }
        return traits;
    }

    private Node coerceTraitValue(PendingTrait trait) {
        if (trait.isAnnotation && trait.value.isNullNode()) {
            ShapeType targetType = this.determineTraitDefinitionType(trait.id);
            if (targetType == ShapeType.STRUCTURE || targetType == ShapeType.MAP) {
                return new ObjectNode(Collections.emptyMap(), trait.value.getSourceLocation());
            }
            if (targetType == ShapeType.LIST || targetType == ShapeType.SET) {
                return new ArrayNode(Collections.emptyList(), trait.value.getSourceLocation());
            }
        }
        return trait.value;
    }

    private ShapeType determineTraitDefinitionType(ShapeId traitId) {
        assert (this.pendingShapes.containsKey(traitId) || this.builtShapes.containsKey(traitId));
        if (this.pendingShapes.containsKey(traitId)) {
            return this.pendingShapes.get(traitId).getShapeType();
        }
        return this.builtShapes.get(traitId).getType();
    }

    private void onDuplicateTrait(ShapeId target, ShapeId traitName, FromSourceLocation previous, Node duplicate) {
        this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(duplicate.getSourceLocation()).shapeId(target).message(String.format("Conflicting `%s` trait found on shape `%s`. The previous trait was defined at `%s`, and a conflicting trait was defined at `%s`.", traitName, target, previous.getSourceLocation(), duplicate.getSourceLocation())).build());
    }

    private void onUnresolvedTraitName(AbstractShapeBuilder shapeBuilder, PendingTrait trait) {
        Severity severity = this.getProperty("assembler.allowUnknownTraits", Boolean.class).orElse(false) != false ? Severity.WARNING : Severity.ERROR;
        this.onError(ValidationEvent.builder().eventId("Model").severity(severity).sourceLocation(trait.value.getSourceLocation()).shapeId(shapeBuilder.getId()).message(String.format("Unable to resolve trait `%s`. If this is a custom trait, then it must be defined before it can be used in a model.", trait.id)).build());
    }

    private void createAndApplyTraitToShape(AbstractShapeBuilder shapeBuilder, ShapeId traitId, Node traitValue) {
        try {
            Trait createdTrait = this.traitFactory.createTrait(traitId, shapeBuilder.getId(), traitValue).orElseGet(() -> new DynamicTrait(traitId, traitValue));
            shapeBuilder.addTrait(createdTrait);
        }
        catch (SourceException e) {
            this.events.add(ValidationEvent.fromSourceException(e, String.format("Error creating trait `%s`: ", Trait.getIdiomaticTraitName(traitId))).toBuilder().shapeId(shapeBuilder.getId()).build());
        }
    }

    void checkForAdditionalProperties(ObjectNode node, ShapeId shape, Collection<String> properties) {
        try {
            node.expectNoAdditionalProperties(properties);
        }
        catch (ExpectationNotMetException e) {
            ValidationEvent event = ValidationEvent.fromSourceException(e).toBuilder().shapeId(shape).severity(Severity.WARNING).build();
            this.onError(event);
        }
    }

    boolean isVersionSupported(String versionString) {
        return versionString.equals("1") || versionString.equals("1.0");
    }

    private static final class ForwardReferenceResolver {
        final ShapeId expectedId;
        final Consumer<ShapeId> consumer;

        ForwardReferenceResolver(ShapeId expectedId, Consumer<ShapeId> consumer) {
            this.expectedId = expectedId;
            this.consumer = consumer;
        }
    }

    private static final class PendingTrait {
        final ShapeId id;
        final Node value;
        final Trait trait;
        final boolean isAnnotation;

        PendingTrait(ShapeId id, Node value, boolean isAnnotation) {
            this.id = id;
            this.value = value;
            this.trait = null;
            this.isAnnotation = isAnnotation;
        }

        PendingTrait(ShapeId id, Trait trait) {
            this.id = id;
            this.trait = trait;
            this.value = null;
            this.isAnnotation = false;
        }
    }
}

