/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.LoaderUtils;
import software.amazon.smithy.model.loader.LoaderVisitor;
import software.amazon.smithy.model.loader.ModelLoader;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.SmithyModelLexer;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.BooleanNode;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.node.NullNode;
import software.amazon.smithy.model.node.NumberNode;
import software.amazon.smithy.model.node.ObjectNode;
import software.amazon.smithy.model.node.StringNode;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.BigDecimalShape;
import software.amazon.smithy.model.shapes.BigIntegerShape;
import software.amazon.smithy.model.shapes.BlobShape;
import software.amazon.smithy.model.shapes.BooleanShape;
import software.amazon.smithy.model.shapes.ByteShape;
import software.amazon.smithy.model.shapes.CollectionShape;
import software.amazon.smithy.model.shapes.DocumentShape;
import software.amazon.smithy.model.shapes.DoubleShape;
import software.amazon.smithy.model.shapes.FloatShape;
import software.amazon.smithy.model.shapes.IntegerShape;
import software.amazon.smithy.model.shapes.ListShape;
import software.amazon.smithy.model.shapes.LongShape;
import software.amazon.smithy.model.shapes.MapShape;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.OperationShape;
import software.amazon.smithy.model.shapes.ResourceShape;
import software.amazon.smithy.model.shapes.ServiceShape;
import software.amazon.smithy.model.shapes.SetShape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeIdSyntaxException;
import software.amazon.smithy.model.shapes.ShortShape;
import software.amazon.smithy.model.shapes.StringShape;
import software.amazon.smithy.model.shapes.StructureShape;
import software.amazon.smithy.model.shapes.TimestampShape;
import software.amazon.smithy.model.shapes.UnionShape;
import software.amazon.smithy.model.traits.DocumentationTrait;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.model.validation.ValidationUtils;
import software.amazon.smithy.utils.ListUtils;
import software.amazon.smithy.utils.MapUtils;
import software.amazon.smithy.utils.Pair;
import software.amazon.smithy.utils.SetUtils;

final class SmithyModelLoader
implements ModelLoader {
    private static final Collection<String> MAP_KEYS = ListUtils.of((Object[])new String[]{"key", "value"});
    private static final Map<String, Consumer<State>> STATEMENTS = new HashMap<String, Consumer<State>>();
    private static final Set<String> SUPPORTS_TRAITS;

    SmithyModelLoader() {
    }

    @Override
    public boolean load(String path, Supplier<String> contentSupplier, LoaderVisitor visitor) {
        if (!path.endsWith(".smithy")) {
            return false;
        }
        visitor.onOpenFile(path);
        SmithyModelLexer lexer = new SmithyModelLexer(path, contentSupplier.get());
        State state = new State(path, lexer, visitor);
        SmithyModelLoader.parse(state);
        return true;
    }

    private static void parse(State state) {
        while (!state.eof()) {
            SmithyModelLexer.Token token = state.expect(SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.ANNOTATION, SmithyModelLexer.TokenType.CONTROL, SmithyModelLexer.TokenType.DOC);
            if (token.type == SmithyModelLexer.TokenType.UNQUOTED) {
                SmithyModelLoader.parseStatement(token, state);
                continue;
            }
            if (token.type == SmithyModelLexer.TokenType.ANNOTATION) {
                state.pendingTraits.add(SmithyModelLoader.parseTraitValue(token, state, TraitValueType.SHAPE));
                continue;
            }
            if (token.type == SmithyModelLexer.TokenType.CONTROL) {
                SmithyModelLoader.parseControlStatement(state, token);
                continue;
            }
            if (token.type != SmithyModelLexer.TokenType.DOC) continue;
            SmithyModelLoader.parseDocComment(state, token, false);
        }
    }

    private static void parseStatement(SmithyModelLexer.Token token, State state) {
        if (!STATEMENTS.containsKey(token.lexeme)) {
            throw state.syntax(String.format("Expected one of %s", ValidationUtils.tickedList(STATEMENTS.keySet())));
        }
        STATEMENTS.get(token.lexeme).accept(state);
    }

    private static void parseControlStatement(State state, SmithyModelLexer.Token token) {
        if (state.visitor.getNamespace() != null || state.definedMetadata) {
            throw state.syntax("A control statement must come before any namespace, metadata, or shape");
        }
        String key = token.lexeme;
        Node value = SmithyModelLoader.parseNodeValue(state, state.next());
        if (key.equals("version")) {
            if (!value.isStringNode()) {
                value.expectStringNode("The $version control statement must have a string value, but found " + Node.printJson(value));
            }
            state.visitor.onVersion(state.currentLocation(), value.expectStringNode().getValue());
        } else {
            state.visitor.onError(ValidationEvent.builder().eventId("Model").sourceLocation(value).severity(Severity.WARNING).message(String.format("Unknown control statement `%s` with value `%s", key, Node.printJson(value))).build());
        }
    }

    private static void parseUseStatement(State state) {
        if (state.definedShapesOrTraits) {
            throw state.syntax("A use statement must come before any shape definition");
        }
        SmithyModelLexer.Token namespaceToken = state.expect(SmithyModelLexer.TokenType.UNQUOTED);
        try {
            state.visitor.onUseShape(ShapeId.from(namespaceToken.lexeme), namespaceToken);
            state.expectNewline();
        }
        catch (ShapeIdSyntaxException e) {
            throw state.syntax(e.getMessage());
        }
    }

    private static void parseDocComment(State state, SmithyModelLexer.Token token, boolean memberScope) {
        StringBuilder builder = new StringBuilder(token.getDocContents());
        while (state.test(SmithyModelLexer.TokenType.DOC)) {
            builder.append('\n').append(state.next().getDocContents());
        }
        state.pendingDocComment = new DocComment(builder.toString(), token.getSourceLocation());
        if (!state.peek().isPresent()) {
            throw state.syntax("Found a documentation comment that doesn't document anything");
        }
        SmithyModelLexer.Token next = state.peek().get();
        if (next.type != SmithyModelLexer.TokenType.ANNOTATION && (next.type != SmithyModelLexer.TokenType.UNQUOTED || !memberScope && !SUPPORTS_TRAITS.contains(next.lexeme))) {
            throw state.syntax("Documentation cannot be applied to `" + next.lexeme + "`");
        }
    }

    private static void parseNamespace(State state) {
        state.visitor.onNamespace(state.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.UNQUOTED}).lexeme, state.current);
    }

    private static void requireNamespaceOrThrow(State state) {
        if (state.visitor.getNamespace() == null) {
            throw state.syntax("A namespace must be set before shapes or traits can be defined");
        }
    }

    private static Pair<String, Node> parseTraitValue(SmithyModelLexer.Token token, State state, TraitValueType type) {
        try {
            SmithyModelLoader.requireNamespaceOrThrow(state);
            ShapeId.fromOptionalNamespace(state.visitor.getNamespace(), token.lexeme);
            Pair result = Pair.of((Object)token.lexeme, (Object)SmithyModelLoader.parseTraitValueBody(state));
            if (type == TraitValueType.APPLY) {
                return result;
            }
            if (!state.peek().isPresent()) {
                throw state.syntax("Found a trait doesn't apply to anything");
            }
            SmithyModelLexer.Token next = state.peek().get();
            if (next.type != SmithyModelLexer.TokenType.ANNOTATION && (next.type != SmithyModelLexer.TokenType.UNQUOTED || type != TraitValueType.MEMBER && !SUPPORTS_TRAITS.contains(next.lexeme))) {
                throw state.syntax("Traits cannot be applied to `" + next.lexeme + "`");
            }
            return result;
        }
        catch (ShapeIdSyntaxException e) {
            throw state.syntax("Invalid trait name syntax. Trait names must adhere to the same syntax as shape IDs.");
        }
    }

    private static Node parseTraitValueBody(State state) {
        if (!state.test(SmithyModelLexer.TokenType.LPAREN)) {
            return new NullNode(state.currentLocation());
        }
        state.expect(SmithyModelLexer.TokenType.LPAREN);
        SmithyModelLexer.Token next = state.expect(SmithyModelLexer.TokenType.RPAREN, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.NUMBER);
        if (next.type == SmithyModelLexer.TokenType.RPAREN) {
            return new ObjectNode(MapUtils.of(), next.getSourceLocation());
        }
        if (state.test(SmithyModelLexer.TokenType.COLON)) {
            if (next.type == SmithyModelLexer.TokenType.QUOTED || next.type == SmithyModelLexer.TokenType.UNQUOTED) {
                return SmithyModelLoader.parseObjectNodeWithKey(state, state.currentLocation(), SmithyModelLexer.TokenType.RPAREN, next);
            }
            throw state.syntax("Expected a string to start a trait value object");
        }
        Node result = next.type == SmithyModelLexer.TokenType.LBRACKET ? SmithyModelLoader.parseArrayNode(state, next.getSourceLocation()) : (next.type == SmithyModelLexer.TokenType.NUMBER ? SmithyModelLoader.parseNumber(next) : SmithyModelLoader.parseNodeValue(state, next));
        state.expect(SmithyModelLexer.TokenType.RPAREN);
        return result;
    }

    private static ShapeId parseShapeName(State state) {
        state.definedShapesOrTraits = true;
        SmithyModelLexer.Token nameToken = state.expect(SmithyModelLexer.TokenType.UNQUOTED);
        String name = nameToken.lexeme;
        ShapeId id = state.visitor.onShapeDefName(name, nameToken);
        state.pendingTraits.forEach(pair -> state.visitor.onTrait(id, (String)pair.getLeft(), (Node)pair.getRight()));
        state.pendingTraits.clear();
        SmithyModelLoader.collectPendingDocString(state, id);
        return id;
    }

    private static void collectPendingDocString(State state, ShapeId id) {
        if (state.pendingDocComment != null) {
            StringNode value = new StringNode(state.pendingDocComment.content, state.pendingDocComment.sourceLocation);
            state.pendingDocComment = null;
            state.visitor.onTrait(id, new DocumentationTrait(((Object)value).toString(), value.getSourceLocation()));
        }
    }

    private static void parseSimpleShape(State state, AbstractShapeBuilder builder) {
        builder.source(state.currentLocation());
        builder.id(SmithyModelLoader.parseShapeName(state));
        state.visitor.onShape(builder);
        state.expectNewline();
    }

    private static void parseService(State state) {
        SourceLocation sourceLocation = state.currentLocation();
        ShapeId shapeId = SmithyModelLoader.parseShapeName(state);
        ServiceShape.Builder builder = (ServiceShape.Builder)((ServiceShape.Builder)new ServiceShape.Builder().id(shapeId)).source(sourceLocation);
        ObjectNode shapeNode = SmithyModelLoader.parseObjectNode(state, state.expect(SmithyModelLexer.TokenType.LBRACE).getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
        shapeNode.warnIfAdditionalProperties(LoaderUtils.SERVICE_PROPERTY_NAMES);
        LoaderUtils.loadServiceObject(builder, shapeId, shapeNode);
        state.visitor.onShape(builder);
        state.expectNewline();
    }

    private static void parseResource(State state) {
        SourceLocation sourceLocation = state.currentLocation();
        ShapeId shapeId = SmithyModelLoader.parseShapeName(state);
        ResourceShape.Builder builder = (ResourceShape.Builder)((ResourceShape.Builder)ResourceShape.builder().id(shapeId)).source(sourceLocation);
        state.visitor.onShape(builder);
        ObjectNode shapeNode = SmithyModelLoader.parseObjectNode(state, state.expect(SmithyModelLexer.TokenType.LBRACE).getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
        shapeNode.warnIfAdditionalProperties(LoaderUtils.RESOURCE_PROPERTY_NAMES);
        LoaderUtils.loadResourceObject(builder, shapeId, shapeNode, state.visitor);
        state.expectNewline();
    }

    private static void parseOperation(State state) {
        SourceLocation sourceLocation = state.currentLocation();
        ShapeId id = SmithyModelLoader.parseShapeName(state);
        OperationShape.Builder builder = (OperationShape.Builder)((OperationShape.Builder)OperationShape.builder().id(id)).source(sourceLocation);
        state.visitor.onShape(builder);
        state.expect(SmithyModelLexer.TokenType.LPAREN);
        SmithyModelLexer.Token next = state.expect(SmithyModelLexer.TokenType.RPAREN, SmithyModelLexer.TokenType.UNQUOTED);
        if (next.type == SmithyModelLexer.TokenType.UNQUOTED) {
            state.visitor.onShapeTarget(next.lexeme, next, builder::input);
            state.expect(SmithyModelLexer.TokenType.RPAREN);
        }
        if (state.test(SmithyModelLexer.TokenType.RETURN)) {
            state.expect(SmithyModelLexer.TokenType.RETURN);
            SmithyModelLexer.Token returnToken = state.expect(SmithyModelLexer.TokenType.UNQUOTED);
            state.visitor.onShapeTarget(returnToken.lexeme, returnToken, builder::output);
        }
        if (state.peek().filter(token -> token.lexeme.equals("errors")).isPresent()) {
            state.next();
            state.expect(SmithyModelLexer.TokenType.LBRACKET);
            if (!state.test(SmithyModelLexer.TokenType.RBRACKET)) {
                while (true) {
                    SmithyModelLexer.Token errorToken = state.expect(SmithyModelLexer.TokenType.UNQUOTED);
                    state.visitor.onShapeTarget(errorToken.lexeme, errorToken, builder::addError);
                    if (state.test(SmithyModelLexer.TokenType.RBRACKET)) break;
                    state.expect(SmithyModelLexer.TokenType.COMMA);
                }
            }
            state.expect(SmithyModelLexer.TokenType.RBRACKET);
        }
        state.expectNewline();
    }

    private static void parseStructuredShape(State state, String shapeType, AbstractShapeBuilder builder) {
        builder.source(state.currentLocation());
        ShapeId id = SmithyModelLoader.parseShapeName(state);
        state.visitor.onShape((AbstractShapeBuilder)builder.id(id));
        SmithyModelLoader.parseStructuredBody(shapeType, state, id);
    }

    private static void parseStructuredBody(String shapeType, State state, ShapeId parent) {
        SmithyModelLoader.parseStructuredContents(shapeType, state, parent, SetUtils.of());
        state.expectNewline();
    }

    private static void parseStructuredContents(String shapeType, State state, ShapeId parent, Collection<String> requiredMembers) {
        state.expect(SmithyModelLexer.TokenType.LBRACE);
        ArrayList<Pair<String, Node>> memberTraits = new ArrayList<Pair<String, Node>>();
        Set<Object> remainingMembers = requiredMembers.isEmpty() ? SetUtils.of() : new HashSet<String>(requiredMembers);
        SmithyModelLexer.Token token = state.expect(SmithyModelLexer.TokenType.ANNOTATION, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.RBRACE, SmithyModelLexer.TokenType.DOC);
        while (token.type != SmithyModelLexer.TokenType.RBRACE) {
            if (token.type == SmithyModelLexer.TokenType.ANNOTATION) {
                memberTraits.add(SmithyModelLoader.parseTraitValue(token, state, TraitValueType.MEMBER));
            } else if (token.type == SmithyModelLexer.TokenType.DOC) {
                SmithyModelLoader.parseDocComment(state, token, true);
            } else {
                String memberName = token.lexeme;
                if (!requiredMembers.isEmpty()) {
                    if (!requiredMembers.contains(memberName)) {
                        throw state.syntax(String.format("Invalid member `%s` found in %s shape `%s`. Expected one of the following members: [%s]", memberName, shapeType, parent, ValidationUtils.tickedList(requiredMembers)));
                    }
                    remainingMembers.remove(memberName);
                }
                ShapeId memberId = parent.withMember(memberName);
                state.expect(SmithyModelLexer.TokenType.COLON);
                SmithyModelLoader.parseMember(state, memberId);
                for (Pair pair : memberTraits) {
                    state.visitor.onTrait(memberId, (String)pair.getLeft(), (Node)pair.getRight());
                }
                memberTraits.clear();
                SmithyModelLoader.collectPendingDocString(state, memberId);
                if (state.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.COMMA, SmithyModelLexer.TokenType.RBRACE}).type == SmithyModelLexer.TokenType.RBRACE) break;
            }
            token = state.expect(SmithyModelLexer.TokenType.ANNOTATION, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.RBRACE, SmithyModelLexer.TokenType.DOC);
        }
        if (!remainingMembers.isEmpty()) {
            throw state.syntax(String.format("Missing required members of %s shape `%s`: [%s]", shapeType, parent, ValidationUtils.tickedList(remainingMembers)));
        }
    }

    private static void parseMember(State state, ShapeId memberId) {
        MemberShape.Builder memberBuilder = (MemberShape.Builder)((MemberShape.Builder)MemberShape.builder().id(memberId)).source(state.currentLocation());
        state.visitor.onShape(memberBuilder);
        SmithyModelLexer.Token targetToken = state.expect(SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.QUOTED);
        state.visitor.onShapeTarget(targetToken.lexeme, targetToken, memberBuilder::target);
    }

    private static void parseCollection(State state, String shapeType, CollectionShape.Builder builder) {
        builder.source(state.currentLocation());
        ShapeId id = SmithyModelLoader.parseShapeName(state);
        SmithyModelLoader.parseStructuredContents(shapeType, state, id, SetUtils.of((Object)"member"));
        state.visitor.onShape((AbstractShapeBuilder)builder.id(id));
        state.expectNewline();
    }

    private static void parseMap(State state) {
        SourceLocation sourceLocation = state.currentLocation();
        ShapeId id = SmithyModelLoader.parseShapeName(state);
        MapShape.Builder builder = (MapShape.Builder)((MapShape.Builder)MapShape.builder().id(id)).source(sourceLocation);
        SmithyModelLoader.parseStructuredContents("map", state, id, MAP_KEYS);
        state.visitor.onShape(builder);
        state.expectNewline();
    }

    private static void parseMetadata(State state) {
        state.definedMetadata = true;
        String key = state.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED}).lexeme;
        state.expect(SmithyModelLexer.TokenType.EQUAL);
        state.visitor.onMetadata(key, SmithyModelLoader.parseNode(state));
        state.expectNewline();
    }

    private static void parseApply(State state) {
        SmithyModelLoader.requireNamespaceOrThrow(state);
        String name = state.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{SmithyModelLexer.TokenType.UNQUOTED}).lexeme;
        ShapeId id = ShapeId.fromOptionalNamespace(state.visitor.getNamespace(), name);
        SmithyModelLexer.Token token = state.expect(SmithyModelLexer.TokenType.ANNOTATION);
        Pair<String, Node> trait = SmithyModelLoader.parseTraitValue(token, state, TraitValueType.APPLY);
        state.visitor.onTrait(id, (String)trait.getLeft(), (Node)trait.getRight());
        state.expectNewline();
    }

    private static Node parseNode(State state) {
        return SmithyModelLoader.parseNodeValue(state, state.expect(SmithyModelLexer.TokenType.LBRACE, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.NUMBER));
    }

    private static Node parseNodeValue(State state, SmithyModelLexer.Token token) {
        switch (token.type) {
            case LBRACE: {
                return SmithyModelLoader.parseObjectNode(state, token.getSourceLocation(), SmithyModelLexer.TokenType.RBRACE);
            }
            case LBRACKET: {
                return SmithyModelLoader.parseArrayNode(state, token.getSourceLocation());
            }
            case QUOTED: {
                return new StringNode(token.lexeme, token.getSourceLocation());
            }
            case NUMBER: {
                return SmithyModelLoader.parseNumber(token);
            }
            case UNQUOTED: {
                return SmithyModelLoader.parseUnquotedNode(state, token);
            }
        }
        throw new IllegalStateException("Parse node value not expected to be called with: " + token);
    }

    private static Node parseUnquotedNode(State state, SmithyModelLexer.Token token) {
        switch (token.lexeme) {
            case "true": {
                return new BooleanNode(true, token.getSourceLocation());
            }
            case "false": {
                return new BooleanNode(false, token.getSourceLocation());
            }
            case "null": {
                return new NullNode(token.getSourceLocation());
            }
        }
        Pair<StringNode, Consumer<String>> pair = StringNode.createLazyString(token.lexeme, token.getSourceLocation());
        Consumer consumer = (Consumer)pair.right;
        state.visitor.onShapeTarget(token.lexeme, token, id -> consumer.accept(id.toString()));
        return (Node)pair.left;
    }

    private static NumberNode parseNumber(SmithyModelLexer.Token token) {
        if (token.lexeme.contains("e") || token.lexeme.contains(".")) {
            return new NumberNode(Double.valueOf(token.lexeme), token.getSourceLocation());
        }
        return new NumberNode(Long.parseLong(token.lexeme), token.getSourceLocation());
    }

    private static ObjectNode parseObjectNode(State state, SourceLocation location, SmithyModelLexer.TokenType closing) {
        return SmithyModelLoader.parseObjectNodeWithKey(state, location, closing, state.expect(SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, closing));
    }

    private static ObjectNode parseObjectNodeWithKey(State state, SourceLocation sloc, SmithyModelLexer.TokenType closing, SmithyModelLexer.Token key) {
        LinkedHashMap<StringNode, Node> entries = new LinkedHashMap<StringNode, Node>();
        while (key.type != closing) {
            state.expect(SmithyModelLexer.TokenType.COLON);
            Node value = SmithyModelLoader.parseNode(state);
            entries.put(new StringNode(key.lexeme, key.getSourceLocation()), value);
            if (state.expect((SmithyModelLexer.TokenType[])new SmithyModelLexer.TokenType[]{closing, SmithyModelLexer.TokenType.COMMA}).type == closing) break;
            key = state.expect(closing, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED);
        }
        return new ObjectNode(entries, sloc);
    }

    private static ArrayNode parseArrayNode(State state, SourceLocation location) {
        ArrayList<Node> values = new ArrayList<Node>();
        SmithyModelLexer.Token next = state.expect(SmithyModelLexer.TokenType.LBRACE, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.NUMBER, SmithyModelLexer.TokenType.RBRACKET);
        while (next.type != SmithyModelLexer.TokenType.RBRACKET) {
            values.add(SmithyModelLoader.parseNodeValue(state, next));
            next = state.expect(SmithyModelLexer.TokenType.RBRACKET, SmithyModelLexer.TokenType.COMMA);
            if (next.type == SmithyModelLexer.TokenType.RBRACKET) break;
            next = state.expect(SmithyModelLexer.TokenType.RBRACKET, SmithyModelLexer.TokenType.LBRACE, SmithyModelLexer.TokenType.LBRACKET, SmithyModelLexer.TokenType.QUOTED, SmithyModelLexer.TokenType.UNQUOTED, SmithyModelLexer.TokenType.NUMBER);
        }
        return new ArrayNode(values, location);
    }

    static {
        STATEMENTS.put("namespace", SmithyModelLoader::parseNamespace);
        STATEMENTS.put("use", SmithyModelLoader::parseUseStatement);
        STATEMENTS.put("service", SmithyModelLoader::parseService);
        STATEMENTS.put("operation", SmithyModelLoader::parseOperation);
        STATEMENTS.put("resource", SmithyModelLoader::parseResource);
        STATEMENTS.put("structure", state -> SmithyModelLoader.parseStructuredShape(state, "structure", StructureShape.builder()));
        STATEMENTS.put("union", state -> SmithyModelLoader.parseStructuredShape(state, "union", UnionShape.builder()));
        STATEMENTS.put("list", state -> SmithyModelLoader.parseCollection(state, "list", ListShape.builder()));
        STATEMENTS.put("set", state -> SmithyModelLoader.parseCollection(state, "set", SetShape.builder()));
        STATEMENTS.put("map", SmithyModelLoader::parseMap);
        STATEMENTS.put("boolean", state -> SmithyModelLoader.parseSimpleShape(state, BooleanShape.builder()));
        STATEMENTS.put("string", state -> SmithyModelLoader.parseSimpleShape(state, StringShape.builder()));
        STATEMENTS.put("blob", state -> SmithyModelLoader.parseSimpleShape(state, BlobShape.builder()));
        STATEMENTS.put("byte", state -> SmithyModelLoader.parseSimpleShape(state, ByteShape.builder()));
        STATEMENTS.put("short", state -> SmithyModelLoader.parseSimpleShape(state, ShortShape.builder()));
        STATEMENTS.put("integer", state -> SmithyModelLoader.parseSimpleShape(state, IntegerShape.builder()));
        STATEMENTS.put("long", state -> SmithyModelLoader.parseSimpleShape(state, LongShape.builder()));
        STATEMENTS.put("float", state -> SmithyModelLoader.parseSimpleShape(state, FloatShape.builder()));
        STATEMENTS.put("document", state -> SmithyModelLoader.parseSimpleShape(state, DocumentShape.builder()));
        STATEMENTS.put("double", state -> SmithyModelLoader.parseSimpleShape(state, DoubleShape.builder()));
        STATEMENTS.put("bigInteger", state -> SmithyModelLoader.parseSimpleShape(state, BigIntegerShape.builder()));
        STATEMENTS.put("bigDecimal", state -> SmithyModelLoader.parseSimpleShape(state, BigDecimalShape.builder()));
        STATEMENTS.put("timestamp", state -> SmithyModelLoader.parseSimpleShape(state, TimestampShape.builder()));
        STATEMENTS.put("metadata", SmithyModelLoader::parseMetadata);
        STATEMENTS.put("apply", SmithyModelLoader::parseApply);
        SUPPORTS_TRAITS = SetUtils.of((Object[])new String[]{"bigDecimal", "bigInteger", "blob", "boolean", "byte", "document", "double", "float", "integer", "list", "long", "map", "operation", "resource", "service", "set", "short", "string", "structure", "timestamp", "union"});
    }

    private static enum TraitValueType {
        SHAPE,
        MEMBER,
        APPLY;

    }

    private static final class State {
        final String filename;
        final SmithyModelLexer lexer;
        final List<Pair<String, Node>> pendingTraits = new ArrayList<Pair<String, Node>>();
        final LoaderVisitor visitor;
        SmithyModelLexer.Token current;
        DocComment pendingDocComment;
        boolean definedMetadata;
        boolean definedShapesOrTraits;

        State(String filename, SmithyModelLexer lexer, LoaderVisitor visitor) {
            this.filename = filename;
            this.lexer = lexer;
            this.visitor = visitor;
        }

        boolean eof() {
            return !this.lexer.hasNext();
        }

        SmithyModelLexer.Token current() {
            return Objects.requireNonNull(this.current, "Call to next must occur before accessing current");
        }

        SourceLocation currentLocation() {
            return this.current.getSourceLocation();
        }

        SmithyModelLexer.Token next() {
            if (!this.lexer.hasNext()) {
                throw this.syntax("Unexpected EOF", this.current != null ? this.current.span : 0);
            }
            this.current = this.lexer.next();
            return this.current;
        }

        SmithyModelLexer.Token expect(SmithyModelLexer.TokenType ... tokens) {
            SmithyModelLexer.Token next = this.next();
            for (SmithyModelLexer.TokenType type : tokens) {
                if (type != next.type) continue;
                return next;
            }
            if (next.type != SmithyModelLexer.TokenType.ERROR) {
                throw this.syntax(tokens.length == 1 ? "Expected " + (Object)((Object)tokens[0]) : "Expected one of " + Arrays.toString((Object[])tokens));
            }
            if (next.lexeme.contains("'") || next.lexeme.contains("\"")) {
                throw this.syntax("Unexpected syntax. Perhaps an unclosed quote?");
            }
            if (next.errorMessage == null) {
                throw this.syntax("Unexpected syntax");
            }
            throw this.syntax(next.errorMessage);
        }

        boolean test(SmithyModelLexer.TokenType type) {
            return !this.eof() && this.lexer.peek().type == type;
        }

        Optional<SmithyModelLexer.Token> peek() {
            return Optional.ofNullable(this.lexer.peek());
        }

        void expectNewline() {
            if (this.peek().isPresent() && this.peek().get().line == this.current().line) {
                this.next();
                throw this.syntax("Expected a new line before this token");
            }
        }

        ModelSyntaxException syntax(String message) {
            return this.syntax(message, 0);
        }

        ModelSyntaxException syntax(String message, int offset) {
            SmithyModelLexer.Token token = this.current();
            int line = token.line;
            int column = token.column + offset;
            String lexeme = token.lexeme;
            String formatted = String.format("Parse error at line %d, column %d near `%s`: %s", line, column, lexeme, message);
            return new ModelSyntaxException(formatted, new SourceLocation(this.filename, line, column));
        }
    }

    private static final class DocComment {
        final SourceLocation sourceLocation;
        final String content;

        DocComment(String content, SourceLocation sourceLocation) {
            this.content = content;
            this.sourceLocation = sourceLocation;
        }
    }
}

