/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.Arrays;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.function.Predicate;
import java.util.function.Supplier;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.utils.StringUtils;

final class SmithyModelLexer
implements Iterator<Token> {
    private final String filename;
    private final String input;
    private Token peeked;
    private int line = 1;
    private int column = 1;
    private int position = -1;

    SmithyModelLexer(String filename, String input) {
        this.filename = filename;
        if (input.indexOf(13) > -1) {
            input = input.replaceAll("\r\n?", "\n");
        }
        this.input = input;
    }

    Token peek() {
        if (this.peeked == null) {
            this.peeked = this.maybeGetNext();
        }
        return this.peeked;
    }

    @Override
    public boolean hasNext() {
        return this.peek() != null;
    }

    @Override
    public Token next() {
        Token next = this.maybeGetNext();
        if (next == null) {
            throw new NoSuchElementException();
        }
        return next;
    }

    private Token maybeGetNext() {
        if (this.peeked != null) {
            Token value = this.peeked;
            this.peeked = null;
            return value;
        }
        while (this.position < this.input.length() - 1) {
            Token token = this.parseCharacter(this.consume());
            if (token == null) continue;
            return token;
        }
        return null;
    }

    private Token parseCharacter(char c) {
        switch (c) {
            case '\t': 
            case '\n': 
            case ' ': {
                break;
            }
            case '(': {
                return new Token(this.filename, TokenType.LPAREN, "(", this.line, this.column - 1, 1, null);
            }
            case ')': {
                return new Token(this.filename, TokenType.RPAREN, ")", this.line, this.column - 1, 1, null);
            }
            case '{': {
                return new Token(this.filename, TokenType.LBRACE, "{", this.line, this.column - 1, 1, null);
            }
            case '}': {
                return new Token(this.filename, TokenType.RBRACE, "}", this.line, this.column - 1, 1, null);
            }
            case '[': {
                return new Token(this.filename, TokenType.LBRACKET, "[", this.line, this.column - 1, 1, null);
            }
            case ']': {
                return new Token(this.filename, TokenType.RBRACKET, "]", this.line, this.column - 1, 1, null);
            }
            case '=': {
                return new Token(this.filename, TokenType.EQUAL, "=", this.line, this.column - 1, 1, null);
            }
            case ':': {
                return new Token(this.filename, TokenType.COLON, ":", this.line, this.column - 1, 1, null);
            }
            case ',': {
                return new Token(this.filename, TokenType.COMMA, ",", this.line, this.column - 1, 1, null);
            }
            case '/': {
                Token parsedToken = this.parseToken(TokenType.DOC, this::parseComment);
                if (parsedToken == null) break;
                return parsedToken;
            }
            case '$': {
                return this.parseToken(TokenType.CONTROL, this::parseControl);
            }
            case '@': {
                return this.parseToken(TokenType.ANNOTATION, this::parseTrait);
            }
            case '\"': {
                return this.parseToken(TokenType.QUOTED, this::parseQuotes);
            }
            case '\'': {
                return this.parseToken(TokenType.QUOTED, this::parseSingleQuotes);
            }
            case '-': {
                if (this.peekChar() == '>') {
                    this.consume();
                    return new Token(this.filename, TokenType.RETURN, "->", this.line, this.column - 1, 2, null);
                }
                return this.parseToken(TokenType.NUMBER, this::parseNumber);
            }
            default: {
                if (this.isDigit(c)) {
                    return this.parseToken(TokenType.NUMBER, this::parseNumber);
                }
                if (this.isIdentifierStart(c)) {
                    return this.parseToken(TokenType.UNQUOTED, this::parseIdentifier);
                }
                return new Token(this.filename, TokenType.ERROR, String.valueOf(c), this.line, this.column - 1, 1, null);
            }
        }
        return null;
    }

    private boolean isDigit(char c) {
        return c >= '0' && c <= '9';
    }

    private boolean isIdentifierStart(char c) {
        return c == '_' || c >= 'A' && c <= 'Z' || c >= 'a' && c <= 'z';
    }

    private char consume() {
        char c;
        if (this.position >= this.input.length() - 1) {
            throw new RuntimeException("Unexpected EOF");
        }
        if ((c = this.input.charAt(++this.position)) == '\n') {
            ++this.line;
            this.column = 1;
        } else {
            ++this.column;
        }
        return c;
    }

    private char expect(char ... expectedChars) {
        char c = this.consume();
        for (char expected : expectedChars) {
            if (c != expected) continue;
            return c;
        }
        throw new RuntimeException("Expected one of the following characters: " + Arrays.toString(expectedChars));
    }

    private char peekChar() {
        return this.peekChar(1);
    }

    private char peekChar(int offset) {
        return this.position + offset < this.input.length() ? this.input.charAt(this.position + offset) : (char)'\u0000';
    }

    private Token parseToken(TokenType type, Supplier<String> parser) {
        int currentLine = this.line;
        int currentColumn = this.column - 1;
        int startPosition = this.position;
        try {
            String lexeme = parser.get();
            return lexeme == null ? null : new Token(this.filename, type, lexeme, currentLine, currentColumn, 1 + this.position - startPosition, null);
        }
        catch (RuntimeException e) {
            return new Token(this.filename, TokenType.ERROR, this.input.substring(startPosition, this.position), currentLine, currentColumn, 1 + this.position - startPosition, e.getMessage());
        }
    }

    private String parseComment() {
        int startPosition = this.position;
        this.expect('/');
        boolean isDocComment = this.peekChar() == '/';
        this.consumeUntilNoLongerMatches(c -> c.charValue() != '\n');
        return !isDocComment ? null : this.input.substring(startPosition, this.position + 1);
    }

    private int consumeUntilNoLongerMatches(Predicate<Character> predicate) {
        char peekedChar;
        int startPosition = this.position;
        while ((peekedChar = this.peekChar()) != '\u0000' && predicate.test(Character.valueOf(peekedChar))) {
            this.consume();
        }
        return this.position - startPosition;
    }

    private String parseControl() {
        this.consume();
        String identifier = this.parseIdentifier();
        this.consumeUntilNoLongerMatches(Character::isWhitespace);
        this.expect(':');
        return identifier;
    }

    private String parseTrait() {
        this.consume();
        return this.parseIdentifier();
    }

    private String parseIdentifier() {
        int startPosition = this.position;
        char start = this.input.charAt(this.position);
        if (!this.isIdentifierStart(start)) {
            throw new IllegalArgumentException("Expected [A-Za-z_]");
        }
        this.consumeUntilNoLongerMatches(c -> this.isIdentifierStart(c.charValue()) || this.isDigit(c.charValue()) || c.charValue() == '#' || c.charValue() == '$' || c.charValue() == '.');
        return this.input.substring(startPosition, this.position + 1);
    }

    private String parseSingleQuotes() {
        char next;
        int startPosition = this.position;
        while ((next = this.consume()) != '\'') {
            if (next != '\\') continue;
            this.consume();
        }
        return this.parseStringContents(this.input.substring(startPosition, this.position + 1));
    }

    private String parseQuotes() {
        boolean triple;
        int startPosition = this.position;
        boolean bl = triple = this.peekChar() == '\"' && this.peekChar(2) == '\"';
        if (triple) {
            this.consume();
            this.consume();
        }
        while (true) {
            char next;
            if ((next = this.consume()) == '\"' && (!triple || this.peekChar() == '\"' && this.peekChar(2) == '\"')) {
                if (!triple) break;
                this.consume();
                this.consume();
                break;
            }
            if (next != '\\') continue;
            this.consume();
        }
        return this.parseStringContents(this.input.substring(startPosition, this.position + 1));
    }

    private String parseNumber() {
        int startPosition = this.position;
        char current = this.input.charAt(this.position);
        if (current == '0' && this.isDigit(this.peekChar())) {
            throw new IllegalArgumentException("Invalid number");
        }
        this.consumeUntilNoLongerMatches(this::isDigit);
        char peek = this.peekChar();
        if (peek == '.') {
            this.consume();
            if (this.consumeUntilNoLongerMatches(this::isDigit) == 0) {
                throw new IllegalArgumentException("Invalid number");
            }
        }
        if ((peek = this.peekChar()) == 'e' || peek == 'E') {
            this.consume();
            peek = this.peekChar();
            if (peek == '+' || peek == '-') {
                this.consume();
            }
            if (this.consumeUntilNoLongerMatches(this::isDigit) == 0) {
                throw new IllegalArgumentException("Invalid number");
            }
        }
        return this.input.substring(startPosition, this.position + 1);
    }

    private String parseStringContents(String lexeme) {
        int offset = 1;
        if (lexeme.startsWith("\"\"\"")) {
            lexeme = this.formatTextBlock(lexeme);
            offset = 0;
        }
        StringBuilder result = new StringBuilder(lexeme.length() - offset * 2);
        LexerState state = LexerState.NORMAL;
        int hexCount = 0;
        int unicode = 0;
        block18: for (int i = offset; i < lexeme.length() - offset; ++i) {
            char c = lexeme.charAt(i);
            switch (state) {
                case NORMAL: {
                    if (c == '\\') {
                        state = LexerState.AFTER_ESCAPE;
                        continue block18;
                    }
                    result.append(c);
                    continue block18;
                }
                case AFTER_ESCAPE: {
                    state = LexerState.NORMAL;
                    switch (c) {
                        case '\"': {
                            result.append('\"');
                            continue block18;
                        }
                        case '\'': {
                            result.append('\'');
                            continue block18;
                        }
                        case '\\': {
                            result.append('\\');
                            continue block18;
                        }
                        case '/': {
                            result.append('/');
                            continue block18;
                        }
                        case 'b': {
                            result.append('\b');
                            continue block18;
                        }
                        case 'f': {
                            result.append('\f');
                            continue block18;
                        }
                        case 'n': {
                            result.append('\n');
                            continue block18;
                        }
                        case 'r': {
                            result.append('\r');
                            continue block18;
                        }
                        case 't': {
                            result.append('\t');
                            continue block18;
                        }
                        case 'u': {
                            state = LexerState.UNICODE;
                            continue block18;
                        }
                        case '\n': {
                            continue block18;
                        }
                    }
                    throw new IllegalArgumentException("Invalid escape found in string: `\\" + c + "`");
                }
                case UNICODE: {
                    if (c >= '0' && c <= '9') {
                        unicode = unicode << 4 | c - 48;
                    } else if (c >= 'a' && c <= 'f') {
                        unicode = unicode << 4 | 10 + c - 97;
                    } else if (c >= 'A' && c <= 'F') {
                        unicode = unicode << 4 | 10 + c - 65;
                    } else {
                        throw new IllegalArgumentException("Invalid unicode escape character: `" + c + "`");
                    }
                    if (++hexCount != 4) continue block18;
                    result.append((char)unicode);
                    hexCount = 0;
                    state = LexerState.NORMAL;
                    continue block18;
                }
                default: {
                    throw new IllegalStateException("Unreachable");
                }
            }
        }
        if (state == LexerState.UNICODE) {
            throw new IllegalArgumentException("Invalid unclosed unicode escape found in string");
        }
        return result.toString();
    }

    private String formatTextBlock(String lexeme) {
        int i;
        if ((lexeme = lexeme.substring(3, lexeme.length() - 3)).isEmpty()) {
            throw new IllegalArgumentException("Invalid text block: text block is empty");
        }
        if (lexeme.charAt(0) != '\n') {
            throw new IllegalArgumentException("Invalid text block: text block must start with a new line (LF)");
        }
        StringBuilder buffer = new StringBuilder();
        int longestPadding = Integer.MAX_VALUE;
        String[] lines = lexeme.split("\n", -1);
        for (i = 1; i < lines.length; ++i) {
            int padding = this.computeLeadingWhitespace(lines[i], i == lines.length - 1);
            if (padding <= -1 || padding >= longestPadding) continue;
            longestPadding = padding;
        }
        for (i = 1; i < lines.length; ++i) {
            String formattedLine;
            String line = lines[i];
            if (!line.isEmpty() && (formattedLine = SmithyModelLexer.createTextBlockLine(line, longestPadding)) != null) {
                buffer.append(formattedLine);
            }
            if (i >= lines.length - 1) continue;
            buffer.append('\n');
        }
        return buffer.toString();
    }

    private int computeLeadingWhitespace(String line, boolean isLastLine) {
        if (line.isEmpty()) {
            return -1;
        }
        for (int offset = 0; offset < line.length(); ++offset) {
            if (line.charAt(offset) == ' ') continue;
            return offset;
        }
        return isLastLine ? line.length() : -1;
    }

    private static String createTextBlockLine(String line, int longestPadding) {
        int endPosition;
        int startPosition = Math.min(longestPadding, line.length());
        for (endPosition = line.length() - 1; endPosition > 0 && line.charAt(endPosition) == ' '; --endPosition) {
        }
        return endPosition > startPosition ? line.substring(startPosition, endPosition + 1) : null;
    }

    static enum LexerState {
        NORMAL,
        AFTER_ESCAPE,
        UNICODE;

    }

    static final class Token
    implements FromSourceLocation {
        final String filename;
        final TokenType type;
        final String lexeme;
        final String errorMessage;
        final int line;
        final int column;
        final int span;

        Token(String filename, TokenType type, String lexeme, int line, int column, int span, String errorMessage) {
            this.filename = filename;
            this.type = type;
            this.lexeme = lexeme;
            this.line = line;
            this.column = column;
            this.span = span;
            this.errorMessage = errorMessage;
        }

        public String toString() {
            return !StringUtils.isBlank((CharSequence)this.errorMessage) ? String.format("%s(%s, %d:%d)", this.type.name(), this.errorMessage, this.line, this.column) : String.format("%s(%s, %d:%d)", this.type.name(), this.lexeme, this.line, this.column);
        }

        @Override
        public SourceLocation getSourceLocation() {
            return new SourceLocation(this.filename, this.line, this.column);
        }

        public String getDocContents() {
            if (this.type != TokenType.DOC) {
                throw new IllegalStateException("Not a doc comment token");
            }
            return this.lexeme.startsWith("/// ") ? this.lexeme.substring(4) : this.lexeme.substring(3);
        }
    }

    static enum TokenType {
        DOC,
        RETURN,
        CONTROL,
        UNQUOTED,
        LPAREN,
        RPAREN,
        LBRACE,
        RBRACE,
        LBRACKET,
        RBRACKET,
        EQUAL,
        COLON,
        COMMA,
        ANNOTATION,
        QUOTED,
        NUMBER,
        ERROR;

    }
}

