/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.smithy.model.loader;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.logging.Logger;
import software.amazon.smithy.model.FromSourceLocation;
import software.amazon.smithy.model.Model;
import software.amazon.smithy.model.SourceException;
import software.amazon.smithy.model.SourceLocation;
import software.amazon.smithy.model.loader.ModelSyntaxException;
import software.amazon.smithy.model.loader.Prelude;
import software.amazon.smithy.model.loader.UseException;
import software.amazon.smithy.model.node.ArrayNode;
import software.amazon.smithy.model.node.Node;
import software.amazon.smithy.model.shapes.AbstractShapeBuilder;
import software.amazon.smithy.model.shapes.MemberShape;
import software.amazon.smithy.model.shapes.Shape;
import software.amazon.smithy.model.shapes.ShapeId;
import software.amazon.smithy.model.shapes.ShapeIdSyntaxException;
import software.amazon.smithy.model.shapes.ShapeIndex;
import software.amazon.smithy.model.shapes.ShapeType;
import software.amazon.smithy.model.traits.DynamicTrait;
import software.amazon.smithy.model.traits.Trait;
import software.amazon.smithy.model.traits.TraitDefinition;
import software.amazon.smithy.model.traits.TraitFactory;
import software.amazon.smithy.model.validation.Severity;
import software.amazon.smithy.model.validation.ValidatedResult;
import software.amazon.smithy.model.validation.ValidationEvent;
import software.amazon.smithy.utils.SmithyBuilder;

final class LoaderVisitor {
    private static final String[] SUPPORTED_VERSION_PARTS = "0.3.0".split("\\.");
    private static final Logger LOGGER = Logger.getLogger(LoaderVisitor.class.getName());
    private static final TraitDefinition.Provider TRAIT_DEF_PROVIDER = new TraitDefinition.Provider();
    private boolean calledOnEnd;
    private String smithyVersion;
    private String[] smithyVersionParts;
    private SourceLocation versionSourceLocation;
    private final TraitFactory traitFactory;
    private final Map<String, Object> properties;
    private final List<ValidationEvent> events = new ArrayList<ValidationEvent>();
    private final Map<String, Node> metadata = new HashMap<String, Node>();
    private final Map<ShapeId, List<PendingTrait>> pendingTraits = new HashMap<ShapeId, List<PendingTrait>>();
    private final List<ForwardReferenceResolver> forwardReferenceResolvers = new ArrayList<ForwardReferenceResolver>();
    private final Map<ShapeId, AbstractShapeBuilder> pendingShapes = new HashMap<ShapeId, AbstractShapeBuilder>();
    private final Map<ShapeId, Shape> builtShapes = new HashMap<ShapeId, Shape>();
    private final Map<ShapeId, TraitDefinition> builtTraitDefinitions = new HashMap<ShapeId, TraitDefinition>();
    private String namespace;
    private final Map<String, ShapeId> useShapes = new HashMap<String, ShapeId>();

    LoaderVisitor(TraitFactory traitFactory) {
        this(traitFactory, Collections.emptyMap());
    }

    LoaderVisitor(TraitFactory traitFactory, Map<String, Object> properties) {
        this.traitFactory = traitFactory;
        this.properties = properties;
    }

    public boolean hasDefinedShape(ShapeId id) {
        return this.builtShapes.containsKey(id) || this.pendingShapes.containsKey(id);
    }

    public void onOpenFile(String filename) {
        LOGGER.fine(() -> "Beginning to parse " + filename);
        this.namespace = null;
        this.useShapes.clear();
    }

    public String getNamespace() {
        return this.namespace;
    }

    public void onNamespace(String namespace, FromSourceLocation source) {
        if (!ShapeId.isValidNamespace(namespace)) {
            String msg = String.format("Invalid namespace name `%s`", namespace);
            throw new ModelSyntaxException(msg, source);
        }
        this.namespace = namespace;
    }

    public void onShapeTarget(String target, FromSourceLocation sourceLocation, Consumer<ShapeId> resolver) {
        this.assertNamespaceIsPresent(SourceLocation.none());
        try {
            ShapeId expectedId = ShapeId.fromOptionalNamespace(this.namespace, target);
            if (this.namespace.equals("smithy.api") || this.hasDefinedShape(expectedId) || target.contains("#")) {
                resolver.accept(expectedId);
            } else if (this.useShapes.containsKey(target)) {
                resolver.accept(this.useShapes.get(target));
            } else {
                this.forwardReferenceResolvers.add(new ForwardReferenceResolver(expectedId, resolver));
            }
        }
        catch (ShapeIdSyntaxException e) {
            throw new SourceException("Error resolving shape target; " + e.getMessage(), sourceLocation, e);
        }
    }

    public boolean hasProperty(String property) {
        return this.properties.containsKey(property);
    }

    public Optional<Object> getProperty(String property) {
        return Optional.ofNullable(this.properties.get(property));
    }

    public <T> Optional<T> getProperty(String property, Class<T> type) {
        return this.getProperty(property).map(value -> {
            if (!type.isInstance(value)) {
                throw new ClassCastException(String.format("Expected `%s` property of the LoaderVisitor to be a `%s`, but found a `%s`", property, type.getName(), value.getClass().getName()));
            }
            return value;
        });
    }

    public void onVersion(SourceLocation sourceLocation, String smithyVersion) {
        if (this.smithyVersion != null && this.smithyVersion.equals(smithyVersion)) {
            return;
        }
        CharSequence[] versionParts = smithyVersion.split("\\.");
        LoaderVisitor.validateVersionNumber(sourceLocation, (String[])versionParts);
        if (this.smithyVersion != null && !LoaderVisitor.areVersionsCompatible(this.smithyVersionParts, (String[])versionParts)) {
            throw new SourceException(String.format("Cannot set Smithy version to `%s` because it was previously set to an incompatible version `%s` in %s", smithyVersion, this.smithyVersion, this.versionSourceLocation), sourceLocation);
        }
        if (!LoaderVisitor.isSupportedVersion((String[])versionParts)) {
            throw new SourceException(String.format("Invalid Smithy version provided: `%s`. Expected a version compatible with the tooling version of `%s`. Perhaps you need to update your version of Smithy?", String.join((CharSequence)".", versionParts), "0.3.0"), sourceLocation);
        }
        this.smithyVersion = smithyVersion;
        this.smithyVersionParts = versionParts;
        this.versionSourceLocation = sourceLocation;
        this.validateState(sourceLocation);
    }

    private static void validateVersionNumber(SourceLocation sourceLocation, String[] versionParts) {
        if (versionParts.length < 2 || versionParts.length > 3) {
            throw new SourceException("Smithy version number should have 2 or 3 parts: " + String.join((CharSequence)".", versionParts), sourceLocation);
        }
        for (String part : versionParts) {
            if (part.isEmpty()) {
                throw new SourceException("Invalid Smithy version number: " + String.join((CharSequence)".", versionParts), sourceLocation);
            }
            for (int i = 0; i < part.length(); ++i) {
                if (Character.isDigit(part.charAt(i))) continue;
                throw new SourceException("Invalid Smithy version number: " + String.join((CharSequence)".", versionParts), sourceLocation);
            }
        }
    }

    private static boolean areVersionsCompatible(String[] left, String[] right) {
        if (!left[0].equals(right[0])) {
            return false;
        }
        return !left[0].equals("0") || left[1].equals(right[1]);
    }

    private static boolean isSupportedVersion(String[] version) {
        if (!LoaderVisitor.areVersionsCompatible(SUPPORTED_VERSION_PARTS, version)) {
            return false;
        }
        return version[1].equals(SUPPORTED_VERSION_PARTS[1]) || Integer.parseInt(version[1]) < Integer.parseInt(SUPPORTED_VERSION_PARTS[1]);
    }

    private void validateState(FromSourceLocation sourceLocation) {
        if (this.calledOnEnd) {
            throw new IllegalStateException("Cannot call visitor method because visitor has called onEnd");
        }
        if (this.smithyVersion == null) {
            LOGGER.warning(String.format("No Smithy version explicitly specified in %s, so assuming version of %s", sourceLocation.getSourceLocation().getFilename(), "0.3.0"));
            this.onVersion(sourceLocation.getSourceLocation(), "0.3.0");
        }
    }

    public void onError(ValidationEvent event) {
        this.events.add(Objects.requireNonNull(event));
    }

    public ShapeId onShapeDefName(String name, FromSourceLocation source) {
        this.validateState(source);
        this.assertNamespaceIsPresent(source);
        if (this.useShapes.containsKey(name)) {
            String msg = String.format("shape name `%s` conflicts with imported shape `%s`", name, this.useShapes.get(name));
            throw new UseException(msg, source);
        }
        try {
            return ShapeId.fromRelative(this.namespace, name);
        }
        catch (ShapeIdSyntaxException e) {
            throw new ModelSyntaxException("Invalid shape name: " + name, source);
        }
    }

    private void assertNamespaceIsPresent(FromSourceLocation source) {
        if (this.namespace == null) {
            throw new ModelSyntaxException("A namespace must be set before shapes or traits can be defined", source);
        }
    }

    public void onShape(AbstractShapeBuilder shapeBuilder) {
        this.validateState(shapeBuilder);
        ShapeId id = (ShapeId)SmithyBuilder.requiredState((String)"id", (Object)shapeBuilder.getId());
        if (this.validateOnShape(id, shapeBuilder)) {
            this.pendingShapes.put(id, shapeBuilder);
        }
    }

    public void onShape(Shape shape) {
        this.validateState(shape);
        if (this.validateOnShape(shape.getId(), shape)) {
            this.builtShapes.put(shape.getId(), shape);
        }
        shape.getTrait(TraitDefinition.class).ifPresent(def -> this.onTraitDefinition(shape.getId(), (TraitDefinition)def));
    }

    private void onTraitDefinition(ShapeId target, TraitDefinition definition) {
        this.builtTraitDefinitions.put(target, definition);
    }

    private boolean validateOnShape(ShapeId id, FromSourceLocation source) {
        if (!this.hasDefinedShape(id)) {
            return true;
        }
        if (!Prelude.isPreludeShape(id)) {
            SourceLocation previous = Optional.ofNullable((FromSourceLocation)this.pendingShapes.get(id)).orElseGet(() -> this.builtShapes.get(id)).getSourceLocation();
            if (previous != SourceLocation.NONE && previous.equals(source.getSourceLocation())) {
                LOGGER.warning(() -> "Ignoring duplicate shape definition defined in the same file: " + id + " defined at " + source.getSourceLocation());
            } else {
                this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(source).message(String.format("Duplicate shape definition for `%s` found at `%s` and `%s`", id, previous.getSourceLocation(), source.getSourceLocation())).build());
            }
        }
        return false;
    }

    public void onTrait(ShapeId target, String traitName, Node traitValue) {
        this.onShapeTarget(traitName, traitValue.getSourceLocation(), id -> {
            if (id.equals(TraitDefinition.ID)) {
                TraitDefinition traitDef = TRAIT_DEF_PROVIDER.createTrait(target, traitValue);
                this.onTraitDefinition(target, traitDef);
                this.onTrait(target, traitDef);
            } else {
                PendingTrait pendingTrait = new PendingTrait((ShapeId)id, traitValue);
                this.pendingTraits.computeIfAbsent(target, targetId -> new ArrayList()).add(pendingTrait);
            }
        });
    }

    public void onTrait(ShapeId target, Trait trait) {
        PendingTrait pending = new PendingTrait(target, trait);
        this.pendingTraits.computeIfAbsent(target, targetId -> new ArrayList()).add(pending);
    }

    public void onMetadata(String key, Node value) {
        this.validateState(value);
        if (!this.metadata.containsKey(key)) {
            this.metadata.put(key, value);
        } else if (this.metadata.get(key).isArrayNode() && value.isArrayNode()) {
            ArrayNode previous = this.metadata.get(key).expectArrayNode();
            ArrayList<Node> merged = new ArrayList<Node>(previous.getElements());
            merged.addAll(value.expectArrayNode().getElements());
            ArrayNode mergedArray = new ArrayNode(merged, value.getSourceLocation());
            this.metadata.put(key, mergedArray);
        } else if (!this.metadata.get(key).equals(value)) {
            this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(value).message(String.format("Metadata conflict for key `%s`. Defined in both `%s` and `%s`", key, value.getSourceLocation(), this.metadata.get(key).getSourceLocation())).build());
        } else {
            LOGGER.fine(() -> "Ignoring duplicate metadata definition of " + key);
        }
    }

    void onUseShape(ShapeId id, FromSourceLocation location) {
        this.validateState(location);
        ShapeId previous = this.useShapes.put(id.getName(), id);
        if (previous != null) {
            throw new UseException(String.format("Cannot use name `%s` because it conflicts with `%s`", id, previous), location);
        }
    }

    public ValidatedResult<Model> onEnd() {
        this.validateState(SourceLocation.NONE);
        this.calledOnEnd = true;
        Model.Builder modelBuilder = Model.builder().smithyVersion(this.smithyVersion).metadata(this.metadata);
        ShapeIndex.Builder shapeIndexBuilder = ShapeIndex.builder();
        this.finalizeShapeTargets();
        this.finalizePendingTraits();
        ArrayList<ShapeId> needsConversion = new ArrayList<ShapeId>();
        for (AbstractShapeBuilder builder : this.pendingShapes.values()) {
            if (!(builder instanceof MemberShape.Builder)) continue;
            needsConversion.add(builder.getId().withoutMember());
        }
        needsConversion.forEach(this::resolveShapeBuilder);
        for (AbstractShapeBuilder shape : this.pendingShapes.values()) {
            MemberShape member;
            if (shape.getClass() != MemberShape.Builder.class || (member = (MemberShape)this.buildShape(shapeIndexBuilder, shape)) == null) continue;
            AbstractShapeBuilder container = this.pendingShapes.get(shape.getId().withoutMember());
            if (container == null) {
                this.events.add(ValidationEvent.builder().shape(member).eventId("Model").severity(Severity.ERROR).message(String.format("Member shape `%s` added to non-existent shape", member.getId())).build());
                continue;
            }
            container.addMember(member);
        }
        for (AbstractShapeBuilder shape : this.pendingShapes.values()) {
            if (shape.getClass() == MemberShape.Builder.class) continue;
            this.buildShape(shapeIndexBuilder, shape);
        }
        shapeIndexBuilder.addShapes(this.builtShapes.values());
        modelBuilder.shapeIndex(shapeIndexBuilder.build());
        return new ValidatedResult<Model>(modelBuilder.build(), this.events);
    }

    private void finalizeShapeTargets() {
        for (ForwardReferenceResolver resolver : this.forwardReferenceResolvers) {
            ShapeId preludeId;
            if (!this.hasDefinedShape(resolver.expectedId) && Prelude.isPublicPreludeShape(preludeId = ShapeId.fromParts("smithy.api", resolver.expectedId.asRelativeReference()))) {
                resolver.consumer.accept(preludeId);
                continue;
            }
            resolver.consumer.accept(resolver.expectedId);
        }
        this.forwardReferenceResolvers.clear();
    }

    private void finalizePendingTraits() {
        for (Map.Entry<ShapeId, List<PendingTrait>> entry : this.pendingTraits.entrySet()) {
            ShapeId target = entry.getKey();
            List<PendingTrait> pendingTraits = entry.getValue();
            AbstractShapeBuilder builder = this.resolveShapeBuilder(target);
            if (builder == null) {
                this.emitErrorsForEachInvalidTraitTarget(target, pendingTraits);
                continue;
            }
            for (PendingTrait pendingTrait : pendingTraits) {
                if (pendingTrait.trait == null) continue;
                builder.addTrait(pendingTrait.trait);
            }
            for (Map.Entry entry2 : this.computeTraits(builder, pendingTraits).entrySet()) {
                this.createAndApplyTraitToShape(builder, (ShapeId)entry2.getKey(), (Node)entry2.getValue());
            }
        }
    }

    private AbstractShapeBuilder resolveShapeBuilder(ShapeId id) {
        if (this.pendingShapes.containsKey(id)) {
            return this.pendingShapes.get(id);
        }
        if (this.builtShapes.containsKey(id)) {
            Object builder = Shape.shapeToBuilder(this.builtShapes.remove(id));
            this.pendingShapes.put(id, (AbstractShapeBuilder)builder);
            return builder;
        }
        return null;
    }

    private void emitErrorsForEachInvalidTraitTarget(ShapeId target, List<PendingTrait> pendingTraits) {
        for (PendingTrait pendingTrait : pendingTraits) {
            this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(pendingTrait.value.getSourceLocation()).message(String.format("Trait `%s` applied to unknown shape `%s`", Trait.getIdiomaticTraitName(pendingTrait.id), target)).build());
        }
    }

    private Shape buildShape(ShapeIndex.Builder shapeIndexBuilder, AbstractShapeBuilder shapeBuilder) {
        try {
            Shape result = (Shape)shapeBuilder.build();
            shapeIndexBuilder.addShape(result);
            return result;
        }
        catch (SourceException e) {
            this.onError(ValidationEvent.fromSourceException(e).toBuilder().shapeId(shapeBuilder.getId()).build());
            return null;
        }
    }

    private Map<ShapeId, Node> computeTraits(AbstractShapeBuilder shapeBuilder, List<PendingTrait> pending) {
        HashMap<ShapeId, Node> traits = new HashMap<ShapeId, Node>();
        for (PendingTrait trait : pending) {
            if (trait.trait != null) continue;
            TraitDefinition definition = this.builtTraitDefinitions.get(trait.id);
            if (definition == null) {
                this.onUnresolvedTraitName(shapeBuilder, trait);
                continue;
            }
            ShapeId traitId = trait.id;
            Node value = this.coerceTraitValue(trait.value, trait.id);
            Node previous = (Node)traits.get(traitId);
            if (previous == null) {
                traits.put(traitId, value);
                continue;
            }
            if (previous.isArrayNode() && value.isArrayNode()) {
                traits.put(traitId, value.asArrayNode().get().merge(previous.asArrayNode().get()));
                continue;
            }
            if (previous.equals(value)) {
                LOGGER.fine(() -> String.format("Ignoring duplicate %s trait value on %s", traitId, shapeBuilder.getId()));
                continue;
            }
            this.onDuplicateTrait(shapeBuilder.getId(), traitId, previous, value);
        }
        return traits;
    }

    private Node coerceTraitValue(Node value, ShapeId traitId) {
        return Trait.coerceTraitValue(value, this.determineTraitDefinitionType(traitId));
    }

    private ShapeType determineTraitDefinitionType(ShapeId traitId) {
        if (this.pendingShapes.containsKey(traitId)) {
            return this.pendingShapes.get(traitId).getShapeType();
        }
        if (this.builtShapes.containsKey(traitId)) {
            return this.builtShapes.get(traitId).getType();
        }
        throw new IllegalStateException("Trait definition trait is applied to a non-existent shape: " + traitId);
    }

    private void onDuplicateTrait(ShapeId target, ShapeId traitName, FromSourceLocation previous, Node duplicate) {
        this.onError(ValidationEvent.builder().eventId("Model").severity(Severity.ERROR).sourceLocation(duplicate.getSourceLocation()).shapeId(target).message(String.format("Conflicting `%s` trait found on shape `%s`. The previous trait was defined at `%s`, and a conflicting trait was defined at `%s`.", traitName, target, previous.getSourceLocation(), duplicate.getSourceLocation())).build());
    }

    private void onUnresolvedTraitName(AbstractShapeBuilder shapeBuilder, PendingTrait trait) {
        Severity severity = this.getProperty("assembler.allowUnknownTraits", Boolean.class).orElse(false) != false ? Severity.WARNING : Severity.ERROR;
        this.onError(ValidationEvent.builder().eventId("Model").severity(severity).sourceLocation(trait.value.getSourceLocation()).shapeId(shapeBuilder.getId()).message(String.format("Unable to resolve trait `%s`. If this is a custom trait, then it must be defined before it can be used in a model.", trait.id)).build());
    }

    private void createAndApplyTraitToShape(AbstractShapeBuilder shapeBuilder, ShapeId traitId, Node traitValue) {
        try {
            Trait createdTrait = this.traitFactory.createTrait(traitId, shapeBuilder.getId(), traitValue).orElseGet(() -> new DynamicTrait(traitId, traitValue));
            shapeBuilder.addTrait(createdTrait);
        }
        catch (SourceException e) {
            this.events.add(ValidationEvent.fromSourceException(e, String.format("Error creating trait `%s`: ", Trait.getIdiomaticTraitName(traitId))).toBuilder().shapeId(shapeBuilder.getId()).build());
        }
    }

    private static final class ForwardReferenceResolver {
        final ShapeId expectedId;
        final Consumer<ShapeId> consumer;

        ForwardReferenceResolver(ShapeId expectedId, Consumer<ShapeId> consumer) {
            this.expectedId = expectedId;
            this.consumer = consumer;
        }
    }

    private static final class PendingTrait {
        final ShapeId id;
        final Node value;
        final Trait trait;

        PendingTrait(ShapeId id, Node value) {
            this.id = id;
            this.value = value;
            this.trait = null;
        }

        PendingTrait(ShapeId id, Trait trait) {
            this.id = id;
            this.trait = trait;
            this.value = null;
        }
    }
}

