/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.documentdb.jdbc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.mongodb.MongoException;
import com.mongodb.client.AggregateIterable;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Instant;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.documentdb.jdbc.DocumentDbAllowDiskUseOption;
import software.amazon.documentdb.jdbc.DocumentDbConnection;
import software.amazon.documentdb.jdbc.DocumentDbConnectionProperties;
import software.amazon.documentdb.jdbc.DocumentDbResultSet;
import software.amazon.documentdb.jdbc.common.utilities.JdbcColumnMetaData;
import software.amazon.documentdb.jdbc.common.utilities.SqlError;
import software.amazon.documentdb.jdbc.common.utilities.SqlState;
import software.amazon.documentdb.jdbc.query.DocumentDbMqlQueryContext;
import software.amazon.documentdb.jdbc.query.DocumentDbQueryMappingService;

public class DocumentDbQueryExecutor {
    private static final int OPERATION_CANCELLED_CODE = 11601;
    private static final Logger LOGGER = LoggerFactory.getLogger(DocumentDbQueryExecutor.class);
    private final Object queryStateLock = new Object();
    private final Statement statement;
    private final DocumentDbConnectionProperties connectionProperties;
    private final DocumentDbQueryMappingService queryMapper;
    private int fetchSize;
    private int queryTimeout;
    private DocumentDbAllowDiskUseOption allowDiskUse;
    private String queryId = null;
    private QueryState queryState = QueryState.NOT_STARTED;

    DocumentDbQueryExecutor(Statement statement, DocumentDbConnectionProperties connectionProperties, DocumentDbQueryMappingService queryMapper, int queryTimeoutSecs, int fetchSize) {
        this.statement = statement;
        this.connectionProperties = connectionProperties;
        this.queryMapper = queryMapper;
        this.fetchSize = fetchSize;
        this.queryTimeout = queryTimeoutSecs;
        this.allowDiskUse = connectionProperties.getAllowDiskUseOption();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void cancelQuery(boolean isClosing) throws SQLException {
        Object object = this.queryStateLock;
        synchronized (object) {
            if (this.queryState.equals((Object)QueryState.CANCELED)) {
                return;
            }
            if (this.queryState.equals((Object)QueryState.NOT_STARTED)) {
                if (isClosing) {
                    return;
                }
                throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_NOT_STARTED_OR_COMPLETE, new Object[0]);
            }
            this.performCancel();
            this.queryState = QueryState.CANCELED;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ResultSet executeQuery(String query) throws SQLException {
        Object object = this.queryStateLock;
        synchronized (object) {
            if (this.queryState.equals((Object)QueryState.IN_PROGRESS)) {
                throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_IN_PROGRESS, new Object[0]);
            }
            this.queryState = QueryState.IN_PROGRESS;
            this.queryId = UUID.randomUUID().toString();
        }
        try {
            ResultSet resultSet = this.runQuery(query);
            Object object2 = this.queryStateLock;
            synchronized (object2) {
                if (this.queryState.equals((Object)QueryState.CANCELED)) {
                    this.resetQueryState();
                    throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_CANCELED, new Object[0]);
                }
            }
            object2 = resultSet;
            return object2;
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            Object object3 = this.queryStateLock;
            synchronized (object3) {
                if (e instanceof MongoException && ((MongoException)((Object)e)).getCode() == 11601 && this.queryState.equals((Object)QueryState.CANCELED)) {
                    throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_CANCELED, new Object[0]);
                }
                throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_FAILED, e);
            }
        }
        finally {
            this.resetQueryState();
        }
    }

    @VisibleForTesting
    protected ResultSet runQuery(String sql) throws SQLException {
        Instant beginTranslation = Instant.now();
        LOGGER.info("Query {}: Beginning translation of query.", (Object)this.queryId);
        LOGGER.debug("Query {}: {}", (Object)this.queryId, (Object)sql);
        long maxRows = this.statement.getLargeMaxRows();
        DocumentDbMqlQueryContext queryContext = this.queryMapper.get(sql, maxRows);
        LOGGER.info("Query {}: Took {} ms to translate query.", (Object)this.queryId, (Object)(Instant.now().toEpochMilli() - beginTranslation.toEpochMilli()));
        if (!(this.statement.getConnection() instanceof DocumentDbConnection)) {
            throw new SQLException("Unexpected operation state.");
        }
        Instant beginExecution = Instant.now();
        DocumentDbConnection connection = (DocumentDbConnection)this.statement.getConnection();
        DocumentDbConnectionProperties properties = connection.getConnectionProperties();
        MongoClient client = connection.getMongoClient();
        MongoDatabase database = client.getDatabase(properties.getDatabase());
        MongoCollection collection = database.getCollection(queryContext.getCollectionName());
        List<Bson> aggregateOperations = queryContext.getAggregateOperations();
        AggregateIterable iterable = collection.aggregate(aggregateOperations);
        if (this.getQueryTimeout() > 0) {
            iterable = iterable.maxTime((long)this.getQueryTimeout(), TimeUnit.SECONDS);
        }
        if (this.getFetchSize() > 0) {
            iterable = iterable.batchSize(this.getFetchSize());
        }
        if (this.getAllowDiskUse() == DocumentDbAllowDiskUseOption.ENABLE) {
            iterable = iterable.allowDiskUse(Boolean.valueOf(true));
        } else if (this.getAllowDiskUse() == DocumentDbAllowDiskUseOption.DISABLE) {
            iterable = iterable.allowDiskUse(Boolean.valueOf(false));
        }
        ImmutableList columnMetaData = ImmutableList.copyOf(queryContext.getColumnMetaData());
        MongoCursor iterator = iterable.iterator();
        LOGGER.info("Query {}: Took {} ms to execute query and retrieve first batch of results.", (Object)this.queryId, (Object)(Instant.now().toEpochMilli() - beginExecution.toEpochMilli()));
        LOGGER.debug("Query {}: Executed on collection {} with following pipeline operations: {}", new Object[]{this.queryId, queryContext.getCollectionName(), queryContext.getAggregateOperations().toString()});
        return new DocumentDbResultSet(this.statement, (MongoCursor<Document>)iterator, (ImmutableList<JdbcColumnMetaData>)columnMetaData, queryContext.getPaths());
    }

    private void resetQueryState() {
        this.queryState = QueryState.NOT_STARTED;
        this.queryId = null;
    }

    private void performCancel() throws SQLException {
        try (MongoClient client = this.connectionProperties.createMongoClient();){
            MongoDatabase database = client.getDatabase("admin");
            Document currentOp = database.runCommand((Bson)new Document("currentOp", (Object)1).append("$ownOps", (Object)true).append("command.comment", (Object)this.queryId));
            if (!(currentOp.get((Object)"inprog") instanceof List)) {
                throw new SQLException("Unexpected operation state.");
            }
            List ops = (List)currentOp.get((Object)"inprog");
            if (ops.isEmpty()) {
                throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_NOT_STARTED_OR_COMPLETE, new Object[0]);
            }
            if (ops.size() != 1) {
                throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_CANNOT_BE_CANCELED, "More than one running operation matched the query ID.");
            }
            if (!(ops.get(0) instanceof Document)) {
                throw new SQLException("Unexpected operation state.");
            }
            Object opId = ((Document)ops.get(0)).get((Object)"opid");
            if (opId == null) {
                throw new SQLException("Unexpected operation state.");
            }
            Document killOp = database.runCommand((Bson)new Document("killOp", (Object)1).append("op", opId));
            if (!killOp.get((Object)"ok").equals(1.0)) {
                throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_CANNOT_BE_CANCELED, killOp.get((Object)"info"));
            }
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            throw SqlError.createSQLException(LOGGER, SqlState.OPERATION_CANCELED, SqlError.QUERY_CANNOT_BE_CANCELED, e);
        }
    }

    protected String getQueryId() {
        return this.queryId;
    }

    protected int getQueryTimeout() {
        return this.queryTimeout;
    }

    protected void setQueryTimeout(int queryTimeout) {
        this.queryTimeout = queryTimeout;
    }

    protected int getFetchSize() {
        return this.fetchSize;
    }

    protected void setFetchSize(int fetchSize) {
        this.fetchSize = fetchSize;
    }

    protected DocumentDbAllowDiskUseOption getAllowDiskUse() {
        return this.allowDiskUse;
    }

    protected void setAllowDiskUse(DocumentDbAllowDiskUseOption allowDiskUse) {
        this.allowDiskUse = allowDiskUse;
    }

    private static enum QueryState {
        NOT_STARTED,
        IN_PROGRESS,
        CANCELED;

    }
}

