/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.xray.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * X-Ray reevaluates insights periodically until they are resolved, and records each intermediate state in an event. You
 * can review incident events in the Impact Timeline on the Inspect page in the X-Ray console.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InsightEvent implements SdkPojo, Serializable, ToCopyableBuilder<InsightEvent.Builder, InsightEvent> {
    private static final SdkField<String> SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Summary")
            .getter(getter(InsightEvent::summary)).setter(setter(Builder::summary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Summary").build()).build();

    private static final SdkField<Instant> EVENT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EventTime").getter(getter(InsightEvent::eventTime)).setter(setter(Builder::eventTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventTime").build()).build();

    private static final SdkField<RequestImpactStatistics> CLIENT_REQUEST_IMPACT_STATISTICS_FIELD = SdkField
            .<RequestImpactStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("ClientRequestImpactStatistics")
            .getter(getter(InsightEvent::clientRequestImpactStatistics))
            .setter(setter(Builder::clientRequestImpactStatistics))
            .constructor(RequestImpactStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestImpactStatistics")
                    .build()).build();

    private static final SdkField<RequestImpactStatistics> ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD = SdkField
            .<RequestImpactStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("RootCauseServiceRequestImpactStatistics")
            .getter(getter(InsightEvent::rootCauseServiceRequestImpactStatistics))
            .setter(setter(Builder::rootCauseServiceRequestImpactStatistics))
            .constructor(RequestImpactStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("RootCauseServiceRequestImpactStatistics").build()).build();

    private static final SdkField<List<AnomalousService>> TOP_ANOMALOUS_SERVICES_FIELD = SdkField
            .<List<AnomalousService>> builder(MarshallingType.LIST)
            .memberName("TopAnomalousServices")
            .getter(getter(InsightEvent::topAnomalousServices))
            .setter(setter(Builder::topAnomalousServices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TopAnomalousServices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnomalousService> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnomalousService::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUMMARY_FIELD,
            EVENT_TIME_FIELD, CLIENT_REQUEST_IMPACT_STATISTICS_FIELD, ROOT_CAUSE_SERVICE_REQUEST_IMPACT_STATISTICS_FIELD,
            TOP_ANOMALOUS_SERVICES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String summary;

    private final Instant eventTime;

    private final RequestImpactStatistics clientRequestImpactStatistics;

    private final RequestImpactStatistics rootCauseServiceRequestImpactStatistics;

    private final List<AnomalousService> topAnomalousServices;

    private InsightEvent(BuilderImpl builder) {
        this.summary = builder.summary;
        this.eventTime = builder.eventTime;
        this.clientRequestImpactStatistics = builder.clientRequestImpactStatistics;
        this.rootCauseServiceRequestImpactStatistics = builder.rootCauseServiceRequestImpactStatistics;
        this.topAnomalousServices = builder.topAnomalousServices;
    }

    /**
     * <p>
     * A brief description of the event.
     * </p>
     * 
     * @return A brief description of the event.
     */
    public final String summary() {
        return summary;
    }

    /**
     * <p>
     * The time, in Unix seconds, at which the event was recorded.
     * </p>
     * 
     * @return The time, in Unix seconds, at which the event was recorded.
     */
    public final Instant eventTime() {
        return eventTime;
    }

    /**
     * <p>
     * The impact statistics of the client side service. This includes the number of requests to the client service and
     * whether the requests were faults or okay.
     * </p>
     * 
     * @return The impact statistics of the client side service. This includes the number of requests to the client
     *         service and whether the requests were faults or okay.
     */
    public final RequestImpactStatistics clientRequestImpactStatistics() {
        return clientRequestImpactStatistics;
    }

    /**
     * <p>
     * The impact statistics of the root cause service. This includes the number of requests to the client service and
     * whether the requests were faults or okay.
     * </p>
     * 
     * @return The impact statistics of the root cause service. This includes the number of requests to the client
     *         service and whether the requests were faults or okay.
     */
    public final RequestImpactStatistics rootCauseServiceRequestImpactStatistics() {
        return rootCauseServiceRequestImpactStatistics;
    }

    /**
     * Returns true if the TopAnomalousServices property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasTopAnomalousServices() {
        return topAnomalousServices != null && !(topAnomalousServices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The service during the event that is most impacted by the incident.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTopAnomalousServices()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The service during the event that is most impacted by the incident.
     */
    public final List<AnomalousService> topAnomalousServices() {
        return topAnomalousServices;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(summary());
        hashCode = 31 * hashCode + Objects.hashCode(eventTime());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestImpactStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(rootCauseServiceRequestImpactStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(hasTopAnomalousServices() ? topAnomalousServices() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InsightEvent)) {
            return false;
        }
        InsightEvent other = (InsightEvent) obj;
        return Objects.equals(summary(), other.summary()) && Objects.equals(eventTime(), other.eventTime())
                && Objects.equals(clientRequestImpactStatistics(), other.clientRequestImpactStatistics())
                && Objects.equals(rootCauseServiceRequestImpactStatistics(), other.rootCauseServiceRequestImpactStatistics())
                && hasTopAnomalousServices() == other.hasTopAnomalousServices()
                && Objects.equals(topAnomalousServices(), other.topAnomalousServices());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InsightEvent").add("Summary", summary()).add("EventTime", eventTime())
                .add("ClientRequestImpactStatistics", clientRequestImpactStatistics())
                .add("RootCauseServiceRequestImpactStatistics", rootCauseServiceRequestImpactStatistics())
                .add("TopAnomalousServices", hasTopAnomalousServices() ? topAnomalousServices() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Summary":
            return Optional.ofNullable(clazz.cast(summary()));
        case "EventTime":
            return Optional.ofNullable(clazz.cast(eventTime()));
        case "ClientRequestImpactStatistics":
            return Optional.ofNullable(clazz.cast(clientRequestImpactStatistics()));
        case "RootCauseServiceRequestImpactStatistics":
            return Optional.ofNullable(clazz.cast(rootCauseServiceRequestImpactStatistics()));
        case "TopAnomalousServices":
            return Optional.ofNullable(clazz.cast(topAnomalousServices()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InsightEvent, T> g) {
        return obj -> g.apply((InsightEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InsightEvent> {
        /**
         * <p>
         * A brief description of the event.
         * </p>
         * 
         * @param summary
         *        A brief description of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder summary(String summary);

        /**
         * <p>
         * The time, in Unix seconds, at which the event was recorded.
         * </p>
         * 
         * @param eventTime
         *        The time, in Unix seconds, at which the event was recorded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTime(Instant eventTime);

        /**
         * <p>
         * The impact statistics of the client side service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * 
         * @param clientRequestImpactStatistics
         *        The impact statistics of the client side service. This includes the number of requests to the client
         *        service and whether the requests were faults or okay.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestImpactStatistics(RequestImpactStatistics clientRequestImpactStatistics);

        /**
         * <p>
         * The impact statistics of the client side service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * This is a convenience that creates an instance of the {@link RequestImpactStatistics.Builder} avoiding the
         * need to create one manually via {@link RequestImpactStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link RequestImpactStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #clientRequestImpactStatistics(RequestImpactStatistics)}.
         * 
         * @param clientRequestImpactStatistics
         *        a consumer that will call methods on {@link RequestImpactStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clientRequestImpactStatistics(RequestImpactStatistics)
         */
        default Builder clientRequestImpactStatistics(Consumer<RequestImpactStatistics.Builder> clientRequestImpactStatistics) {
            return clientRequestImpactStatistics(RequestImpactStatistics.builder().applyMutation(clientRequestImpactStatistics)
                    .build());
        }

        /**
         * <p>
         * The impact statistics of the root cause service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * 
         * @param rootCauseServiceRequestImpactStatistics
         *        The impact statistics of the root cause service. This includes the number of requests to the client
         *        service and whether the requests were faults or okay.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootCauseServiceRequestImpactStatistics(RequestImpactStatistics rootCauseServiceRequestImpactStatistics);

        /**
         * <p>
         * The impact statistics of the root cause service. This includes the number of requests to the client service
         * and whether the requests were faults or okay.
         * </p>
         * This is a convenience that creates an instance of the {@link RequestImpactStatistics.Builder} avoiding the
         * need to create one manually via {@link RequestImpactStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link RequestImpactStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #rootCauseServiceRequestImpactStatistics(RequestImpactStatistics)}.
         * 
         * @param rootCauseServiceRequestImpactStatistics
         *        a consumer that will call methods on {@link RequestImpactStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rootCauseServiceRequestImpactStatistics(RequestImpactStatistics)
         */
        default Builder rootCauseServiceRequestImpactStatistics(
                Consumer<RequestImpactStatistics.Builder> rootCauseServiceRequestImpactStatistics) {
            return rootCauseServiceRequestImpactStatistics(RequestImpactStatistics.builder()
                    .applyMutation(rootCauseServiceRequestImpactStatistics).build());
        }

        /**
         * <p>
         * The service during the event that is most impacted by the incident.
         * </p>
         * 
         * @param topAnomalousServices
         *        The service during the event that is most impacted by the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topAnomalousServices(Collection<AnomalousService> topAnomalousServices);

        /**
         * <p>
         * The service during the event that is most impacted by the incident.
         * </p>
         * 
         * @param topAnomalousServices
         *        The service during the event that is most impacted by the incident.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topAnomalousServices(AnomalousService... topAnomalousServices);

        /**
         * <p>
         * The service during the event that is most impacted by the incident.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AnomalousService>.Builder} avoiding the
         * need to create one manually via {@link List<AnomalousService>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AnomalousService>.Builder#build()} is called immediately and
         * its result is passed to {@link #topAnomalousServices(List<AnomalousService>)}.
         * 
         * @param topAnomalousServices
         *        a consumer that will call methods on {@link List<AnomalousService>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #topAnomalousServices(List<AnomalousService>)
         */
        Builder topAnomalousServices(Consumer<AnomalousService.Builder>... topAnomalousServices);
    }

    static final class BuilderImpl implements Builder {
        private String summary;

        private Instant eventTime;

        private RequestImpactStatistics clientRequestImpactStatistics;

        private RequestImpactStatistics rootCauseServiceRequestImpactStatistics;

        private List<AnomalousService> topAnomalousServices = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InsightEvent model) {
            summary(model.summary);
            eventTime(model.eventTime);
            clientRequestImpactStatistics(model.clientRequestImpactStatistics);
            rootCauseServiceRequestImpactStatistics(model.rootCauseServiceRequestImpactStatistics);
            topAnomalousServices(model.topAnomalousServices);
        }

        public final String getSummary() {
            return summary;
        }

        @Override
        public final Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public final void setSummary(String summary) {
            this.summary = summary;
        }

        public final Instant getEventTime() {
            return eventTime;
        }

        @Override
        public final Builder eventTime(Instant eventTime) {
            this.eventTime = eventTime;
            return this;
        }

        public final void setEventTime(Instant eventTime) {
            this.eventTime = eventTime;
        }

        public final RequestImpactStatistics.Builder getClientRequestImpactStatistics() {
            return clientRequestImpactStatistics != null ? clientRequestImpactStatistics.toBuilder() : null;
        }

        @Override
        public final Builder clientRequestImpactStatistics(RequestImpactStatistics clientRequestImpactStatistics) {
            this.clientRequestImpactStatistics = clientRequestImpactStatistics;
            return this;
        }

        public final void setClientRequestImpactStatistics(RequestImpactStatistics.BuilderImpl clientRequestImpactStatistics) {
            this.clientRequestImpactStatistics = clientRequestImpactStatistics != null ? clientRequestImpactStatistics.build()
                    : null;
        }

        public final RequestImpactStatistics.Builder getRootCauseServiceRequestImpactStatistics() {
            return rootCauseServiceRequestImpactStatistics != null ? rootCauseServiceRequestImpactStatistics.toBuilder() : null;
        }

        @Override
        public final Builder rootCauseServiceRequestImpactStatistics(
                RequestImpactStatistics rootCauseServiceRequestImpactStatistics) {
            this.rootCauseServiceRequestImpactStatistics = rootCauseServiceRequestImpactStatistics;
            return this;
        }

        public final void setRootCauseServiceRequestImpactStatistics(
                RequestImpactStatistics.BuilderImpl rootCauseServiceRequestImpactStatistics) {
            this.rootCauseServiceRequestImpactStatistics = rootCauseServiceRequestImpactStatistics != null ? rootCauseServiceRequestImpactStatistics
                    .build() : null;
        }

        public final Collection<AnomalousService.Builder> getTopAnomalousServices() {
            if (topAnomalousServices instanceof SdkAutoConstructList) {
                return null;
            }
            return topAnomalousServices != null ? topAnomalousServices.stream().map(AnomalousService::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder topAnomalousServices(Collection<AnomalousService> topAnomalousServices) {
            this.topAnomalousServices = AnomalousServiceListCopier.copy(topAnomalousServices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topAnomalousServices(AnomalousService... topAnomalousServices) {
            topAnomalousServices(Arrays.asList(topAnomalousServices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder topAnomalousServices(Consumer<AnomalousService.Builder>... topAnomalousServices) {
            topAnomalousServices(Stream.of(topAnomalousServices).map(c -> AnomalousService.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTopAnomalousServices(Collection<AnomalousService.BuilderImpl> topAnomalousServices) {
            this.topAnomalousServices = AnomalousServiceListCopier.copyFromBuilder(topAnomalousServices);
        }

        @Override
        public InsightEvent build() {
            return new InsightEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
