/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.workspaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a connection alias. Connection aliases are used for cross-Region redirection. For more information, see <a
 * href="https://docs.aws.amazon.com/workspaces/latest/adminguide/cross-region-redirection.html"> Cross-Region
 * Redirection for Amazon WorkSpaces</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConnectionAlias implements SdkPojo, Serializable, ToCopyableBuilder<ConnectionAlias.Builder, ConnectionAlias> {
    private static final SdkField<String> CONNECTION_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionString").getter(getter(ConnectionAlias::connectionString))
            .setter(setter(Builder::connectionString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionString").build()).build();

    private static final SdkField<String> ALIAS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AliasId").getter(getter(ConnectionAlias::aliasId)).setter(setter(Builder::aliasId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AliasId").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(ConnectionAlias::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerAccountId").getter(getter(ConnectionAlias::ownerAccountId)).setter(setter(Builder::ownerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerAccountId").build()).build();

    private static final SdkField<List<ConnectionAliasAssociation>> ASSOCIATIONS_FIELD = SdkField
            .<List<ConnectionAliasAssociation>> builder(MarshallingType.LIST)
            .memberName("Associations")
            .getter(getter(ConnectionAlias::associations))
            .setter(setter(Builder::associations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Associations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConnectionAliasAssociation> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConnectionAliasAssociation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONNECTION_STRING_FIELD,
            ALIAS_ID_FIELD, STATE_FIELD, OWNER_ACCOUNT_ID_FIELD, ASSOCIATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String connectionString;

    private final String aliasId;

    private final String state;

    private final String ownerAccountId;

    private final List<ConnectionAliasAssociation> associations;

    private ConnectionAlias(BuilderImpl builder) {
        this.connectionString = builder.connectionString;
        this.aliasId = builder.aliasId;
        this.state = builder.state;
        this.ownerAccountId = builder.ownerAccountId;
        this.associations = builder.associations;
    }

    /**
     * <p>
     * The connection string specified for the connection alias. The connection string must be in the form of a fully
     * qualified domain name (FQDN), such as <code>www.example.com</code>.
     * </p>
     * 
     * @return The connection string specified for the connection alias. The connection string must be in the form of a
     *         fully qualified domain name (FQDN), such as <code>www.example.com</code>.
     */
    public final String connectionString() {
        return connectionString;
    }

    /**
     * <p>
     * The identifier of the connection alias.
     * </p>
     * 
     * @return The identifier of the connection alias.
     */
    public final String aliasId() {
        return aliasId;
    }

    /**
     * <p>
     * The current state of the connection alias.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ConnectionAliasState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the connection alias.
     * @see ConnectionAliasState
     */
    public final ConnectionAliasState state() {
        return ConnectionAliasState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the connection alias.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ConnectionAliasState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the connection alias.
     * @see ConnectionAliasState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The identifier of the AWS account that owns the connection alias.
     * </p>
     * 
     * @return The identifier of the AWS account that owns the connection alias.
     */
    public final String ownerAccountId() {
        return ownerAccountId;
    }

    /**
     * Returns true if the Associations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAssociations() {
        return associations != null && !(associations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The association status of the connection alias.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAssociations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The association status of the connection alias.
     */
    public final List<ConnectionAliasAssociation> associations() {
        return associations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(connectionString());
        hashCode = 31 * hashCode + Objects.hashCode(aliasId());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssociations() ? associations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConnectionAlias)) {
            return false;
        }
        ConnectionAlias other = (ConnectionAlias) obj;
        return Objects.equals(connectionString(), other.connectionString()) && Objects.equals(aliasId(), other.aliasId())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(ownerAccountId(), other.ownerAccountId()) && hasAssociations() == other.hasAssociations()
                && Objects.equals(associations(), other.associations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConnectionAlias").add("ConnectionString", connectionString()).add("AliasId", aliasId())
                .add("State", stateAsString()).add("OwnerAccountId", ownerAccountId())
                .add("Associations", hasAssociations() ? associations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConnectionString":
            return Optional.ofNullable(clazz.cast(connectionString()));
        case "AliasId":
            return Optional.ofNullable(clazz.cast(aliasId()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "OwnerAccountId":
            return Optional.ofNullable(clazz.cast(ownerAccountId()));
        case "Associations":
            return Optional.ofNullable(clazz.cast(associations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConnectionAlias, T> g) {
        return obj -> g.apply((ConnectionAlias) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConnectionAlias> {
        /**
         * <p>
         * The connection string specified for the connection alias. The connection string must be in the form of a
         * fully qualified domain name (FQDN), such as <code>www.example.com</code>.
         * </p>
         * 
         * @param connectionString
         *        The connection string specified for the connection alias. The connection string must be in the form of
         *        a fully qualified domain name (FQDN), such as <code>www.example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionString(String connectionString);

        /**
         * <p>
         * The identifier of the connection alias.
         * </p>
         * 
         * @param aliasId
         *        The identifier of the connection alias.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasId(String aliasId);

        /**
         * <p>
         * The current state of the connection alias.
         * </p>
         * 
         * @param state
         *        The current state of the connection alias.
         * @see ConnectionAliasState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionAliasState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the connection alias.
         * </p>
         * 
         * @param state
         *        The current state of the connection alias.
         * @see ConnectionAliasState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConnectionAliasState
         */
        Builder state(ConnectionAliasState state);

        /**
         * <p>
         * The identifier of the AWS account that owns the connection alias.
         * </p>
         * 
         * @param ownerAccountId
         *        The identifier of the AWS account that owns the connection alias.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccountId(String ownerAccountId);

        /**
         * <p>
         * The association status of the connection alias.
         * </p>
         * 
         * @param associations
         *        The association status of the connection alias.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associations(Collection<ConnectionAliasAssociation> associations);

        /**
         * <p>
         * The association status of the connection alias.
         * </p>
         * 
         * @param associations
         *        The association status of the connection alias.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associations(ConnectionAliasAssociation... associations);

        /**
         * <p>
         * The association status of the connection alias.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ConnectionAliasAssociation>.Builder}
         * avoiding the need to create one manually via {@link List<ConnectionAliasAssociation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ConnectionAliasAssociation>.Builder#build()} is called
         * immediately and its result is passed to {@link #associations(List<ConnectionAliasAssociation>)}.
         * 
         * @param associations
         *        a consumer that will call methods on {@link List<ConnectionAliasAssociation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associations(List<ConnectionAliasAssociation>)
         */
        Builder associations(Consumer<ConnectionAliasAssociation.Builder>... associations);
    }

    static final class BuilderImpl implements Builder {
        private String connectionString;

        private String aliasId;

        private String state;

        private String ownerAccountId;

        private List<ConnectionAliasAssociation> associations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConnectionAlias model) {
            connectionString(model.connectionString);
            aliasId(model.aliasId);
            state(model.state);
            ownerAccountId(model.ownerAccountId);
            associations(model.associations);
        }

        public final String getConnectionString() {
            return connectionString;
        }

        @Override
        public final Builder connectionString(String connectionString) {
            this.connectionString = connectionString;
            return this;
        }

        public final void setConnectionString(String connectionString) {
            this.connectionString = connectionString;
        }

        public final String getAliasId() {
            return aliasId;
        }

        @Override
        public final Builder aliasId(String aliasId) {
            this.aliasId = aliasId;
            return this;
        }

        public final void setAliasId(String aliasId) {
            this.aliasId = aliasId;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ConnectionAliasState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getOwnerAccountId() {
            return ownerAccountId;
        }

        @Override
        public final Builder ownerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
            return this;
        }

        public final void setOwnerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
        }

        public final Collection<ConnectionAliasAssociation.Builder> getAssociations() {
            if (associations instanceof SdkAutoConstructList) {
                return null;
            }
            return associations != null ? associations.stream().map(ConnectionAliasAssociation::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder associations(Collection<ConnectionAliasAssociation> associations) {
            this.associations = ConnectionAliasAssociationListCopier.copy(associations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associations(ConnectionAliasAssociation... associations) {
            associations(Arrays.asList(associations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associations(Consumer<ConnectionAliasAssociation.Builder>... associations) {
            associations(Stream.of(associations).map(c -> ConnectionAliasAssociation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAssociations(Collection<ConnectionAliasAssociation.BuilderImpl> associations) {
            this.associations = ConnectionAliasAssociationListCopier.copyFromBuilder(associations);
        }

        @Override
        public ConnectionAlias build() {
            return new ConnectionAlias(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
